package com.digiwin.athena.atmc.common.mongodb;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.BooleanUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.RedisScript;

import java.util.Collections;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * 分布式锁工具类
 *
 * @author majfa
 * @date 2021/05/23
 */
@Slf4j
public abstract class DistributeLocker {
    private static final long TOTAL_SLEEP = 200L;

    private static final long SLEEP_STEP = 50L;

    /**
     * 尝试获取分布式锁，并最多锁定30秒
     *
     * @param redisTemplate
     * @param lockName      锁名
     * @param action        获取锁后，要执行的动作
     */
    public static void tryLock30s(RedisTemplate redisTemplate, String lockName, Action action) {
        boolean isLocked = false;
        String lockVal = UUID.randomUUID().toString();
        try {
            isLocked = tryLock(redisTemplate, lockName, lockVal, 30L, TimeUnit.SECONDS);
        } catch (InterruptedException ex) {
            log.info("lockName：{},{}", lockName, ex);
            // 吃掉异常
            Thread.currentThread().interrupt();
        }

        if (isLocked) {
            try {
                action.action();
            } finally {
                // releaseLock 还是safeReleaseLock？
                releaseLock(redisTemplate, lockName);
            }
        }
    }

    /**
     * 尝试获取分布式锁
     *
     * @param redisTemplate
     * @param lockKey       锁名
     * @param lockVal       锁内容
     * @param expireTime    过期时间
     * @param timeUnit      过期时间单位
     * @return
     * @throws InterruptedException
     */
    public static boolean tryLock(RedisTemplate redisTemplate, String lockKey, String lockVal, long expireTime, TimeUnit timeUnit) throws InterruptedException {
        long totalSleep = 0L;
        Boolean isLocked = redisTemplate.opsForValue().setIfAbsent(lockKey, lockVal, expireTime, timeUnit);
        while ((null == isLocked || !isLocked) && totalSleep < TOTAL_SLEEP) {// 允许重试三次
            Thread.sleep(SLEEP_STEP);
            isLocked = redisTemplate.opsForValue().setIfAbsent(lockKey, lockVal, expireTime, timeUnit);
            totalSleep += SLEEP_STEP;
        }

        return BooleanUtils.isTrue(isLocked);
    }

    /**
     * 释放分布式锁
     *
     * @param redisTemplate
     * @param lockKey       锁名
     * @param lockVal       值，只有lockVal和redis中lockKey对应的值相等，才会去释放锁
     */
    public static void safeReleaseLock(RedisTemplate redisTemplate, String lockKey, String lockVal) {
        // lua脚本实现原子操作：K-V相等才会删除K-V
        String script = "if ARGV[1]==redis.call('get', KEYS[1]) then redis.call('del', KEYS[1]) end return 'OK'";
        RedisScript<String> redisScript = RedisScript.of(script, String.class);
        redisTemplate.execute(redisScript, Collections.singletonList(lockKey), lockVal);
        redisTemplate.delete(lockKey);
    }

    /**
     * 释放分布式锁
     *
     * @param redisTemplate
     * @param lockKey       锁名
     */
    public static void releaseLock(RedisTemplate redisTemplate, String lockKey) {
        redisTemplate.delete(lockKey);
    }

    /**
     * 获取到分布式锁后，要执行的动作定义类
     */
    public interface Action {
        /**
         * 获取到分布式锁后，要执行的动作
         */
        void action();
    }
}
