package com.digiwin.athena.atmc.common.service.bpmworkitemappendix.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SpringUtil;
import com.digiwin.athena.atmc.common.dao.BpmActivityWorkitemMapper;
import com.digiwin.athena.atmc.common.dao.BpmWorkitemAppendixMapper;
import com.digiwin.athena.atmc.common.enums.WorkitemAppendixType;
import com.digiwin.athena.atmc.common.service.ptm.PtmBacklogTransformService;
import com.digiwin.athena.atmc.common.service.bpmworkitemappendix.BpmWorkitemAppendixService;
import com.digiwin.athena.atmc.common.domain.backlog.ReplyTaskMessageDTO;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.BpmActivityWorkitem;
import com.digiwin.athena.atmc.common.domain.BpmWorkitemAppendix;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.PtmBacklog;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 需辅助名单的任务执行处理类
 *
 * @author gonghongxing at 2021-06-08
 */
@Slf4j
@Service
public class BpmWorkitemAppendixServiceImpl implements BpmWorkitemAppendixService {

    @Autowired
    private BpmWorkitemAppendixMapper bpmWorkitemAppendixMapper;

    @Autowired
    private BpmActivityWorkitemMapper bpmActivityWorkitemMapper;

    @Autowired
    private PtmBacklogTransformService ptmBacklogTransformService;

    private static MessageUtils staticMessageUtils;

    /**
     * 回复型任务 - 流程引擎，增加待办的附加信息
     * @param workitemAppendixType
     * @param workItemId
     * @param email
     * @param supplierName
     * @return
     */
    @Override
    public BpmWorkitemAppendix addReplyTaskByBpmWorkItemId(WorkitemAppendixType workitemAppendixType, Long workItemId, String email, String supplierName){
        return addReplyTask(workitemAppendixType, workItemId, email, supplierName);
    }

    /**
     * 回复型任务 - PTM，增加待办的附加信息
     * @param workitemAppendixType
     * @param ptmBacklogId
     * @param contact
     * @param supplierName
     * @return
     */
    @Override
    public BpmWorkitemAppendix addReplyTaskByPtmBacklogId(WorkitemAppendixType workitemAppendixType, Long ptmBacklogId, String contact, String supplierName){
        return addReplyTask(workitemAppendixType, ptmBacklogId, contact, supplierName);
    }

    private BpmWorkitemAppendix addReplyTask(WorkitemAppendixType workitemAppendixType, Long originalId, String contact, String supplierName){
        if(workitemAppendixType!=WorkitemAppendixType.Normal){
            BpmWorkitemAppendix appendix = BpmWorkitemAppendix.builder()
                    .id(originalId)
                    .errorCode(workitemAppendixType.getValue())
                    .email(contact)
                    .supplierName(supplierName)
                    .build();

            bpmWorkitemAppendixMapper.insert(appendix);
            return bpmWorkitemAppendixMapper.selectById(appendix.getId());
        }

        return null;
    }

    /**
     * 回复型任务- 根据backlogId查找对应的 附加信息
     * @param backlogId
     * @return
     */
    @Override
    public BpmWorkitemAppendix getByBacklogId(Long backlogId){
        BpmWorkitemAppendix result = null;
        Long originalBacklogId = null;

        BpmActivityWorkitem workitem = bpmActivityWorkitemMapper.selectPartialById(backlogId);
        if(workitem!=null){
            List<BpmActivityWorkitem> workitemList = bpmActivityWorkitemMapper.selectAllByStepSqlId(workitem.getActivityStepId());
            if(CollectionUtils.isNotEmpty(workitemList)){
                for (BpmActivityWorkitem wi : workitemList) {
                    if(Objects.equals(wi.getType(),1) && !Objects.equals(wi.getId(), backlogId)){
                        originalBacklogId = wi.getId();
                        break;
                    }
                }
            }
        }else{
            //PTM
            PtmBacklog backlog = ptmBacklogTransformService.getReplyOriginalBacklog(backlogId);
            if(backlog!=null){
                originalBacklogId = backlog.getBacklogId();
            }
        }

        if(originalBacklogId!=null){
            result = bpmWorkitemAppendixMapper.selectById(originalBacklogId);
        }

        return result;
    }

    /**
     * 回复型任务- 根据backlogId查找对应的 附加信息
     * @param backlogId
     * @return
     */
    @Override
    public BpmWorkitemAppendix getByPtmBacklogId(Long backlogId){
        BpmWorkitemAppendix result = null;
        Long originalBacklogId = null;

        PtmBacklog backlog = ptmBacklogTransformService.getReplyOriginalBacklog(backlogId);
        if(backlog!=null){
            originalBacklogId = backlog.getBacklogId();
        }

        if(originalBacklogId!=null){
            result = bpmWorkitemAppendixMapper.selectById(originalBacklogId);
        }

        return result;
    }


    /**
     * 获取 回复型 任务的错误信息
     * @param backlogId
     * @return
     */
    @Override
    public ReplyTaskMessageDTO getReplyTaskMessage(Long backlogId){
        ReplyTaskMessageDTO result = ReplyTaskMessageDTO.builder()
                .code(0)
                .build();
        BpmWorkitemAppendix appendix = getByBacklogId(backlogId);
        if(appendix!=null){
            result = translateReplyTaskMessage(appendix.getErrorCode());
        }

        return result;
    }

    @Override
    public List<BpmWorkitemAppendix> getByBacklogIdBatch(List<Long> backlogIds) {
        return bpmWorkitemAppendixMapper.selectList(
                new LambdaQueryWrapper<BpmWorkitemAppendix>()
                        .in(BpmWorkitemAppendix::getId, backlogIds)
                        .ne(BpmWorkitemAppendix::getErrorCode, 1008)
                        .ne(BpmWorkitemAppendix::getErrorCode,0));
    }

    @Override
    public Map<Long, String> getTaskCardAppendixMessage(List<Long> cardIds) {
        Map<Long, String> map = new HashMap<>();
        List<BpmWorkitemAppendix> appendixList = new ArrayList<>();
        int batchSize = 1000;
        int totalElements = cardIds.size();

        for (int i = 0; i < totalElements; i += batchSize) {
            int fromIndex = i;
            int toIndex = Math.min(i + batchSize, totalElements);
            List<Long> batch = cardIds.subList(fromIndex, toIndex);
            List<BpmWorkitemAppendix> subList = getByBacklogIdBatch(batch);
            appendixList.addAll(subList);
        }
        map = appendixList.stream()
                .collect(Collectors.toMap(
                        BpmWorkitemAppendix::getId
                        ,e->translateReplyTaskMessage(e.getErrorCode()).getMessage()));
        return map;
    }

    public static ReplyTaskMessageDTO translateReplyTaskMessage(Integer workitemAppendixType){
        if(staticMessageUtils==null){
            staticMessageUtils = SpringUtil.getBean(MessageUtils.class);
        }

        ReplyTaskMessageDTO result = ReplyTaskMessageDTO.builder()
                .code(0)
                .build();
        if(workitemAppendixType!=null){
            result.setCode(workitemAppendixType);

            switch (workitemAppendixType){
                case 1001:
                    //供应商无邮箱地址
                    result.setMessage(staticMessageUtils.getMessage("reply.email.NoEmail.1001"));
                    break;

                case 1002:
                    //对方还未注册云账号
                    result.setMessage(staticMessageUtils.getMessage("reply.email.NoRegister.1002"));
                    break;

                case 1003:
                    //账号未授权
                    result.setMessage(staticMessageUtils.getMessage("reply.email.NoPermission.1003"));
                    break;

                case 1004:
                    //授权已满
                    result.setMessage(staticMessageUtils.getMessage("reply.email.PermissionFull.1004"));
                    break;

                case 1005:
                    //未加入指定租户
                    result.setMessage(staticMessageUtils.getMessage("reply.email.NoTenant.1005"));
                    break;

                case 1006:
                    //已完成注册申请
                    result.setMessage(staticMessageUtils.getMessage("reply.email.Invited.1006"));
                    break;

                case 1007:
                    //提出邀请中
                    result.setMessage(staticMessageUtils.getMessage("reply.email.Inviting.1007"));
                    break;

                case 1008:
                    //重发成功
                    result.setMessage(staticMessageUtils.getMessage("reply.email.ReSend.1008"));
                    break;

                case 1009:
                    //手机号未注册
                    result.setMessage(staticMessageUtils.getMessage("reply.email.phone.1009"));
                    break;
            }
        }

        return result;
    }

}
