package com.digiwin.athena.cdme.core.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.nacos.common.utils.Objects;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.constant.MqttConstant;
import com.digiwin.athena.cdme.enumerate.OpType;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.Optional;

/**
 * @author yang.xiao
 * @version V1.0
 * @Description CDC规则过滤
 * @date 2023/10/18 15:27
 * @Copyright 鼎捷软件股份有限公司
 */
public final class FilterUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(FilterUtil.class);

    public static boolean filterData(JSONObject data, String op, MonitorRuleCdcModel monitorRuleCdcModel) {
        if (executeValidFilter(monitorRuleCdcModel)) {
            LOGGER.info("侦测：{}->侦测引擎CDC消息被关闭状态过滤:{}",monitorRuleCdcModel.getRuleId(), data);
            return true;
        }

        if (executeStartTimeFilter(monitorRuleCdcModel)) {
            LOGGER.info("侦测：{}->侦测引擎CDC消息被开始时间过滤:{}",monitorRuleCdcModel.getRuleId(), data);
            return true;
        }

        if (FieldConstant.OP_UPDATE.equals(op)) {
            if (executeChangeFieldsFilter(data, monitorRuleCdcModel)) {
                LOGGER.info("侦测引擎CDC消息被变化字段过滤:{}",monitorRuleCdcModel.getRuleId(), data);
                return true;
            }
        }

        if(MqttConstant.CATEGORY.equals(monitorRuleCdcModel.getCategory())){
            //先执行系统级的锅炉条件
            if(StringUtil.isNotBlank(monitorRuleCdcModel.getFilterCondition())){
                if (executeFilterConditionsFilter(data, monitorRuleCdcModel)) {
                    LOGGER.info("侦测：{}->侦测引擎MQTT消息被过滤系统级条件过滤:{}",monitorRuleCdcModel.getRuleId(), data);
                    return true;
                }
            }
            if(executeConvFilterConditionsFilter(data, monitorRuleCdcModel)){
                LOGGER.info("侦测：{}->侦测引擎MQTT消息被过滤条件过滤:{}",monitorRuleCdcModel.getRuleId(), data);
                return true;
            }
        } else {
            if (executeFilterConditionsFilter(data, monitorRuleCdcModel)) {
                LOGGER.info("侦测：{}->侦测引擎CDC消息被过滤条件过滤:{}",monitorRuleCdcModel.getRuleId(), data);
                return true;
            }
        }


        //返回true将会被丢弃
        return false;
    }

    private static boolean executeStartTimeFilter(MonitorRuleCdcModel monitorRuleCdcModel) {
        return !LocalTimeUtil.isBefore(LocalTimeUtil.getDateTime(monitorRuleCdcModel.getStartTime()), LocalTimeUtil.getCurrentDateTime());
    }

    private static boolean executeValidFilter(MonitorRuleCdcModel monitorRuleCdcModel) {
        return !FieldConstant.DATA_ENABLED.equals(monitorRuleCdcModel.getValid());
    }

    /**
     * 过滤条件过滤
     * @param data
     * @param monitorRuleCdcModel
     * @return
     */
    private static boolean executeFilterConditionsFilter(JSONObject data, MonitorRuleCdcModel monitorRuleCdcModel) {
        JSONArray filterConditions = JsonUtil.parseArray(monitorRuleCdcModel.getFilterCondition());
        for (Object filterCondition :filterConditions) {
            String left = ((JSONObject)filterCondition).getString(FieldConstant.DATA_LEFT);
            String right = ((JSONObject)filterCondition).getString(FieldConstant.DATA_RIGHT);
            String op = ((JSONObject)filterCondition).getString(FieldConstant.DATA_OP);
            String afterValue = ObjectUtils.isEmpty(data.getJSONObject(FieldConstant.DATA_AFTER)) ? data.getJSONObject(FieldConstant.DATA_BEFORE).getString(left) : data.getJSONObject(FieldConstant.DATA_AFTER).getString(left);

            if (OpType.EQUAL.toString().equals(op)) {
                if (!afterValue.equals(right)) {
                    return true;
                }
            }
            if (OpType.NOT_EQUAL.toString().equals(op)) {
                if (afterValue.equals(right)) {
                    return true;
                }
            }
            if (OpType.IS_NULL.toString().equals(op)) {
                if (StringUtil.isNotBlank(afterValue)) {
                    return true;
                }
            }
            if (OpType.IS_NOT_NULL.toString().equals(op)) {
                if (StringUtil.isEmpty(afterValue)) {
                    return true;
                }
            }
            if (OpType.GREATER_EQUAL.toString().equals(op)) {
                BigDecimal rightBigDecimal = new BigDecimal(right);
                BigDecimal afterBigDecimal = new BigDecimal(afterValue);
                return afterBigDecimal.compareTo(rightBigDecimal) < 0 || !afterValue.equals(right);
            }
            if (OpType.LESS_EQUAL.toString().equals(op)) {
                BigDecimal rightBigDecimal = new BigDecimal(right);
                BigDecimal afterBigDecimal = new BigDecimal(afterValue);
                return afterBigDecimal.compareTo(rightBigDecimal) > -1;
            }
            if (OpType.GREATER_THAN.toString().equals(op)) {
                BigDecimal rightBigDecimal = new BigDecimal(right);
                BigDecimal afterBigDecimal = new BigDecimal(afterValue);
                return afterBigDecimal.compareTo(rightBigDecimal) < 0;
            }
            if (OpType.LESS_THAN.toString().equals(op)) {
                BigDecimal rightBigDecimal = new BigDecimal(right);
                BigDecimal afterBigDecimal = new BigDecimal(afterValue);
                return afterBigDecimal.compareTo(rightBigDecimal) > 0;
            }
        }
        return false;
    }

    /**
     * 变化字段过滤
     * @param data
     * @param monitorRuleCdcModel
     * @return
     */
    private static boolean executeChangeFieldsFilter(JSONObject data, MonitorRuleCdcModel monitorRuleCdcModel) {
        JSONArray changeFields = JsonUtil.parseArray(monitorRuleCdcModel.getChangeFields());
        for (Object changeField :changeFields) {
            String field = ((JSONObject)changeField).getString(FieldConstant.DATA_NAME);
            String beforeValue = data.getJSONObject(FieldConstant.DATA_BEFORE).getString(field);
            String afterValue = data.getJSONObject(FieldConstant.DATA_AFTER).getString(field);
            if (beforeValue.equals(afterValue)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 有效性过滤
     * @param monitorRuleCdcModel
     * @return
     */
    public static boolean executeAvailabilityFilter(MonitorRuleCdcModel monitorRuleCdcModel) {
        if (Objects.isNull(monitorRuleCdcModel)) {
            return true;
        }
        return false;
    }

    private static boolean executeConvFilterConditionsFilter(JSONObject data, MonitorRuleCdcModel monitorRuleCdcModel) {
        if (StringUtils.isEmpty(monitorRuleCdcModel.getConversionFilterCondition())) {
            return false;
        }

        JSONArray filterConditions = JsonUtil.parseArray(monitorRuleCdcModel.getConversionFilterCondition());
        // 只要有一个条件组满足即满足，返回：false；条件组内如果有一个条件不满足，该条件组不满足，该条件组是：true
        for (int i = 0; i < filterConditions.size(); i++) {
            JSONObject filterCon = filterConditions.getJSONObject(i);
            JSONArray items = filterCon.getJSONArray(MqttConstant.RULE_CONVERSION_FILTER_CONDITION_ITEMS);
            boolean conGroupFlag = true;
            for (int j = 0; j < items.size(); j++) {
                JSONObject item = items.getJSONObject(j);
                String left = item.getString(FieldConstant.DATA_LEFT);
                String right = item.getString(FieldConstant.DATA_RIGHT);
                String op = item.getString(FieldConstant.DATA_OP);
                String afterValue = data.getJSONObject(FieldConstant.DATA_AFTER).getString(left);

                // 条件组中有条件不满足
                if (valideExp(left, right, op, afterValue)) {
                    conGroupFlag = false;
                    break;
                }
            }

            if (conGroupFlag) {
                return false;
            }
        }

        // 没有条件组满足
        return true;
    }

    private static boolean valideExp(String left, String right, String op, String afterValue) {
        if (OpType.EQUAL.toString().equals(op)) {
            if (!afterValue.equals(right)) {
                return true;
            }
        }
        if (OpType.NOT_EQUAL.toString().equals(op)) {
            if (afterValue.equals(right)) {
                return true;
            }
        }
        if (OpType.IS_NULL.toString().equals(op)) {
            if (StringUtil.isNotBlank(afterValue)) {
                return true;
            }
        }
        if (OpType.IS_NOT_NULL.toString().equals(op)) {
            if (StringUtil.isEmpty(afterValue)) {
                return true;
            }
        }
        if (OpType.GREATER_EQUAL.toString().equals(op)) {
            BigDecimal rightBigDecimal = new BigDecimal(right);
            BigDecimal afterBigDecimal = new BigDecimal(afterValue);
            return afterBigDecimal.compareTo(rightBigDecimal) == -1 || !afterValue.equals(right);
        }
        if (OpType.LESS_EQUAL.toString().equals(op)) {
            BigDecimal rightBigDecimal = new BigDecimal(right);
            BigDecimal afterBigDecimal = new BigDecimal(afterValue);
            return afterBigDecimal.compareTo(rightBigDecimal) != -1;
        }
        if (OpType.GREATER_THAN.toString().equals(op)) {
            BigDecimal rightBigDecimal = new BigDecimal(right);
            BigDecimal afterBigDecimal = new BigDecimal(afterValue);
            return afterBigDecimal.compareTo(rightBigDecimal) == -1;
        }
        if (OpType.LESS_THAN.toString().equals(op)) {
            BigDecimal rightBigDecimal = new BigDecimal(right);
            BigDecimal afterBigDecimal = new BigDecimal(afterValue);
            return afterBigDecimal.compareTo(rightBigDecimal) == 1;
        }

        if(OpType.LEFT_LIKE.toString().equals(op)){
            if(!afterValue.startsWith(right)){
                return true;
            }
        }

        if (OpType.IN.toString().equals(op) || OpType.NOT_IN.toString().equals(op)) {
            JSONArray jsonArray = JSON.parseArray(right);
            Optional<Object> first = jsonArray.stream().filter(s -> s.equals(afterValue)).findFirst();
            if (first.isPresent()) {
                if (OpType.IN.toString().equals(op)) {
                    return false;
                }

                if (OpType.NOT_IN.toString().equals(op)) {
                    return true;
                }
            } else {
                if(OpType.IN.toString().equals(op)) {
                    return true;
                }
            }
        }

        return false;
    }
}