package com.digiwin.athena.cdme.provider;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.aop.RouterKey;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.SecurityUtil;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.repository.model.MonitorDupModel;
import com.digiwin.athena.cdme.service.client.IIamClient;
import com.digiwin.athena.cdme.service.srp.cache.ICacheService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorDupService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import org.springframework.stereotype.Service;

/**
 * @description: redis服务接口
 * @author: dongwh
 * @date: 2020/7/21 14:27
 */
@Service("cdmeRedisOperationService")
public class RedisOperationService implements IOptRedisService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RedisOperationService.class);

    private static final String NO_ACCESS_OPERATOR_REDIS = "无权限操作redis";

    @Autowired
    private IMonitorDupService monitorDupService;

    @Autowired
    private ICacheService redisService;

    @Autowired
    private IIamClient iamClient;

    @Override
    @RouterKey
    public String postCleanCache(Map<String, Object> param) {
        LOGGER.info("操作postCleanCache的入参[{}]", param);

        if (!SecurityUtil.isEnvValid((String) param.get("secure"), FieldConstant.SECURE_TYPE_REDISOPS)) {
            return NO_ACCESS_OPERATOR_REDIS;
        }

        return cleanRedisByKey(param);
    }

    private String cleanRedisByKey(Map<String, Object> param) {
        String key = (String) param.get("key");
        if (!redisService.hasKey(key)) {
            return "当前key值不存在！";
        }
        JSONArray value = redisService.get(key);
        LOGGER.info("操作前key:({}), value:({})", key, value);
        redisService.set(key, new JSONArray());
        /** 同步更新sd_monitor_dup表*/
        try {
            updateDupModelByCondition(key, new JSONArray().toString());
        } catch (Exception e) {
            LOGGER.error("同步清除DB数据异常，请查看！", e);
            /** 还原redis数据*/
            redisService.set(key, value);
            return "清除缓存数据异常，请查看！";
        }
        return redisService.get(key).toString();
    }

    @Override
    @RouterKey
    public String postReadCache(Map<String, Object> param) {
        LOGGER.info("操作postReadCache的入参[{}]", param);

        if (!SecurityUtil.isEnvValid((String) param.get("secure"), FieldConstant.SECURE_TYPE_REDISOPS)) {
            return NO_ACCESS_OPERATOR_REDIS;
        }

        return readRedis(param);
    }

    private String readRedis(Map<String, Object> param) {
        String key = (String) param.get("key");
        if (!redisService.hasKey(key)) {
            return "Key:[" + key + "] in redis is not found!";
        }
        String result = redisService.get(key).toString();

        LOGGER.info("操作postReadCache的key:({}),结果是[{}]", key, result);

        return result;
    }

    @Override
    @RouterKey
    public String postSetCache(Map<String, Object> param) {
        LOGGER.info("操作postSetCache的入参[{}]", param);

        if (!SecurityUtil.isEnvValid((String) param.get("secure"), FieldConstant.SECURE_TYPE_REDISOPS)) {
            return NO_ACCESS_OPERATOR_REDIS;
        }

        return setRedisByKey(param);
    }

    private String setRedisByKey(Map<String, Object> param) {
        String key = (String) param.get("key");
        if (!redisService.hasKey(key)) {
            return "当前key值不存在！";
        }
        if (StringUtil.isBlank((String) param.get("value"))) {
            return "更新的缓存值不能为空";
        }

        JSONArray newCache;
        try {
            newCache = JSON.parseArray((String) param.get("value"));
        } catch (Exception e) {
            LOGGER.error("获取redis更新数据异常", e);
            return "更新缓存值数据格式不属于JSONArray格式，请查看！";
        }
        JSONArray oldCache = redisService.get(key);
        LOGGER.info("操作前key:({}), value:({})", key, oldCache);

        redisService.set(key, newCache);
        /** 同步插入sd_monitor_dup表*/
        try {
            updateDupModelByCondition(key, newCache.toString());
        } catch (Exception e) {
            LOGGER.error("同步更新DB数据异常，请查看！", e);
            /** 还原redis数据*/
            redisService.set(key, oldCache);
            return "更新缓存数据异常，请查看！";
        }
        String result = redisService.get(key).toString();

        LOGGER.info("操作postSetCache的key:({}),结果是[{}]", key, result);

        return result;
    }

//    @Override
//    @RouterKey
//    public Set<String> postKeysByPattern(String pattern, String secure) {
//        LOGGER.info("操作postKeysByPattern的入参[{}]", pattern);
//
//        if (!SecurityUtil.isEnvValid(secure, FieldConstant.SECURE_TYPE_REDISOPS)) {
//            Set<String> result = new HashSet<>();
//            result.add(NO_ACCESS_OPERATOR_REDIS);
//            return result;
//        }
//
//        return keysByPattern(pattern);
//    }
//
//    private Set<String> keysByPattern(String pattern) {
//        Set<String> keys = redisService.keys(pattern);
//        LOGGER.info("操作keysByPattern共查询出[{}]个缓存, 分别是: [{}]", keys.size(), keys);
//        return keys;
//    }
//
//    @Override
//    @RouterKey
//    public String postDeleteKeysByPattern(String pattern, String secure) {
//        LOGGER.info("操作postDeleteKeysByPattern的入参[{}]", pattern);
//
//        if (!SecurityUtil.isEnvValid(secure, FieldConstant.SECURE_TYPE_REDISOPS)) {
//            return NO_ACCESS_OPERATOR_REDIS;
//        }
//
//        Set<String> keys = keysByPattern(pattern);
//        if (CollectionUtil.isEmpty(keys)) {
//            LOGGER.warn("暂无匹配[{}]规则的redis key 存在！", pattern);
//            return "暂无匹配redis key值!";
//        }
//
//        boolean flag = redisService.delete(keys);
//        return flag ? "清除缓存成功!" : "清除缓存失败";
//    }

    /**
     * 通过redisKey更新sd_monitor_dup表
     *
     * @param redisKey monitor-athenaPaasW-002-TCm01-eocSiteId-WFGP
     * @return
     */
    private void updateDupModelByCondition(String redisKey, String cacheValue) {
        String[] splitStr = redisKey.split("-");
        MonitorDupModel dupModel = new MonitorDupModel(splitStr[1], splitStr[2], splitStr[3], splitStr[4], splitStr[5], cacheValue);
        monitorDupService.editByRuleIdAndTenantIdAndProdAndEocmap(dupModel);
    }

    @Override
    @RouterKey
    public String postCleanCache(Map<String, Object> param, String tenantId) {
        LOGGER.info("操作postCleanCache的入参[param:{}，TenantId:{}]", param, tenantId);

        if (StringUtil.isBlank(tenantId)) {
            return "租户Id为空!";
        }

        boolean isTestTenant = iamClient.isTest(tenantId);
        if (!isTestTenant) {
            return "非测试租户,不允许此操作！";
        }

        return cleanRedisByKey(param);
    }

    @Override
    @RouterKey
    public String postReadCache(Map<String, Object> param, String tenantId) {
        LOGGER.info("操作postReadCache的入参[param:{}，TenantId:{}]", param, tenantId);

        if (StringUtil.isBlank(tenantId)) {
            return "租户Id为空!";
        }

        boolean isTestTenant = iamClient.isTest(tenantId);
        if (!isTestTenant) {
            return "非测试租户,不允许此操作！";
        }

        return readRedis(param);
    }

    @Override
    @RouterKey
    public String postSetCache(Map<String, Object> param, String tenantId) {
        LOGGER.info("操作postSetCache的入参[param:{}，TenantId:{}]", param, tenantId);

        if (StringUtil.isBlank(tenantId)) {
            return "租户Id为空!";
        }

        boolean isTestTenant = iamClient.isTest(tenantId);
        if (!isTestTenant) {
            return "非测试租户,不允许此操作！";
        }

        return setRedisByKey(param);
    }

    @Override
    @RouterKey
    public String postReadRuleCache(Map<String, Object> param) {
        LOGGER.info("操作postReadRuleCache的入参[{}]", param);

        if (!SecurityUtil.isEnvValid((String) param.get("secure"), FieldConstant.SECURE_TYPE_REDISOPS)) {
            return NO_ACCESS_OPERATOR_REDIS;
        }

        String key = (String) param.get("key");
        if (!redisService.hasKey(key)) {
            return "Key:[" + key + "] in redis is not found!";
        }

        Map<Object, Object> cdcRuleList = redisService.hGetAll(key);
        String result = JSON.toJSONString(cdcRuleList);
        LOGGER.info("操作postReadCache的key:({}),结果是[{}]", key, result);

        return result;
    }

    @Override
    @RouterKey
    public String postCleanRuleCache(Map<String, Object> param) {
        LOGGER.info("操作postCleanRuleCache的入参[{}]", param);

        if (!SecurityUtil.isEnvValid((String) param.get("secure"), FieldConstant.SECURE_TYPE_REDISOPS)) {
            return NO_ACCESS_OPERATOR_REDIS;
        }

        String key = (String) param.get("key");
        if(StringUtils.isNotEmpty(key) && key.substring(0, FieldConstant.REDIS_CDC_KEY_PREFIX.length()).equals(FieldConstant.REDIS_CDC_KEY_PREFIX)){
            List<String> list = new ArrayList<>();
            list.add(key);
            redisService.delete(list);
        }

        return "success";
    }


}

