package com.digiwin.athena.cdme.core.config;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import javax.annotation.Resource;

/**
 * @author huwei
 */
@Configuration
public class CdmeRedisTemplateConfig {

    private static final String PREFIX = "cdme:";

    @Bean("cdmeRedisTemplate")
    public RedisTemplate<String, Object> redisTemplate(
        RedisConnectionFactory redisConnectionFactory, Environment environment) {
        return getRedisTemplate(redisConnectionFactory, environment, new StringRedisSerializer(),
            new PrefixRedisSerializer());
    }

    public RedisTemplate<String, Object> getRedisTemplate(
        RedisConnectionFactory redisConnectionFactory,
        Environment environment,
        RedisSerializer<String> redisSerializer,
        RedisSerializer<String> keySerializerRedisSerializer) {
        // 2024-6-25 falcon 需求單#32474 统一设置使用DWRedisService保存对象时，不保存null值
        // 舊的行為會輸出 null, 因此默認值為 true
        String serializeNullString = environment.getProperty("dap.redis.serialize-null", "true");
        boolean serializeNull = Boolean.parseBoolean(serializeNullString);
        GenericJackson2JsonRedisSerializer serializer;
        if (serializeNull) {
            serializer = new GenericJackson2JsonRedisSerializer();
        } else {
            ObjectMapper om = new ObjectMapper();
            om.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            // 2024-6-25 falcon 兼容調用空構造函數會設定的方法, 下列設定會在序列化時, 把 Object 類型設定到 @class 字段中
            om.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL,
                JsonTypeInfo.As.PROPERTY); //NOSONAR
            serializer = new GenericJackson2JsonRedisSerializer(om);
        }
        RedisTemplate<String, Object> redis = new RedisTemplate<>();
        redis.setKeySerializer(keySerializerRedisSerializer);
        redis.setValueSerializer(serializer);
        redis.setHashKeySerializer(redisSerializer);
        redis.setHashValueSerializer(serializer);
        redis.setConnectionFactory(redisConnectionFactory);
        redis.afterPropertiesSet();
        return redis;
    }
    static class PrefixRedisSerializer extends StringRedisSerializer {
        @Override
        public byte[] serialize(String s) {
            if (s == null) {
                return null;
            }
            String realKey = PREFIX + s;
            return super.serialize(realKey);
        }

        @Override
        public String deserialize(byte[] bytes) {
            String s = bytes == null ? null : new String(bytes);
            assert s != null;
            int index = s.indexOf(PREFIX);
            if (index != -1) {
                return s.substring(PREFIX.length());
            }
            return s;
        }
    }
}
