package com.digiwin.athena.cdme.provider;

import com.digiwin.app.common.DWApplicationConfigUtils;
import com.digiwin.app.iot.mqtt.ClientType;
import com.digiwin.app.iot.mqtt.DWMqttClientV2Factory;
import com.digiwin.app.iot.mqtt.config.DWMqttConfig;
import com.digiwin.athena.cdme.core.constant.MqttConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.ArgumentValidException;
import com.digiwin.athena.cdme.core.handler.MqttClientSingle;
import com.digiwin.athena.cdme.core.util.MqttUtil;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.mq.consumer.mqtt.MqttConsumerCallback;
import com.digiwin.athena.cdme.pojo.request.MqttRequest;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import com.digiwin.athena.cdme.repository.model.MqttServerConfigModel;
import com.digiwin.athena.cdme.service.client.IIamClient;
import com.digiwin.athena.cdme.service.facade.detection.IMonitorFacadeService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRuleCdcService;
import com.digiwin.athena.cdme.service.srp.db.impl.MqttServerConfigService;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.paho.client.mqttv3.IMqttAsyncClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * @author yang.xiao
 * @version V1.0
 * @Description
 * @date 2024/6/25 14:22
 * @Copyright 鼎捷软件股份有限公司
 */
@Service("cdmeMqttOpsService")
public class MqttOpsService implements IMqttOpsService {

    private static final Logger LOGGER = LoggerFactory.getLogger(MqttOpsService.class);

    @Autowired
    private IMonitorRuleCdcService monitorRuleCdcService;
    @Autowired
    private MqttServerConfigService mqttServerConfigService;
    @Autowired
    private IMonitorFacadeService monitorFacadeService;
    @Autowired
    @Qualifier("cdmeMsgExecutor")
    private ThreadPoolTaskExecutor cdmeMsgExecutor;
    private final IIamClient iamClient;

    public MqttOpsService(IIamClient iamClient) {
        this.iamClient = iamClient;
    }

    @Override
    public String connect(MqttRequest request) {
        checkTenant(request.getTenantId());
        if (StringUtils.isBlank(request.getSource())) {
            return "MQTT来源不可为空";
        }
        IMqttAsyncClient mqttAsyncClient = getiMqttAsyncClient(request);
        if (null == mqttAsyncClient) {
            return "MQTT客户端不存在";
        }

        if (mqttAsyncClient.isConnected()) {
            return "MQTT客户端已连接，请勿重复连接";
        }

        try {
            mqttAsyncClient.connect();
        } catch (MqttException e) {
            return "MQTT连接失败：" + e.toString();
        }

        try {
            Thread.sleep(1000L);
        } catch (InterruptedException e) {
            LOGGER.info(e.getMessage());
        }
        List<MonitorRuleCdcModel> ruleList = monitorRuleCdcService.getListByCategory(MqttConstant.CATEGORY);
        MqttServerConfigModel mqttServerConfigModel = MqttClientSingle.getInstance().pullConfig(request.getSource());
        if(CollectionUtils.isNotEmpty(ruleList)){
            Map<String, List<MonitorRuleCdcModel>> map = ruleList.stream().collect(Collectors.groupingBy(t -> t.getDbName()));
            map.forEach((dbname, rules) -> {
                if (CollectionUtils.isNotEmpty(rules)) {
                    rules.forEach(rule -> {
                        try {
                            mqttAsyncClient.subscribe(MqttUtil.getTopicByDbname(rule.getTenantSid(), rule.getTableName(), dbname,mqttServerConfigModel.getZone()), mqttServerConfigModel.getOps()).waitForCompletion(1000);
                        } catch (Exception e) {
                            LOGGER.warn("订阅topic失败, tenantId={}， topic={}，错误：{}", rule.getTenantId(), rule.getTableName(), e);
                        }
                    });
                }
            });
        }
        return "MQTT连接成功";
    }

    private static IMqttAsyncClient getiMqttAsyncClient(MqttRequest request) {
        IMqttAsyncClient mqttAsyncClient = MqttClientSingle.getInstance().pull(request.getSource());
        return mqttAsyncClient;
    }

    @Override
    public String disconnect(MqttRequest request) {
        checkTenant(request.getTenantId());
        if (StringUtils.isBlank(request.getSource())) {
            return "MQTT来源不可为空";
        }
        IMqttAsyncClient mqttAsyncClient = getiMqttAsyncClient(request);
        if (null == mqttAsyncClient) {
            return "MQTT客户端不存在";
        }

        if (!mqttAsyncClient.isConnected()) {
            return "MQTT客户端已断开，请勿重复断开";
        }

        try {
            mqttAsyncClient.disconnect();
        } catch (MqttException e) {
            return "MQTT断开失败：" + e;
        }
        return "MQTT断开成功";
    }

    @Override
    public String sub(MqttRequest request) {
        if (StringUtil.isEmpty(request.getTopic())) {
            return "主题不能为空";
        }
        IMqttAsyncClient mqttAsyncClient = getiMqttAsyncClient(request);
        if (StringUtils.isBlank(request.getSource())) {
            return "MQTT来源不可为空";
        }
        if (null == mqttAsyncClient) {
            return "MQTT客户端不存在";
        }

        if (!mqttAsyncClient.isConnected()) {
            return "MQTT客户端已断开，请先连接MQTT客户端";
        }
        MqttServerConfigModel mqttServerConfigModel = MqttClientSingle.getInstance().pullConfig(request.getSource());
        try {
            mqttAsyncClient.subscribe(request.getTopic(), mqttServerConfigModel.getOps()).waitForCompletion(1000);
        } catch (MqttException e) {
            return "MQTT订阅失败：" + e.toString();        }
        return "订阅成功";
    }

    @Override
    public String unsub(MqttRequest request) {
        if (StringUtil.isEmpty(request.getTopic())) {
            return "主题不能为空";
        }
        IMqttAsyncClient mqttAsyncClient = getiMqttAsyncClient(request);
        if (StringUtils.isBlank(request.getSource())) {
            return "MQTT来源不可为空";
        }
        if (null == mqttAsyncClient) {
            return "MQTT客户端不存在";
        }

        if (!mqttAsyncClient.isConnected()) {
            return "MQTT客户端已断开，请先连接MQTT客户端";
        }

        try {
            mqttAsyncClient.unsubscribe(request.getTopic());
        } catch (MqttException e) {
            return "MQTT解除订阅失败：" + e;
        }
        return "解除订阅成功";
    }

    @Override
    public void reset(MqttRequest request) {
        if (!Boolean.parseBoolean(DWApplicationConfigUtils.getProperty("cdme.mqttOpen"))) {
            LOGGER.info("MQTT开关关闭，不启动==================================");
            return;
        }
        try {
            List<MqttServerConfigModel> mqttServerEntityList = mqttServerConfigService.getServerList();
            if (CollectionUtils.isNotEmpty(mqttServerEntityList)) {
                if(StringUtils.isNotBlank(request.getBusinessSources())){
                    mqttServerEntityList = mqttServerEntityList.stream().filter(res -> StringUtils.equals(res.getBusinessSources()
                            , request.getBusinessSources())).collect(Collectors.toList());
                }
                if (CollectionUtils.isNotEmpty(mqttServerEntityList)) {
                    //按照集群分组进行创建client
                    Map<String, List<MqttServerConfigModel>> collect = mqttServerEntityList.stream()
                            .collect(Collectors.groupingBy(res -> res.getBusinessSources() + "_" + res.getClusterGroup()));
                    collect.forEach((groupKey, items) -> {
                        try {
                            initMqttClient(items);
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    });
                }
            }
        } catch (Exception e) {
            LOGGER.error("MQTT 启动失败，失败原因：{}==================================", e);
        }
    }

    @Override
    public void initMqttByBusinessSource(String businessSource) throws Exception {
        MqttServerConfigModel serverListByBusinessSources = mqttServerConfigService.getServerListByBusinessSources(businessSource);
        initMqttClient(Lists.newArrayList(serverListByBusinessSources));
    }

    public void initMqttClient(List<MqttServerConfigModel> mqttServerEntityList) throws Exception {
        if (!Boolean.parseBoolean(DWApplicationConfigUtils.getProperty("cdme.mqttOpen"))) {
            LOGGER.info("MQTT开关关闭，不启动==================================");
            return;
        }
        for (MqttServerConfigModel mqttServerConfigModel : mqttServerEntityList) {
            DWMqttConfig dwMqttConfig = DWMqttConfig.getDefaultMqttConfig().clone();
            dwMqttConfig.setServerURI(mqttServerConfigModel.getServerHost());
            dwMqttConfig.setClientId("MQME_SUB_" + UUID.randomUUID().toString().replaceAll("-", ""));
            MqttConnectOptions connectionOptions = dwMqttConfig.getConnectOptions();
            if(StringUtils.isNotBlank(mqttServerConfigModel.getUsername())){
                connectionOptions.setUserName(mqttServerConfigModel.getUsername());
            }
            if(StringUtils.isNotBlank(mqttServerConfigModel.getPwd())){
                connectionOptions.setPassword(mqttServerConfigModel.getPwd().toCharArray());
            }
            dwMqttConfig.setConnectOptions(connectionOptions);
            DWMqttClientV2Factory clientFactory = new DWMqttClientV2Factory(dwMqttConfig);
            IMqttAsyncClient mqttClient = (IMqttAsyncClient)
                    clientFactory.createAndConnectClient(
                            dwMqttConfig.getClientId(), ClientType.ASYNC, new MqttConsumerCallback(monitorFacadeService, monitorRuleCdcService, cdmeMsgExecutor, mqttServerConfigService));
            LOGGER.info("客户端：{}MQTT 启动成功,ops:{}", dwMqttConfig.getClientId(),mqttServerConfigModel.getOps());
            LOGGER.info("MQTT 开始订阅主题来源为：{} ，客户端端为：{},下的topic",mqttServerConfigModel.getBusinessSources(), mqttServerConfigModel.getClientId());
            subscribeTopics(dwMqttConfig, mqttClient, mqttServerConfigModel,mqttServerConfigModel.getOps());
            MqttClientSingle.getInstance().push(mqttServerConfigModel.getBusinessSources(), mqttClient);
            MqttClientSingle.getInstance().pushConfig(mqttServerConfigModel.getBusinessSources(),mqttServerConfigModel);
        }
    }

    private void subscribeTopics(DWMqttConfig mqttConfig, IMqttAsyncClient mqttClient, MqttServerConfigModel mqttServerConfigModel,int ops) {
        List<MonitorRuleCdcModel> ruleList = monitorRuleCdcService.getByCategory(MqttConstant.CATEGORY, mqttServerConfigModel.getBusinessSources());
        if (CollectionUtils.isEmpty(ruleList)) {
            return;
        }
        StringBuffer errorMsg = new StringBuffer();
        if (CollectionUtils.isNotEmpty(ruleList)) {
            ruleList.forEach(rule -> {
                try {
                    mqttClient.subscribe(MqttUtil.getTopicByDbname(rule.getTenantSid(), rule.getTableName(), mqttServerConfigModel.getBusinessSources(),mqttServerConfigModel.getZone()), ops).waitForCompletion(mqttConfig.getWaitForCompletion());
                    LOGGER.info("订阅topic, tenantId={}， topic={} 成功", rule.getTenantId(), rule.getTableName());
                } catch (MqttException e) {
                    LOGGER.warn("订阅topic失败, tenantId={}， topic={}，错误：{}", rule.getTenantId(), rule.getTableName(), e);
                    errorMsg.append(rule.getTenantId()).append(",").append(rule.getTableName()).append(";");
                }
            });
        }
        if (StringUtils.isNotBlank(errorMsg.toString())) {
            LOGGER.warn(String.format("订阅topic失败，失败的租户和topic：{}", errorMsg));
        }
    }

    private void checkTenant(String tenantId) {
        if (StringUtil.isBlank(tenantId)) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 解析token判断tenantId是否一致 **/
        if (!iamClient.isTenantIdMatchByToken(tenantId)) {
            throw new ArgumentValidException(ErrorCodeEnum.TOKEN_NOT_MATCH_TENANT_ID);
        }
    }
}