package com.digiwin.athena.cdme.service.facade.detection.impl;

import com.alibaba.fastjson.JSONArray;
import com.digiwin.app.container.exceptions.DWRuntimeException;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.pojo.bo.DetectionAndExecuteContextBO;
import com.digiwin.athena.cdme.pojo.bo.DetectionRespBO;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ProcessPageDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.repository.model.MonitorRecordModel;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.service.client.IMonitorResultLogClient;
import com.digiwin.athena.cdme.service.client.request.MonitorFailResultLogReq;
import com.digiwin.athena.cdme.service.client.request.MonitorStartLogReq;
import com.digiwin.athena.cdme.service.client.request.MonitorSuccessResultLogReq;
import com.digiwin.athena.cdme.service.facade.detection.IDetectionEventService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRecordService;
import org.apache.commons.lang3.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;

/**
 * @description:
 * @author: liunansheng
 * @date: 2022/11/4 16:31
 */
@Service("cdmeDetectionEventService")
public class DetectionEventService implements IDetectionEventService {

    private static Logger LOGGER = LoggerFactory.getLogger(DetectionEventService.class);

    private final IMonitorResultLogClient monitorResultLogClient;

    private final IMonitorRecordService recordService;

    public DetectionEventService(IMonitorResultLogClient monitorResultLogClient, IMonitorRecordService recordService) {
        this.monitorResultLogClient = monitorResultLogClient;
        this.recordService = recordService;
    }

    @Override
    public void startDetection(MonitorRuleModel ruleModel) {
        try {
            EocDto ruleEoc = new EocDto(ruleModel.getEocCompanyId(), ruleModel.getEocSiteId(), ruleModel.getEocRegionId());
            MonitorStartLogReq req = new MonitorStartLogReq(LocalDateTime.now(), ruleModel.getTenantId(), ruleModel.getRuleId(), ruleEoc);
            monitorResultLogClient.logMonitorResult(req);
        } catch (Exception e) {
            LOGGER.error("startDetection event error", e);
        }
    }

    @Override
    public void preHandleException(MonitorRuleModel ruleModel, DWRuntimeException exception) {
        try {
            EocDto ruleEoc = new EocDto(ruleModel.getEocCompanyId(), ruleModel.getEocSiteId(), ruleModel.getEocRegionId());
            MonitorFailResultLogReq monitorFailResultLogReq = new MonitorFailResultLogReq(exception.getErrorCode(), "APIException", exception.getMessage(), ruleModel.getTenantId(), ruleModel.getRuleId(), ruleEoc);
            monitorResultLogClient.logMonitorResult(monitorFailResultLogReq);
        } catch (Exception e) {
            LOGGER.error("preHandleException event error", e);
        }
    }

    @Override
    public void endDetection(String postType, MonitorRuleModel ruleModel, List<DetectionAndExecuteContextBO> contexts) {
        try {
            if (CollectionUtil.isEmpty(contexts)) {
                return;
            }
            for (DetectionAndExecuteContextBO context : contexts) {
                saveRecord(postType, ruleModel, context);
                sendResultMessage(ruleModel, context);
            }
        } catch (Exception e) {
            LOGGER.error("endDetection event error", e);
        }
    }

    private void saveRecord(String postType, MonitorRuleModel ruleModel, DetectionAndExecuteContextBO context) {
        try {
            EocDto eoc = context.getEocDto();
            MonitorRecordModel recordInfo = new MonitorRecordModel(postType, ruleModel, eoc, context.getMonitorTime());
            recordInfo.setMonitorResult(JsonUtil.getJsonString(context.getDetectionRs()));
            String detectionStatus = Objects.toString(BooleanUtils.toInteger(context.getDetectionRs().getSuccess()));
            recordInfo.setMonitorStatus(detectionStatus);
            if (context.getExecuteRs() != null) {
                String executeStatus = Objects.toString(BooleanUtils.toInteger(context.getExecuteRs().getSuccess()));
                recordInfo.setActionStatus(executeStatus);
                ProcessPageDto execuePageRs = context.getExecuteRs().getData();
                if (null == execuePageRs) {
                    recordInfo.setActionResult(JsonUtil.getJsonString(context.getExecuteRs()));
                } else {
                    recordInfo.setActionResult(execuePageRs.toString());
                }
            }
            recordInfo.setUpdateTime(LocalDateTime.now());
            recordService.save(recordInfo);
        } catch (Exception e) {
            LOGGER.error("save record error", e);
        }
    }

    private void sendResultMessage(MonitorRuleModel ruleModel, DetectionAndExecuteContextBO context) {
        try {
            ResultDto<DetectionRespBO> detectionRs = context.getDetectionRs();
            EocDto eoc = context.getEocDto();
            if (MonitorHelper.isResultFail(detectionRs)) {
                MonitorFailResultLogReq monitorFailResultLogReq = new MonitorFailResultLogReq(detectionRs.getCode(), "APIException", detectionRs.getMessage(), ruleModel.getTenantId(), ruleModel.getRuleId(), eoc);
                monitorResultLogClient.logMonitorResult(monitorFailResultLogReq);
                return;
            }
            ResultDto<ProcessPageDto> executeRs = context.getExecuteRs();
            if (null == executeRs) {
                return;
            }
            ProcessPageDto processPageDto = executeRs.getData();
            if (null == processPageDto && !executeRs.getSuccess()) {
                MonitorFailResultLogReq monitorFailResultLogReq = new MonitorFailResultLogReq(executeRs.getCode(), "APIException", executeRs.getMessage(), ruleModel.getTenantId(), ruleModel.getRuleId(), eoc);
                monitorResultLogClient.logMonitorResult(monitorFailResultLogReq);
                return;
            }
            if (null == processPageDto && executeRs.getSuccess()) {
                MonitorSuccessResultLogReq monitorSuccessResultLogReq = new MonitorSuccessResultLogReq(context.getMonitorTime(), 0, new JSONArray(), ruleModel.getTenantId(), ruleModel.getRuleId(), eoc);
                monitorResultLogClient.logMonitorResult(monitorSuccessResultLogReq);
                return;
            }
            if (Objects.nonNull(processPageDto) && CollectionUtil.isNotEmpty(processPageDto.getSuccessChangeObjects())) {
                for (JSONArray onePageData : processPageDto.getSuccessChangeObjects()) {
                    int state = CollectionUtil.isEmpty(onePageData) ? 0 : 1;
                    MonitorSuccessResultLogReq monitorSuccessResultLogReq = new MonitorSuccessResultLogReq(context.getMonitorTime(), state, onePageData, ruleModel.getTenantId(), ruleModel.getRuleId(), eoc);
                    monitorResultLogClient.logMonitorResult(monitorSuccessResultLogReq);
                }
            }
            if (Objects.nonNull(processPageDto) && CollectionUtil.isNotEmpty(processPageDto.getErrorResponse())) {
                for (ResultDto failRs : processPageDto.getErrorResponse()) {
                    MonitorFailResultLogReq monitorFailResultLogReq = new MonitorFailResultLogReq(failRs.getCode(), "APIException", failRs.getMessage(), ruleModel.getTenantId(), ruleModel.getRuleId(), eoc);
                    monitorResultLogClient.logMonitorResult(monitorFailResultLogReq);
                }
            }
        } catch (Exception e) {
            LOGGER.error("send message to itsys error", e);
        }
    }
}
