package com.digiwin.athena.cdme.service.facade.rulesync.impl;

import com.alibaba.fastjson.JSONArray;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.enums.ResultEnum;
import com.digiwin.athena.cdme.core.exception.ArgumentValidException;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.exception.RollbackException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.*;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerInsModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerModel;
import com.digiwin.athena.cdme.service.client.IScheduleClient;
import com.digiwin.athena.cdme.service.client.IThemeMapClient;
import com.digiwin.athena.cdme.service.facade.detection.data.IDupDataCacheFacadeService;
import com.digiwin.athena.cdme.service.facade.eoc.IMatchFacadeService;
import com.digiwin.athena.cdme.service.facade.ops.IRuleTriggerOperatorFacadeService;
import com.digiwin.athena.cdme.service.facade.rulesync.AbstractCreateRuleService;
import com.digiwin.athena.cdme.service.facade.rulesync.ICrossLevelCreateFacadeService;
import com.digiwin.athena.cdme.service.srp.cache.ILockService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.locks.Lock;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/8/26 13:39
 */
@Service("cdmeCrossLevelCreateFacadeService")
public class CrossLevelCreateFacadeService extends AbstractCreateRuleService implements ICrossLevelCreateFacadeService {

    private static final Logger LOGGER = LoggerFactory.getLogger(CrossLevelCreateFacadeService.class);

    private final IMatchFacadeService eocMatchService;

    private final IDupDataCacheFacadeService dupDataCacheFacadeService;

    private final IScheduleClient scheduleClient;

    private final ILockService lockService;

    public CrossLevelCreateFacadeService(IRuleTriggerOperatorFacadeService ruleTriggerOperatorFacadeService, IThemeMapClient themeMapClient,
                                         IDupDataCacheFacadeService dupDataCacheFacadeService, IScheduleClient scheduleClient,
                                         IMatchFacadeService eocMatchService, ILockService lockService) {
        super(ruleTriggerOperatorFacadeService, themeMapClient);
        this.dupDataCacheFacadeService = dupDataCacheFacadeService;
        this.scheduleClient = scheduleClient;
        this.eocMatchService = eocMatchService;
        this.lockService = lockService;
    }

    @Override
    @Transactional(value = "dw-transactionManager")
    public ResultDto syncCrossLevelRule(CrossLevelDto crossLevelDto) {
        if (CollectionUtil.isEmpty(crossLevelDto.getNewEocList())) {
            throw new ArgumentValidException(ErrorCodeEnum.CROSS_LEVEL_NEW_EOC_EMPTY);
        }
        /** 获取原侦测规则与排程规则的dto对象 */
        MonitorRuleDto ruleDto = getRelativeRuleInfo(crossLevelDto);
        /** 如果源排程启用则禁用，并返回原始排程状态*/
        ResultDto<String> statusResult = disableIfSrcScheduleEnable(ruleDto);
        if (MonitorHelper.isResultFail(statusResult)) {
            throw new BusinessException(statusResult.getCode(), statusResult.getMessage());
        }
        /** 获取分布式锁 */
        Lock lock = lockService.lock(MonitorHelper.buildLockKey(ruleDto.getRuleModel()),
                MonitorHelper.buildOldLockKey(ruleDto.getRuleModel()));
        try {
            /** 跨层级更改侦测规则 */
            syncCrossLevelRule(crossLevelDto, ruleDto, statusResult.getData());
            /** 规则更新成功后，删除原来的侦测规则排程实例 */
            ResultDto deleteScheduleResult = scheduleClient.deleteSchedule(ruleDto.getTriggerId(), ruleDto.getTenantId());
            if (MonitorHelper.isResultFail(deleteScheduleResult)) {
                throw new BusinessException(deleteScheduleResult.getCode(), deleteScheduleResult.getMessage());
            }
            return ResultHelper.generateResult(true, ResultEnum.RULE_SCHEDULE_SUCCESS.getMessage());
        } finally {
            /** 释放分布式锁 */
            lockService.unlock(lock);
        }
    }

    private ResultDto syncCrossLevelRule(CrossLevelDto crossLevelDto, MonitorRuleDto ruleDto, String scheduleStatus) {
        String ruleId = crossLevelDto.getRuleId();
        String tenantId = crossLevelDto.getTenantId();
        EocDto oldEoc = crossLevelDto.getEocMap();
        List<EocDto> newEocs = crossLevelDto.getNewEocList();
        LOGGER.info("跨层级批量创建侦测规则入参:ruleId:{}, tenantId:{}, eocMap:{}", ruleId, tenantId, JsonUtil.getJsonString(newEocs));
        /** redis的key、value映射 */
        Map<String, JSONArray> cacheMap = new HashMap<>();
        /** triggerIns表与新的运营单元对应匹配关系 */
        List<TriggerRepDto> triggerRepDtos = new ArrayList<>();
        try {
            List<MonitorTriggerInsModel> insModels = ruleTriggerOperatorFacadeService.queryTriggerInsByTriggerId(ruleDto.getRuleModel().getTriggerId());
            triggerRepDtos = eocMatchService.processEocMatchIns(newEocs, insModels, ruleDto.getRuleModel());
            /** 创建原侦测规则redis的key、value映射关系 */
            for (TriggerRepDto triggerRepDto : triggerRepDtos) {
                /** 当前运营单元从未发起过侦测时无需同步redis缓存和dup表数据 */
                if (null != triggerRepDto.getIns()) {
                    String redisKey = MonitorHelper.buildRedisKey(tenantId, ruleId, triggerRepDto.getIns().getEocCompanyId(), triggerRepDto.getIns().getEocSiteId(), ruleDto.getRuleModel().getProductName());
                    cacheMap.put(redisKey, dupDataCacheFacadeService.getDupData(redisKey));
                }
            }
            /** 删除老的侦测DB数据，包含redis数据。 */
            removeRule(ruleDto, oldEoc, triggerRepDtos);
            /** 循环创建新的侦测规则，不包含排程 */
            List<MonitorRuleDto> batchRules = new ArrayList<>();
            for (TriggerRepDto triggerRepDto : triggerRepDtos) {
                if (null != triggerRepDto.getIns()) {
                    String redisKey = MonitorHelper.buildRedisKey(tenantId, ruleId, triggerRepDto.getIns().getEocCompanyId(), triggerRepDto.getIns().getEocSiteId(), ruleDto.getRuleModel().getProductName());
                    batchRules.add(createMonitorRule(ruleDto.getRuleModel(), triggerRepDto, cacheMap.get(redisKey), scheduleStatus));
                } else {
                    batchRules.add(createMonitorRule(ruleDto.getRuleModel(), triggerRepDto, null, scheduleStatus));
                }
            }
            /** 批量创建规则排程 */
            ResultDto batchScheduleResult = scheduleClient.createSchedule(batchRules, scheduleStatus);
            if (MonitorHelper.isResultFail(batchScheduleResult)) {
                LOGGER.error("批量创建排程失败，回滚数据！");
                throw new RollbackException(batchScheduleResult.getCode(), batchScheduleResult.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("跨层级更新规则失败，插入老缓存数据，恢复排程状态，回滚数据！", e);
            processRollback(ruleDto, cacheMap, scheduleStatus, triggerRepDtos);
            throw e;
        }
        return ResultHelper.generateResult(true, ResultEnum.RULE_SCHEDULE_SUCCESS.getMessage());
    }

    private MonitorRuleDto createMonitorRule(MonitorRuleModel ruleModel, TriggerRepDto triggerRepDto, JSONArray cache, String scheduleStatus) {
        String ruleId = ruleModel.getRuleId();
        String tenantId = ruleModel.getTenantId();
        /** 插入新增规则数据 */
        ResultDto<MonitorRuleDto> saveResult = saveRule(tenantId, triggerRepDto.getEocDto(), ruleId, scheduleStatus);
        if (MonitorHelper.isResultFail(saveResult)) {
            throw new BusinessException(saveResult.getCode(), saveResult.getMessage());
        }
        /** 复制原有规则的缓存到新的规则 */
        if (null != cache) {
            replicateCache(saveResult.getData().getRuleModel(), triggerRepDto.getEocDto(), cache);
        }
        /** 复制原有规则trigger_ins表中的last_monitor_time到新的规则 */
        if (null != triggerRepDto.getIns()) {
            replicateTriggerIns(triggerRepDto, saveResult.getData().getTriggerModel());
        }
        LOGGER.info("跨层级创建侦测规则成功，ruleId:{}, tenantId:{}, eocMap:{}", ruleId, tenantId, JsonUtil.getJsonString(triggerRepDto.getEocDto()));
        return saveResult.getData();
    }

    /**
     * 复制原有规则的缓存到新的规则
     *
     * @param ruleModel
     * @param eocDto
     * @param cache
     */
    private void replicateCache(MonitorRuleModel ruleModel, EocDto eocDto, JSONArray cache) {
        String cacheKey = getCacheKey(ruleModel.getRuleId(), ruleModel.getTenantId(), ruleModel.getProductName(), eocDto);
        dupDataCacheFacadeService.saveDupData(ruleModel, eocDto, cacheKey, cache);
    }

    /**
     * 复制原有规则trigger_ins表中的last_monitor_time到新的规则
     *
     * @param triggerRepDto
     * @param triggerModel
     */
    private void replicateTriggerIns(TriggerRepDto triggerRepDto, MonitorTriggerModel triggerModel) {
        LocalDateTime lastLastMonitorTime = triggerRepDto.getIns().getLastMonitorTime();
        MonitorTriggerInsModel newTriggerInsModel = new MonitorTriggerInsModel(triggerModel.getId(), triggerRepDto.getEocDto().getEocCompanyId(),
                triggerRepDto.getEocDto().getEocSiteId(), lastLastMonitorTime != null ? lastLastMonitorTime : triggerModel.getCreateTime());
        ruleTriggerOperatorFacadeService.saveTriggerIns(newTriggerInsModel);
    }

    /**
     * 处理异常后的回滚流程：出现异常，插入原来的缓存，恢复原来规则的排程状态
     *
     * @param ruleDto
     * @param scheduleStatus
     */
    private void processRollback(MonitorRuleDto ruleDto, Map<String, JSONArray> cacheMap, String scheduleStatus, List<TriggerRepDto> triggerRepDtos) {
        /** 恢复原规则的缓存 */
        cacheMap.forEach((k, v) -> dupDataCacheFacadeService.cacheDupData(k, v));

        /** 删除新增的缓存 */
        for (TriggerRepDto triggerRepDto : triggerRepDtos) {
            String cacheKey = getCacheKey(ruleDto.getRuleModel().getRuleId(), ruleDto.getRuleModel().getTenantId(), ruleDto.getRuleModel().getProductName(), triggerRepDto.getEocDto());
            boolean isSuccess = dupDataCacheFacadeService.deleteCacheByKey(cacheKey);
            if (!isSuccess) {
                LOGGER.error("CrossLevelCreateService删除新增redis缓存异常，请查看！");
            }
        }

        /** 恢复原规则的排程状态 */
        if (FieldConstant.ENABLE_STATUS_Y.equals(scheduleStatus)) {
            scheduleClient.createSchedule(Collections.singletonList(ruleDto), scheduleStatus);
        }
    }

    private String getCacheKey(String ruleId, String tenantId, String productName, EocDto eocDto) {
        return MonitorHelper.buildCrossLevelRedisKey(new MonitorRuleModel(ruleId, tenantId, productName), eocDto);
    }

    /**
     * 删除侦测规则数据
     *
     * @param ruleDto
     * @param eocDto
     */
    protected void removeRule(MonitorRuleDto ruleDto, EocDto eocDto, List<TriggerRepDto> triggerRepDtos) {
        /** 控制表的插入 删除 更新顺序，放置并发情况下数据库行锁竞争产生死锁 */
        ruleTriggerOperatorFacadeService.deleteRule(ruleDto, eocDto);
        for (TriggerRepDto triggerRepDto : triggerRepDtos) {
            if (null != triggerRepDto.getIns()) {
                dupDataCacheFacadeService.cleanDeduplicateData(ruleDto.getRuleModel(), triggerRepDto);
            }
        }
    }

    /**
     * 获取侦测规则以及排程规则相关信息
     *
     * @param crossLevelDto
     * @return
     */
    private MonitorRuleDto getRelativeRuleInfo(CrossLevelDto crossLevelDto) {
        MonitorRuleDto ruleDto = ruleTriggerOperatorFacadeService.queryRuleDetailByRuleId(crossLevelDto.getRuleId(),
                crossLevelDto.getTenantId(), crossLevelDto.getEocMap());
        MonitorRuleModel ruleModel = ruleDto.getRuleModel();
        if (null == ruleModel || StringUtil.isBlank(ruleModel.getMonitorRule())) {
            throw new BusinessException(ErrorCodeEnum.RULE_NOT_EXISTS);
        }
        if (FieldValConstant.CATEGORY_REPORT.equals(ruleModel.getCategory())) {
            throw new BusinessException(ErrorCodeEnum.CROSS_LEVEL_UNSUPPORTED_REPORT);
        }
        if (null == ruleDto.getTriggerModel()) {
            throw new BusinessException(ErrorCodeEnum.TRIGGER_NOT_EXISTS);
        }
        return ruleDto;
    }

    /**
     * 如果源排程信息启用就禁用。对于跨层级更新侦测规则，必须要先停掉原来的侦测规则，再创建新的侦测规则
     *
     * @param ruleDto
     * @return
     */
    private ResultDto<String> disableIfSrcScheduleEnable(MonitorRuleDto ruleDto) {
        ResultDto<String> statusResult = scheduleClient.getScheduleStatus(ruleDto.getRuleModel().getTriggerId(), ruleDto.getRuleModel().getTenantId(), DWServiceContext.getContext().getToken());
        if (MonitorHelper.isResultFail(statusResult)) {
            return statusResult;
        }
        if (FieldConstant.ENABLE_STATUS_Y.equals(statusResult.getData())) {
            ResultDto scheduleResult = scheduleClient.stopSchedule(ruleDto.getTriggerId(), ruleDto.getTenantId());
            /** 对于跨层级更新侦测规则，必须要先停掉原来的侦测规则，再创建新的侦测规则 */
            if (MonitorHelper.isResultFail(scheduleResult)) {
                return scheduleResult;
            }
        }
        return ResultHelper.generateResult(true, "", statusResult.getData());
    }
}
