package com.digiwin.athena.cdme.service.srp.cache.impl;

import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.service.srp.cache.ILockService;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.redisson.client.RedisException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;

/**
 * @description: 分布式锁Redisson实现
 * @author: dongwh
 * @date: 2021/7/9 17:45
 */
@Service("cdmeDistributeLockService")
public class DistributeLockService implements ILockService {
    private static final Logger LOGGER = LoggerFactory.getLogger(DistributeLockService.class);

    @Autowired
    @Qualifier("cdmeRedissonClient")
    private RedissonClient redissonClient;

    @Override
    public Lock lock(String lockKey) {

        lockKey = FieldConstant.REDIS_LOCK_KEY_PREFIX + lockKey;

        Lock lock;
        try {
            lock = redissonClient.getLock(lockKey);
            lock.lock();
            LOGGER.info("获取分布式锁成功，锁名称：[{}]", lockKey);
        } catch (RedisException e) {
            LOGGER.error("获取锁异常，请查看！锁名称: " + lockKey, e);
            /**  NOTE 自定义Lock是因为当前方法必须要返回一个Lock实例，用于finally中unlock操作 */
            lock = new CustomLock();
            lock.lock();
            lock.unlock();
        }
        return lock;
    }

    /**
     * redis规范调整兼容 后续可去除
     */
    @Override
    public RLock lock(String lockKey, String oldKey) {

        // 先判断老锁是否存在 如果存在直接return
        RLock lock = this.redissonClient.getLock(oldKey);
        if (lock.isLocked()) {
            return lock;
        }

        lockKey = FieldConstant.REDIS_LOCK_KEY_PREFIX + lockKey;

        lock = this.redissonClient.getLock(lockKey);
        lock.lock();
        LOGGER.info("获取分布式锁成功，锁名称：[{}]", lockKey);

        return lock;
    }

    @Override
    public void unlock(Lock lock) {
        try {
            lock.unlock();
            if (lock instanceof RLock) {
                LOGGER.info("释放分布式锁成功，锁名称：[{}]", ((RLock) lock).getName());
            } else {
                LOGGER.info("释放自定义锁成功");
            }
        } catch (RedisException e) {
            if (lock instanceof RLock) {
                LOGGER.error("释放分布式锁异常，锁名称：[" + ((RLock) lock).getName() + "]", e);
            } else {
                LOGGER.error("释放分布式锁异常，请查看！", e);
            }
        }
    }

    public static class CustomLock implements Lock {
        @Override
        public void lock() {
            LOGGER.info("自定义lock操作，无实际意义！");
        }

        @Override
        public void lockInterruptibly() throws InterruptedException {
            throw new BusinessException("CustomLock中不支持lockInterruptibly操作！");
        }

        @Override
        public boolean tryLock() {
            throw new BusinessException("CustomLock中不支持tryLock操作！");
        }

        @Override
        public boolean tryLock(long time, TimeUnit unit) throws InterruptedException {
            throw new BusinessException("CustomLock中不支持tryLock()操作！");
        }

        @Override
        public void unlock() {
            LOGGER.info("自定义unlock操作，无实际意义！");
        }

        @Override
        public Condition newCondition() {
            return null;
        }
    }
}
