package com.digiwin.athena.cdme.core.aop;

/**
 * @author zhangww
 * @description:
 * @date 2023/3/3 17:05
 */

import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.core.exception.ResponseStatusException;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.stereotype.Component;
import org.springframework.web.client.HttpServerErrorException;

import java.lang.reflect.Method;
import java.util.UUID;

/**
 * 生成traceId切面类
 */
@Aspect
@Component("cdmeExceptionRetryAspect")
@EnableAspectJAutoProxy
public class ExceptionRetryAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExceptionRetryAspect.class);

    @Pointcut("@annotation(com.digiwin.athena.cdme.core.aop.ExceptionRetry)")
    public void retryPointCut() {
    }

    @Around("retryPointCut()")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        Method method = methodSignature.getMethod();
        ExceptionRetry retry = method.getAnnotation(ExceptionRetry.class);
        String name = method.getName();
        Object[] args = joinPoint.getArgs();
        String uuid = UUID.randomUUID().toString();
        LOGGER.info("执行重试切面{}, 方法名称{}, 方法参数{}", uuid, name, JsonUtil.getJsonString(args));
        int times = retry.times();
        long waitTime = retry.waitTime();
        // check param
        if (times <= 0) {
            times = 1;
        }

        for (; times >= 0; times--) {
            try {
                return joinPoint.proceed();
            } catch (Exception e) {
                //不是ResponseStatusException异常直接向上抛出
                if (e.getClass() != ResponseStatusException.class) {
                    throw e;
                }

                HttpServerErrorException e1 = (HttpServerErrorException) e.getCause();
                if (e1.getResponseBodyAsString().indexOf("502 Bad Gateway") == -1) {
                    throw e;
                }

                // 如果接下来没有重试机会的话，直接报错
                if (times <= 0) {
                    LOGGER.warn("执行重试切面{}失败", uuid);
                    throw e;
                }

                // 休眠 等待下次执行
                if (waitTime > 0) {
                    Thread.sleep(waitTime);
                }

                LOGGER.warn("执行重试切面{}, 还有{}次重试机会, 异常类型{}, 异常信息{}, 栈信息{}", uuid, times, e.getClass().getName(), e.getMessage(), e.getStackTrace());
            }
        }
        return false;
    }
}
