package com.digiwin.athena.cdme.core.util;

import com.digiwin.athena.cdme.constant.FieldConstant;

import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Month;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * jdk8 LocalTime LocalDate localDateTime等时间类的转换操作
 *
 * @author zhangzhi@digiwin.com
 * @date 2020/9/24
 */
public final class LocalTimeUtil {
    private LocalTimeUtil() {
    }

    private static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";
    private static final String YYYY_MM_DD_HH_MM_SS_SSS = "yyyy-MM-dd HH:mm:ss.SSS";
    private static final String YYYY_MM_DD = "yyyy-MM-dd";
    private static final String YYYY_MM = "yyyy-MM";
    private static final DateTimeFormatter DATETIME_FMT_BY_DATE_AND_TIME = DateTimeFormatter.ofPattern(YYYY_MM_DD_HH_MM_SS);
    public static final DateTimeFormatter DATETIME_FMT_BY_DATE_AND_TIME_UNTIL_MS = DateTimeFormatter.ofPattern(YYYY_MM_DD_HH_MM_SS_SSS);
    public static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern(YYYY_MM_DD);

    private static final DateTimeFormatter DATE_FMT = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final DateTimeFormatter DATETIME_FMT_BY_MONTH = DateTimeFormatter.ofPattern(YYYY_MM);

    private static final DateTimeFormatter START_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd 00:00:00.000");
    private static final DateTimeFormatter END_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd 23:59:59.999");

    private static final String MONITOR_RULE_DAY = "DAY";
    private static final String MONITOR_RULE_WEEK = "WEEK";
    private static final String MONITOR_RULE_MONTH = "MONTH";
    private static final String MONITOR_RULE_SEASON = "SEASON";
    private static final String MONITOR_RULE_YEAR = "YEAR";
    /**
     * 天的增加方式:加/减
     */
    private static final String PLUS_OR_MINUS_DAYS = "0";
    /**
     * 天的增加方式 当周/月/季度/年第一天
     */
    private static final String FIRST_DAY = "1";
    /**
     * 天的增加方式 当周/月/季度/年最后一天
     */
    private static final String LAST_DAY = "2";

    public static String getCurrentDate(boolean isSeparateByLine) {
        LocalDate now = LocalDate.now();
        if (isSeparateByLine) {
            return now.toString();
        }
        return now.format(DATE_FMT);
    }

    /**
     * 从默认时区中的系统时钟获取当前日期时间
     *
     * @return
     */
    public static LocalDateTime now() {
        return LocalDateTime.now();
    }

    /**
     * localDateTime转为long
     *
     * @param localDateTime
     * @return
     */
    public static long parseLocalDateTime(LocalDateTime localDateTime) {
        return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    /**
     * 解析本地日期时间
     *
     * @param text
     * @return
     */
    public static LocalDateTime parse(CharSequence text) {
        return LocalDateTime.parse(text);
    }

    /**
     * 获取当前年月份
     *
     * @return
     */
    public static String getCurrentMonth() {
        return LocalDate.now().format(DATETIME_FMT_BY_MONTH);
    }

    /**
     * 获取当前时间,不包含毫秒
     *
     * @return
     */
    public static String getCurrentDateTime() {
        return LocalDateTime.now().format(DATETIME_FMT_BY_DATE_AND_TIME);
    }

    /**
     * 获取当前时间,包含毫秒
     * @return
     */
    public static String getDateTimeMs(LocalDateTime time) {
        if (null == time) {
            return null;
        }
        return time.format(DATETIME_FMT_BY_DATE_AND_TIME_UNTIL_MS);
    }

    /**
     * 获取当前时间,不包含毫秒
     * @return
     */
    public static String getDateTime(LocalDateTime time) {
        if (null == time) {
            return null;
        }
        return time.format(DATETIME_FMT_BY_DATE_AND_TIME);
    }

    /**
     * 根据输入pattern返回当前时间字符串
     *
     * @param pattern
     * @return
     */
    public static String getCurrentDateTime(String pattern) {
        return LocalDateTime.now().format(DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 格式化到秒 yyyy-MM-dd HH:mm:ss
     *
     * @param timestamp
     * @return
     */
    private static String getDateTime(long timestamp) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.systemDefault()).format(DATETIME_FMT_BY_DATE_AND_TIME);
    }

    /**
     * getDateTime(long timestamp)重载方法
     *
     * @param timestamp
     * @return
     */
    public static String getDateTime(String timestamp) {
        return getDateTime(Long.parseLong(timestamp));
    }

    /**
     * timestamp 转换成 LocalDateTime 类型
     *
     * @param timestamp
     * @return
     */
    public static LocalDateTime parseDateTime(String timestamp) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(Long.parseLong(timestamp)), ZoneId.systemDefault());
    }

    /**
     * yyyy-MM-dd HH:mm:ss格式时间 转换成 LocalDateTime 类型
     *
     * @param time
     * @return
     */
    public static LocalDateTime parseTime(String time) {
        if (StringUtil.isBlank(time)) {
            return null;
        }
        if (time.length() == 19) {
            return LocalDateTime.parse(time, LocalTimeUtil.DATETIME_FMT_BY_DATE_AND_TIME);
        } else {
            return LocalDateTime.parse(time, LocalTimeUtil.DATETIME_FMT_BY_DATE_AND_TIME_UNTIL_MS);
        }
    }

    /**
     * 格式化到毫秒 yyyy-MM-dd HH:mm:ss:SSS
     *
     * @param timestamp
     * @return
     */
    public static String getDateTimeMs(long timestamp) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.systemDefault()).format(DATETIME_FMT_BY_DATE_AND_TIME_UNTIL_MS);
    }

    /**
     * 判断当前日期是否在比较的日期之前
     *
     * @param srcDate
     * @param compareDate
     * @return
     */
    private static boolean isBefore(String srcDate, String srcFormat, String compareDate, String compareFormat) {
        LocalDateTime localDate = LocalDateTime.parse(srcDate, DateTimeFormatter.ofPattern(srcFormat));
        LocalDateTime localCompareDate = LocalDateTime.parse(compareDate, DateTimeFormatter.ofPattern(compareFormat));
        return localDate.isBefore(localCompareDate);
    }

    /**
     * 判断当前日期是否在比较的日期之前
     *
     * @param srcDate
     * @param compareDate
     * @return
     */
    public static boolean isBefore(String srcDate, String compareDate) {
        return isBefore(srcDate, LocalTimeUtil.YYYY_MM_DD_HH_MM_SS, compareDate, LocalTimeUtil.YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 获取按照day规则移动的日期
     *
     * @param day               移动策略
     * @param addDay            移动天数的数量
     * @param dateTimeFormatter 日期格式化类型
     * @return
     */
    private static String getDateTimeByRuleDay(LocalDateTime monitorTime, String day, long addDay, DateTimeFormatter dateTimeFormatter) {
        if (PLUS_OR_MINUS_DAYS.equals(day)) {
            /*相对于今天向前的某一天*/
            return Optional.of(monitorTime.plusDays(addDay).format(dateTimeFormatter)).orElse("");
        } else if (FIRST_DAY.equals(day)) {
            /*当月的第一天*/
            return Optional.of(monitorTime.with(TemporalAdjusters.firstDayOfMonth()).format(dateTimeFormatter)).orElse("");
        } else if (LAST_DAY.equals(day)) {
            /*当月的最后一天*/
            return Optional.of(monitorTime.with(TemporalAdjusters.lastDayOfMonth()).format(dateTimeFormatter)).orElse("");
        }
        return "";
    }

    /**
     * 获取按照week规则移动的日期
     *
     * @param range             移动单位的数量
     * @param day               移动策略
     * @param addDay            移动天数的数量
     * @param dateTimeFormatter 日期格式化类型
     * @return
     */
    private static String getDateTimeByRuleWeek(LocalDateTime monitorTime, long range, String day, long addDay, DateTimeFormatter dateTimeFormatter) {
        LocalDate localDate = monitorTime.toLocalDate();
        if (PLUS_OR_MINUS_DAYS.equals(day)) {
            /*前几周的某一天*/
            return Optional.of(localDate.plusWeeks(range).with(DayOfWeek.MONDAY).plusDays(addDay).format(dateTimeFormatter)).orElse("");
        } else if (FIRST_DAY.equals(day)) {
            /*前几周的第一天*/
            return Optional.of(localDate.plusWeeks(range).with(DayOfWeek.MONDAY).format(dateTimeFormatter)).orElse("");
        } else if (LAST_DAY.equals(day)) {
            /*前几周的最后一天*/
            return Optional.of(localDate.plusWeeks(range).with(DayOfWeek.SUNDAY).format(dateTimeFormatter)).orElse("");
        }
        return "";
    }

    /**
     * 获取按照month规则移动的日期
     *
     * @param range             移动单位的数量
     * @param day               移动策略
     * @param addDay            移动天数的数量
     * @param dateTimeFormatter 日期格式化类型
     * @return
     */
    private static String getDateTimeByRuleMonth(LocalDateTime monitorTime, long range, String day, long addDay, DateTimeFormatter dateTimeFormatter) {
        LocalDate localDate = monitorTime.toLocalDate();
        if (PLUS_OR_MINUS_DAYS.equals(day)) {
            /*前几月的某一天*/
            return Optional.of(localDate.plusMonths(range).with(TemporalAdjusters.firstDayOfMonth()).plusDays(addDay).format(dateTimeFormatter)).orElse("");
        } else if (FIRST_DAY.equals(day)) {
            /*前几月的第一天*/
            return Optional.of(localDate.plusMonths(range).with(TemporalAdjusters.firstDayOfMonth()).format(dateTimeFormatter)).orElse("");
        } else if (LAST_DAY.equals(day)) {
            /*前几月的最后一天*/
            return Optional.of(localDate.plusMonths(range).with(TemporalAdjusters.lastDayOfMonth()).format(dateTimeFormatter)).orElse("");
        }
        return "";
    }

    /**
     * 获取按照season规则移动的日期
     *
     * @param range             移动单位的数量
     * @param day               移动策略
     * @param addDay            移动天数的数量
     * @param dateTimeFormatter 日期格式化类型
     * @return
     */
    private static String getDateTimeByRuleSeason(LocalDateTime monitorTime, long range, String day, long addDay, DateTimeFormatter dateTimeFormatter) {
        LocalDate localDate = monitorTime.toLocalDate();
        if (PLUS_OR_MINUS_DAYS.equals(day)) {
            /*获取前几季度的某一天*/
            Month month = Month.of(localDate.getMonth().firstMonthOfQuarter().getValue());
            return Optional.of(LocalDate.of(localDate.getYear(), month, 1).plusMonths(range * 3)
                    .plusDays(addDay).format(dateTimeFormatter)).orElse("");
        } else if (FIRST_DAY.equals(day)) {
            /*获取前几季度的第一天*/
            Month month = Month.of(localDate.getMonth().firstMonthOfQuarter().getValue());
            return Optional.of(LocalDate.of(localDate.getYear(), month, 1)
                    .plusMonths(range * 3).format(dateTimeFormatter)).orElse("");
        } else if (LAST_DAY.equals(day)) {
            /*获取前几季度的最后一天*/
            Month month = Month.of(localDate.getMonth().firstMonthOfQuarter().getValue());
            return Optional.of(LocalDate.of(localDate.getYear(), month, 1).plusMonths(range * 3)
                    .with(TemporalAdjusters.lastDayOfMonth()).format(dateTimeFormatter)).orElse("");
        }
        return "";
    }

    /**
     * 获取按照year规则移动的日期
     *
     * @param range             移动单位的数量
     * @param day               移动策略
     * @param addDay            移动天数的数量
     * @param dateTimeFormatter 日期格式化类型
     * @return
     */
    private static String getDateTimeByRuleYear(LocalDateTime monitorTime, long range, String day, long addDay, DateTimeFormatter dateTimeFormatter) {
        LocalDate localDate = monitorTime.toLocalDate();
        if (PLUS_OR_MINUS_DAYS.equals(day)) {
            /*前几年的某一天*/
            return Optional.of(localDate.plusYears(range).with(TemporalAdjusters.firstDayOfYear()).plusDays(addDay).format(dateTimeFormatter)).orElse("");
        } else if (FIRST_DAY.equals(day)) {
            /*前几年的第一天*/
            return Optional.of(localDate.plusYears(range).with(TemporalAdjusters.firstDayOfYear()).format(dateTimeFormatter)).orElse("");
        } else if (LAST_DAY.equals(day)) {
            /*前几年的最后一天*/
            return Optional.of(localDate.plusYears(range).with(TemporalAdjusters.lastDayOfYear()).format(dateTimeFormatter)).orElse("");
        }
        return "";
    }

    /**
     * 按照移动策略获取日期
     *
     * @param period   移动单位
     * @param range    移动单位的数量
     * @param day      移动策略
     * @param addDay   移动天数的数量
     * @param timeType 格式化时间类型
     * @return
     */
    public static String getDateTime(LocalDateTime monitorTime, String period, long range, String day, long addDay, String timeType) {
        DateTimeFormatter formatter = TIME_HOLDER.get(timeType);
        String time = "";
        if (MONITOR_RULE_YEAR.equals(period)) {
            time = getDateTimeByRuleYear(monitorTime, range, day, addDay, formatter);
        } else if (MONITOR_RULE_SEASON.equals(period)) {
            time = getDateTimeByRuleSeason(monitorTime, range, day, addDay, formatter);
        } else if (MONITOR_RULE_MONTH.equals(period)) {
            time = getDateTimeByRuleMonth(monitorTime, range, day, addDay, formatter);
        } else if (MONITOR_RULE_WEEK.equals(period)) {
            time = getDateTimeByRuleWeek(monitorTime, range, day, addDay, formatter);
        } else if (MONITOR_RULE_DAY.equals(period)) {
            time = getDateTimeByRuleDay(monitorTime, day, addDay, formatter);
        }
        return time;
    }

    /*存储动态条件时间类型与格式化对应关系*/
    private static final Map<String, DateTimeFormatter> TIME_HOLDER = new HashMap(8);

    static {
        /*SCAN类型:param_type + time_type*/
        TIME_HOLDER.put(FieldConstant.SCAN_PARAM_TYPE_DATE_TIME_START, START_DATE_TIME_FORMATTER);
        TIME_HOLDER.put(FieldConstant.SCAN_PARAM_TYPE_DATE_TIME_END, END_DATE_TIME_FORMATTER);
        TIME_HOLDER.put(FieldConstant.SCAN_PARAM_TYPE_DATE_START, DATE_FORMATTER);
        TIME_HOLDER.put(FieldConstant.SCAN_PARAM_TYPE_DATE_END, DATE_FORMATTER);
        /*API类型:param_type + time_type*/
        TIME_HOLDER.put(FieldConstant.API_PARAM_TYPE_DATE_START, START_DATE_TIME_FORMATTER);
        TIME_HOLDER.put(FieldConstant.API_PARAM_TYPE_DATE_END, END_DATE_TIME_FORMATTER);
    }

}
