package com.digiwin.athena.cdme.core.util;


import com.digiwin.app.json.gson.DWGsonProvider;
import com.digiwin.app.resource.DWApplicationMessageResourceBundleUtils;
import com.digiwin.app.service.DWEAIResult;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.enums.ResultEnum;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;

import java.util.HashMap;
import java.util.Map;

/**
 * @description: 执行引擎服务返回结果工具类
 * @author: renwm
 * @date: 2020/6/22 15:29
 */
public final class ResultHelper {
    private ResultHelper() {
    }

    /**
     * 根据类型获取返回结果对象
     *
     * @param resultTypeEnum
     * @return
     */
    public static ResultDto generateResult(ResultEnum resultTypeEnum) {
        return generateResult(resultTypeEnum.getSuccess(), resultTypeEnum.getMessage());
    }

    /**
     * 根据类型 和返回结果  获取结果对象
     *
     * @param resultTypeEnum
     * @param result
     * @return
     */
    public static <T> ResultDto<T> generateResult(ResultEnum resultTypeEnum, T result) {
        return generateResult(resultTypeEnum.getSuccess(), resultTypeEnum.getMessage(), result);
    }

    /**
     * 根据状态和消息 获取返回结果对象
     *
     * @param success
     * @param msg
     * @return
     */
    public static ResultDto generateResult(boolean success, String msg) {
        return new ResultDto(success, msg);
    }

    /**
     * 根据状态和消息 获取返回结果对象
     *
     * @param success
     * @param msg
     * @return
     */
    public static <T> ResultDto<T> generateResult(boolean success, String msg, T data) {
        return new ResultDto(success, msg, data);
    }

    /**
     * 根据状态和消息 获取返回结果对象
     *
     * @param success
     * @param code
     * @return
     */
    public static <T> ResultDto<T> generateLocaleResult(boolean success, String code, T... data) {
        String message = DWApplicationMessageResourceBundleUtils. getString(code);
        if (data.length == 0) {
            return new ResultDto(success, message);
        }
        return new ResultDto(success, message, data[0]);
    }

    /**
     * 构建成功结果对象
     * @param data
     * @return
     */
    public static <T> ResultDto<T> generateSuccessResult(T data) {
        return new ResultDto(true, null, data);
    }

    /**
     * 构建失败结果对象
     * @param message
     * @return
     */
    public static <T> ResultDto<T> generateFailResult(String message) {
        return new ResultDto(false, message, null);
    }

    /**
     * 构建失败结果对象
     * @param code
     * @param message
     * @param data
     * @return
     */
    public static <T> ResultDto<T> generateFailResult(String code, String message, T... data) {
        return new ResultDto(false, code, message, data);
    }

    /**
     * 构建失败结果对象
     * @param errorCodeEnum
     * @param data
     * @return
     */
    public static <T> ResultDto<T> generateFailResult(ErrorCodeEnum errorCodeEnum, T... data) {
        String message = DWApplicationMessageResourceBundleUtils. getString(errorCodeEnum.getCode());
        if (StringUtil.isBlank(message)) {
            message = errorCodeEnum.getMessage();
        }
        return new ResultDto(false, errorCodeEnum.getCode(), message, data);
    }

    /**
     * 构建失败结果对象
     * @param code
     * @param data
     * @return
     */
    public static <T> ResultDto<T> generateFailResult(String code, T... data) {
        String message = DWApplicationMessageResourceBundleUtils. getString(code);
        return new ResultDto(false, code, message, data);
    }

    /**
     * 返回结果对象
     *
     * @param code
     * @param desc
     * @return
     */
    public static String generateEAIResult(String code, String desc) {
        return DWGsonProvider.getGson().toJson(new DWEAIResult(code, "", desc, new HashMap<>()), DWEAIResult.class);
    }

    /**
     * 返回结果对象
     *
     * @param errorCodeEnum
     * @return
     */
    public static String generateEAIFailResult(ErrorCodeEnum errorCodeEnum) {
        String message = DWApplicationMessageResourceBundleUtils. getString(errorCodeEnum.getCode());
        return DWGsonProvider.getGson().toJson(new DWEAIResult(FieldValConstant.EAI_FAIL_CODE, "", message, new HashMap<>()), DWEAIResult.class);
    }

    /**
     * 返回结果对象
     *
     * @param code
     * @param desc
     * @param data
     * @return
     */
    public static String generateEAIResult(String code, String desc, Object data) {
        return DWGsonProvider.getGson().toJson(new DWEAIResult(code, "", desc, (Map<String, Object>) data), DWEAIResult.class);
    }

}
