package com.digiwin.athena.cdme.provider;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.aop.RouterKey;
import com.digiwin.athena.cdme.core.config.CdmeEaiProp;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.ActiveDetectionRuleDto;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.service.facade.detection.IActiveMonitorFacadeService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRuleService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.Optional;

/**
 * @description: 侦测对外接口
 * @author: dongwh
 * @date: 2021/1/5 10:03
 */
@Service("cdmeOpenApiService")
public class OpenApiService implements IOpenApiService {
    private static final Logger LOGGER = LoggerFactory.getLogger(OpenApiService.class);
    @Autowired
    private IActiveMonitorFacadeService activeMonitorFacadeService;
    @Autowired
    private CdmeEaiProp eaiProp;
    @Autowired
    private IMonitorRuleService ruleService;

    @Override
    @RouterKey
    public String postCallExecution(Map<String, String> headers, String bodyStr) {
        /** prod: ATHENA; 呼叫產品名稱  id: MonitorEngine; 產品識別碼，提供同一產品多台主機辨別之用 name: 呼叫服務名稱*/
        JSONObject serviceObj = JsonUtil.getObject(headers.get(FieldConstant.EAI_DIGI_SERVICE));
        if (CollectionUtil.isEmpty(serviceObj)) {
            return ResultHelper.generateEAIFailResult(ErrorCodeEnum.DIGI_SERVICE_EMPTY);
        }

        if (!eaiProp.getEaiHostProd().equals(serviceObj.getString(FieldConstant.EAI_PROD))
                || !FieldValConstant.CURRENT_MODULE_NAME.equals(serviceObj.getString(FieldConstant.EAI_ID))) {
            return ResultHelper.generateEAIFailResult(ErrorCodeEnum.DIGI_SERVICE_WRONG);
        }

        String tenantId = serviceObj.getString(FieldConstant.TENANT_ID);
        if (StringUtil.isBlank(tenantId)) {
            return ResultHelper.generateEAIFailResult(ErrorCodeEnum.DIGI_SERVICE_TENANT_ID_EMPTY);
        }

        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        DWServiceContext.getContext().getRequestHeader().put(FieldConstant.ROUTER_KEY, request.getHeader(FieldConstant.ROUTER_KEY));
        JSONObject messageBody = JsonUtil.getObject(bodyStr);

        EocDto eocDto = JsonUtil.getObject(headers.get(FieldConstant.EAI_DIGI_EOC_MAP), EocDto.class);
        if (eocDto == null) {
            eocDto = new EocDto();
        }
        LOGGER.info("doInitiativeExecute 入参：tenantId:{}, eocMap:{}, messageBody:{}", tenantId, eocDto, messageBody);
        String ruleId = getRuleId(messageBody);
        if (StringUtil.isBlank(ruleId)) {
            LOGGER.error("侦测规则传值为空, 请查看!");
            return ResultHelper.generateEAIFailResult(ErrorCodeEnum.REPORT_PARAM_ERR);
        }
        Optional<MonitorRuleModel> ruleOptional = getMonitorRule(ruleId, tenantId, eocDto);
        if (!ruleOptional.isPresent()) {
            LOGGER.error("ruleId：{}，tenantId：{}侦测规则未同步, 请查看!", ruleId, tenantId);
            return ResultHelper.generateEAIFailResult(ErrorCodeEnum.RULE_UN_SYNCHRONIZED);
        }
        MonitorRuleModel ruleModel = ruleOptional.get();
        JSONArray changeObjects = getChangeObjects(messageBody);
        ActiveDetectionRuleDto detectionRuleDto = new ActiveDetectionRuleDto(eocDto, ruleModel, changeObjects);
        ResultDto rs = activeMonitorFacadeService.execute(detectionRuleDto);
        if (MonitorHelper.isResultFail(rs)) {
            LOGGER.error("ruleId：{}，tenantId：{}侦测规则失败:[{}], 请查看!", ruleId, tenantId, rs);
            return ResultHelper.generateEAIResult(FieldValConstant.EAI_FAIL_CODE, rs.getMessage());
        }
        return ResultHelper.generateEAIResult(FieldValConstant.EAI_SUCCESS_CODE, rs.toString());
    }

    private String getRuleId(JSONObject messageBody) {
        return getRequestObjects(messageBody).getJSONObject(0).getString(FieldConstant.RULE_ID);
    }

    /**
     * 获取发起执行引擎的paras参数：change_objects字段
     *
     * @param messageBody
     * @return
     */
    private JSONArray getChangeObjects(JSONObject messageBody) {
        return getRequestObjects(messageBody).getJSONObject(0).getJSONArray(FieldConstant.ESP_CHANGE_OBJECTS);
    }


    /**
     * 获取请求中的request_objects节点
     *
     * @param messageBody
     * @return
     */
    private JSONArray getRequestObjects(JSONObject messageBody) {
        return messageBody.getJSONObject(FieldConstant.ESP_STD_DATA).getJSONObject(FieldConstant.ESP_PARAMETER).getJSONArray(FieldConstant.ESP_REQUEST_OBJECTS);
    }

    /**
     * 获取侦测规则
     *
     * @param ruleId
     * @param tenantId
     * @return
     */
    private Optional<MonitorRuleModel> getMonitorRule(String ruleId, String tenantId, EocDto eocDto) {
        /** 根据条件判断是否有对应的侦测规则 */
        MonitorRuleModel ruleModel = ruleService.getByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto);
        //report非租户级别，按eoc没找到规则则按租户去找
        if (ruleModel == null && MonitorHelper.calculateWeight(eocDto) != 0) {
            ruleModel = ruleService.getByRuleIdAndTenantIdAndEoc(ruleId, tenantId, new EocDto());
        }
        return Optional.ofNullable(ruleModel);
    }
}
