package com.digiwin.athena.cdme.provider;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.ArgumentValidException;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.MonitorRuleDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.request.RuleKeyRequest;
import com.digiwin.athena.cdme.pojo.vo.MonitorOpsRuleVo;
import com.digiwin.athena.cdme.pojo.vo.SyncRuleInfoVo;
import com.digiwin.athena.cdme.provider.converter.EocDtoConverter;
import com.digiwin.athena.cdme.provider.converter.RuleVoConverter;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerModel;
import com.digiwin.athena.cdme.service.client.IIamClient;
import com.digiwin.athena.cdme.service.client.IScheduleClient;
import com.digiwin.athena.cdme.service.facade.ops.IRuleTriggerOperatorFacadeService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

/**
 * @description: 规则运维接口
 * @author: liunansheng
 * @date: 2022/4/2 16:45
 */
@Service("cdmeRuleOpsService")
public class RuleOpsService implements IRuleOpsService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RuleOpsService.class);
    private final IIamClient iamClient;
    private final IRuleTriggerOperatorFacadeService ruleTriggerOperatorFacadeService;
    private final IScheduleClient scheduleClient;

    public RuleOpsService(IIamClient iamClient, IRuleTriggerOperatorFacadeService ruleTriggerOperatorFacadeService, IScheduleClient scheduleClient) {
        this.iamClient = iamClient;
        this.ruleTriggerOperatorFacadeService = ruleTriggerOperatorFacadeService;
        this.scheduleClient = scheduleClient;
    }

    @Override
    public SyncRuleInfoVo getMonitorRule(RuleKeyRequest request) {
        LOGGER.info("检查测试租户同步结果是否成功入参:{}", request);
        /** 校验入参合法性 */
        if (StringUtil.isBlank(request.getRuleId()) || StringUtil.isBlank(request.getTenantId())) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 解析token判断tenantId是否一致 **/
        if (!iamClient.isTenantIdMatchByToken(request.getTenantId())) {
            throw new ArgumentValidException(ErrorCodeEnum.TOKEN_NOT_MATCH_TENANT_ID);
        }
        EocDto eocDto = EocDtoConverter.convertEocDtoByCamelCaseMap(request.getEocMap());
        MonitorRuleDto ruleDto = ruleTriggerOperatorFacadeService.queryRuleDetailByRuleId(request.getRuleId(), request.getTenantId(), eocDto);
        MonitorRuleModel ruleModel = ruleDto.getRuleModel();
        if (null == ruleModel) {
            LOGGER.error("入参{}未查询到对应的侦测规则，请查看！", request);
            return null;
        }
        MonitorOpsRuleVo monitorOpsRuleVo = RuleVoConverter.convertModelToVo(ruleModel);
        /** monitorTrigger不存在 */
        if (FieldValConstant.CATEGORY_REPORT.equals(ruleModel.getCategory())) {
            return new SyncRuleInfoVo(new JSONObject(), monitorOpsRuleVo, false);
        }
        MonitorTriggerModel triggerModel = ruleDto.getTriggerModel();
        JSON standardPollingRule = JsonUtil.parse(triggerModel.getTriggerParam());
        /** 获取当前侦测规则的排程启停状态 */
        ResultDto<String> scheduleStatusResult  = scheduleClient.getScheduleStatus(triggerModel.getId(), request.getTenantId(), DWServiceContext.getContext().getToken());
        /** 排程实例状态逻辑处理 **/
        if (MonitorHelper.isResultFail(scheduleStatusResult)) {
            throw new BusinessException(scheduleStatusResult.getCode(), scheduleStatusResult.getMessage());
        }
        boolean scheduleInstanceStatus = FieldConstant.ENABLE_STATUS_Y.equals(scheduleStatusResult.getData());
        return new SyncRuleInfoVo(standardPollingRule, monitorOpsRuleVo, scheduleInstanceStatus);
    }
}
