package com.digiwin.athena.cdme.provider;

import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.ArgumentValidException;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.CrossLevelDto;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.dto.SyncRuleParamDto;
import com.digiwin.athena.cdme.pojo.request.SyncCrossLevelRuleRequest;
import com.digiwin.athena.cdme.pojo.request.SyncRuleRequest;
import com.digiwin.athena.cdme.provider.converter.EocDtoConverter;
import com.digiwin.athena.cdme.provider.converter.SyncRuleParamDtoConverter;
import com.digiwin.athena.cdme.repository.model.MonitorOptRecordModel;
import com.digiwin.athena.cdme.service.client.IIamClient;
import com.digiwin.athena.cdme.service.facade.rulesync.ICrossLevelCreateFacadeService;
import com.digiwin.athena.cdme.service.facade.rulesync.RuleSyncFacadeProxy;
import com.digiwin.athena.cdme.service.srp.db.IMonitorOptRecordService;
import org.apache.commons.lang3.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

/**
 * @description: 规则同步
 * @author: liunansheng
 * @date: 2022/2/22 9:59
 */
@Service("cdmeRuleSyncService")
public class RuleSyncService implements IRuleSyncService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RuleSyncService.class);

    private final RuleSyncFacadeProxy ruleSyncFacadeProxy;

    private final ICrossLevelCreateFacadeService crossLevelCreateFacadeService;

    private final IMonitorOptRecordService optRecordService;

    private final IIamClient iamClient;

    public RuleSyncService(RuleSyncFacadeProxy ruleSyncFacadeProxy, ICrossLevelCreateFacadeService crossLevelCreateFacadeService, IMonitorOptRecordService optRecordService, IIamClient iamClient) {
        this.ruleSyncFacadeProxy = ruleSyncFacadeProxy;
        this.crossLevelCreateFacadeService = crossLevelCreateFacadeService;
        this.optRecordService = optRecordService;
        this.iamClient = iamClient;
    }

    /**
     * 同步侦测规则
     *
     * @param syncRuleRequest
     * @return
     */
    @Override
    public String postMonitorRule(SyncRuleRequest syncRuleRequest) {
        LOGGER.info("同步侦测规则入参:{}", syncRuleRequest);
        if (StringUtil.isBlank(syncRuleRequest.getRuleId()) || StringUtil.isBlank(syncRuleRequest.getTenantId())
                || StringUtil.isBlank(syncRuleRequest.getChangeType())) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 解析token判断tenantId是否一致 **/
        if (!iamClient.isTenantIdMatchByToken(syncRuleRequest.getTenantId())) {
            throw new ArgumentValidException(ErrorCodeEnum.TOKEN_NOT_MATCH_TENANT_ID);
        }
        EocDto eocDto = EocDtoConverter.convertEocDtoByCamelCaseMap(syncRuleRequest.getEocMap());
        MonitorOptRecordModel monitorOptRecordModel = new MonitorOptRecordModel(syncRuleRequest.getTenantId(), syncRuleRequest.getRuleId(), syncRuleRequest.getChangeType(), eocDto, syncRuleRequest.toString());
        try {
            SyncRuleParamDto ruleDto = SyncRuleParamDtoConverter.convertSyncRuleParamDtoByRequest(syncRuleRequest);
            ResultDto synMonitorRuleResult = ruleSyncFacadeProxy.syncMonitorRuleHandler(ruleDto);
            if (MonitorHelper.isResultFail(synMonitorRuleResult)) {
                LOGGER.error("同步侦测规则失败,参数:{}", syncRuleRequest);
                throw new BusinessException(synMonitorRuleResult.getCode(), synMonitorRuleResult.getMessage());
            }
            monitorOptRecordModel.setStatus(1);
            return synMonitorRuleResult.getMessage();
        } finally {
            optRecordService.save(monitorOptRecordModel);
        }
    }

    @Override
    public String postCrossLevelRule(SyncCrossLevelRuleRequest syncCrossLevelRuleRequest) {
        LOGGER.info("同步跨层级侦测规则入参:{}", syncCrossLevelRuleRequest);
        if (StringUtil.isBlank(syncCrossLevelRuleRequest.getRuleId()) || StringUtil.isBlank(syncCrossLevelRuleRequest.getTenantId())
                || CollectionUtil.isEmpty(syncCrossLevelRuleRequest.getNewEocList())) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        syncCrossLevelRuleRequest.getNewEocList().forEach(eoc -> {
            if (CollectionUtil.isEmpty(eoc)) {
                throw new ArgumentValidException(ErrorCodeEnum.CROSS_LEVEL_NEW_EOC_EMPTY);
            }
        });
        /** 解析token判断tenantId是否一致 **/
        if (!iamClient.isTenantIdMatchByToken(syncCrossLevelRuleRequest.getTenantId())) {
            throw new ArgumentValidException(ErrorCodeEnum.TOKEN_NOT_MATCH_TENANT_ID);
        }
        CrossLevelDto crossLevelDto = SyncRuleParamDtoConverter.convertCrossLevelDtoByRequest(syncCrossLevelRuleRequest);
        MonitorOptRecordModel optRecord = new MonitorOptRecordModel(crossLevelDto, 0);
        try {
            ResultDto synMonitorRuleResult = crossLevelCreateFacadeService.syncCrossLevelRule(crossLevelDto);
            optRecord.setStatus(BooleanUtils.toInteger(synMonitorRuleResult.getSuccess()));
            return synMonitorRuleResult.getMessage();
        } finally {
            optRecordService.save(optRecord);
        }
    }
}
