package com.digiwin.athena.cdme.service.facade.detection.data.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.LocalTimeUtil;
import com.digiwin.athena.cdme.pojo.dto.DetectionRequestDto;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.dto.TimeRangeDto;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.service.client.IEspClient;
import com.digiwin.athena.cdme.service.client.request.DataChangeGetParameter;
import com.digiwin.athena.cdme.service.client.request.EspSdkReq;
import com.digiwin.athena.cdme.service.facade.detection.data.IProcessDetectionService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @description: scan类型侦测
 * @author: dongwh
 * @date: 2020/9/27 11:38
 */
@Service(FieldValConstant.CATEGORY_SCAN)
public class ScanDetectionService implements IProcessDetectionService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ScanDetectionService.class);

    private final IEspClient espClient;

    public ScanDetectionService(IEspClient espClient) {
        this.espClient = espClient;
    }

    @Override
    public ResultDto<JSONArray> processDetection(Map<String, JSONObject> allParam, DetectionRequestDto requestDto) {
        EspSdkReq scanParam = buildParam(allParam, requestDto.getRuleModel(), requestDto.getTimeRange(), requestDto.getEoc());
        LOGGER.info("侦测调用ESP开始*************");
        /** 调用ESP接口,接收返回消息 */
        ResultDto<JSONArray> scanResult = espClient.getChangeObjects(scanParam);
        LOGGER.info("侦测调用ESP结束*************");
        return scanResult;
    }


    private EspSdkReq buildParam(Map<String, JSONObject> allParam, MonitorRuleModel ruleModel, TimeRangeDto timeRange, EocDto eocDto) {
        DataChangeGetParameter parameter = new DataChangeGetParameter();
        /** 地端erp 入参*/
        parameter.setEnterprise_no("");
        parameter.setSite_no("");
        parameter.setTenant_id(ruleModel.getTenantId());
        if (FieldValConstant.CATEGORY_SCAN.equals(ruleModel.getCategory()) && !"CUSTOM".equals(ruleModel.getMonitorType())) {
            parameter.setDatetime_s(LocalTimeUtil.getDateTimeMs(timeRange.getStartTime()));
            parameter.setDatetime_e(LocalTimeUtil.getDateTimeMs(timeRange.getEndTime()));
        }
        JSONObject ruleObj = JsonUtil.getObject(ruleModel.getMonitorRule());
        ruleObj.put(FieldConstant.RULE_ID, ruleModel.getRuleId());
        ruleObj.put(FieldConstant.CATEGORY, ruleModel.getCategory());
        ruleObj.put(FieldConstant.VERSION, "-1");
        ruleObj = parseCondition(allParam, ruleObj);
        /** 去掉不必要下发给esp的节点*/
        ruleObj.fluentRemove(FieldConstant.DYNAMIC_PARAMS).fluentRemove(FieldConstant.STATIC_PARAMS).fluentRemove(FieldConstant.IS_CONTINUE)
                .fluentRemove(FieldConstant.IS_POLLING_IDS).fluentRemove(FieldConstant.IS_DEDUPLICATION).fluentRemove(FieldConstant.BK_INFO)
                .fluentRemove(FieldConstant.CHILL_INTERVAL);
        parameter.setRules(Collections.singletonList(ruleObj));
        return new EspSdkReq(ruleModel, eocDto, parameter, FieldValConstant.DATA_CHANGE_GET);
    }

    protected JSONObject parseCondition(Map<String, JSONObject> allParam, JSONObject ruleObj) {
        /* FIXME 相关的校验后续需要加上,比如dynamic_condition与dynamic_params数量是否一致,字段映射是否匹配 */
        /** 解析 dynamic_params节点信息 */
        JSONObject conditionObj = ruleObj.getJSONObject(FieldConstant.DYNAMIC_CONDITION);
        if (CollectionUtil.isEmpty(conditionObj)) {
            return ruleObj;
        }
        recursionItems(allParam, conditionObj);
        return ruleObj;
    }

    /**
     * 递归处理数据
     *
     * @param allParam
     * @param itemObj
     */
    private void recursionItems(Map<String, JSONObject> allParam, JSONObject itemObj) {
        String type = itemObj.getString(FieldConstant.TYPE);
        if ("SINGLE".equals(type)) {
            handleItem(allParam, itemObj);
        } else {
            itemObj.getJSONArray(FieldConstant.ITEMS).forEach(item -> recursionItems(allParam, (JSONObject) item));
        }
    }

    /**
     * 当type=SINGLE时数据处理逻辑
     * @param allParam
     * @param itemObj
     */
    private void handleItem(Map<String, JSONObject> allParam, JSONObject itemObj) {
        /** 处理OP=IN或NOT_IN逻辑 **/
        if ("IN".equals(itemObj.getString(FieldConstant.OP)) || "NOT_IN".equals(itemObj.getString(FieldConstant.OP))) {
            /** 当TYPE=PARAM，替换RIGHT中的具体值并转化为JSON字符串 **/
            if ("PARAM".equals(itemObj.getString(FieldConstant.RIGHT_VALUE_TYPE))) {
                JSONArray paramKeys = itemObj.getJSONArray(FieldConstant.RIGHT);
                itemObj.put(FieldConstant.RIGHT_VALUE_TYPE, allParam.get(paramKeys.get(FieldValConstant.START_POS)).get(FieldConstant.TYPE));
                itemObj.put(FieldConstant.RIGHT, JsonUtil.getJsonString(getRightValue(itemObj.getString(FieldConstant.RIGHT_VALUE_TYPE), allParam, paramKeys)));
            } else {
                /** 当TYPE!=PARAM，仅将RIGHT数组值转化为JSON字符串 **/
                itemObj.put(FieldConstant.RIGHT, JsonUtil.getJsonString(itemObj.get(FieldConstant.RIGHT)));
            }
        } else if ("PARAM".equals(itemObj.getString(FieldConstant.RIGHT_VALUE_TYPE))) {
            /** 处理OP!=IN或NOT_IN且TYPE=PARAM逻辑 **/
            itemObj.put(FieldConstant.RIGHT_VALUE_TYPE, allParam.get(itemObj.get(FieldConstant.RIGHT)).get(FieldConstant.TYPE));
            itemObj.put(FieldConstant.RIGHT, allParam.get(itemObj.get(FieldConstant.RIGHT)).get(FieldConstant.VALUE));
        }
    }

    /**
     * 根据rightValType决定生成的list对象元素类型种类，rightValType:STRING 字符串，BOOLEAN 布尔值，DATE 日期，TIME 时间，DATE_TIME 日期时间，INT 整型，DECIMAL 带小数点的数值型
     * <p>
     * NOTE 后续需要考虑对于
     *
     * @param rightValType
     * @param allParam
     * @return
     */
    private List getRightValue(String rightValType, Map<String, JSONObject> allParam, JSONArray paramKeys) {
        if ("STRING".equals(rightValType)) {
            return paramKeys.stream().map(key -> allParam.get(key).getString(FieldConstant.VALUE)).collect(Collectors.toList());
        }
        if ("INT".equals(rightValType)) {
            return paramKeys.stream().map(key -> allParam.get(key).getInteger(FieldConstant.VALUE)).collect(Collectors.toList());
        }
        if ("BOOLEAN".equals(rightValType)) {
            return paramKeys.stream().map(key -> allParam.get(key).getBoolean(FieldConstant.VALUE)).collect(Collectors.toList());
        }
        if ("DECIMAL".equals(rightValType)) {
            return paramKeys.stream().map(key -> allParam.get(key).getBigDecimal(FieldConstant.VALUE)).collect(Collectors.toList());
        }
        throw new BusinessException(ErrorCodeEnum.DYNAMIC_RULE_ILLEGAL);
    }

}
