package com.digiwin.athena.cdme.service.facade.eoc.impl;

import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.TriggerRepDto;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerInsModel;
import com.digiwin.athena.cdme.service.client.IEspClient;
import com.digiwin.athena.cdme.service.facade.eoc.IMatchFacadeService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author zhangww
 * @description: 运营单元与路由匹配
 * @date 2021/8/3 14:21
 */
@Service("cdmeEocMatchFacadeService")
public class EocMatchFacadeService implements IMatchFacadeService {

    private final IEspClient espClient;

    public EocMatchFacadeService(IEspClient espClient) {
        this.espClient = espClient;
    }

    @Override
    public List<EocDto> match(MonitorRuleModel ruleModel) {
        EocDto originalRuleEoc = new EocDto(ruleModel.getEocCompanyId(), ruleModel.getEocSiteId(), ruleModel.getEocRegionId());
        List<EocDto> dispatchEocs = espClient.getDispatchEoc(ruleModel.getTenantId(), ruleModel.getProductName());
        return match(dispatchEocs, originalRuleEoc);
    }

    /**
     * 匹配处理运营单元信息
     *
     * @param eocDtos
     * @param originalRuleEoc
     * @return
     */
    private List<EocDto> match(List<EocDto> eocDtos, EocDto originalRuleEoc) {
        if (CollectionUtil.isEmpty(eocDtos)) {
            return Collections.singletonList(originalRuleEoc);
        }
        EocDto dispatchEoc = eocDtos.get(FieldValConstant.START_POS);
        int ruleWeight = MonitorHelper.calculateWeight(originalRuleEoc);
        /** 如果eocMap为空，返回dispatch内容*/
        if (ruleWeight == 0) {
            return eocDtos;
        }
        int dispatchWeight = MonitorHelper.calculateWeight(dispatchEoc);
        /** 如果eocMap为company级别，按照此维度匹配dispatch进行刷选 */
        if (ruleWeight == 3) {
            if (dispatchWeight == 0 || dispatchWeight == 3) {
                return Collections.singletonList(originalRuleEoc);
            }
            if (dispatchWeight == 8) {
                List<EocDto> result = eocDtos.stream().filter(eocDto -> StringUtils.equals(originalRuleEoc.getEocCompanyId(), eocDto.getEocCompanyId()))
                        .collect(Collectors.toList());
                if (CollectionUtil.isEmpty(result)) {
                    throw new BusinessException(ErrorCodeEnum.EOC_NOT_MATCH);
                }
                return result;
            }
        }
        /** 如果eocMap为site级别，直接返回rule中的eocMap */
        /** NOTE 后续还需要处理regionId的场景，此处处理到siteId就已经停止，后续还需要加入regionId的分支*/
        return Collections.singletonList(originalRuleEoc);
    }

    /**
     * 跨层级入参运营单元（newEocList）与triggerIns表中运营单元的匹配关系
     *
     * @param newEocs
     * @param insModels
     * @param ruleModel
     * @return
     */
    @Override
    public List<TriggerRepDto> processEocMatchIns(List<EocDto> newEocs, List<MonitorTriggerInsModel> insModels, MonitorRuleModel ruleModel) {
        /** 通过dispatch获取原侦测规则的运营单元 */
        List<EocDto> dispatchEocs = espClient.getDispatchEoc(ruleModel.getTenantId(), ruleModel.getProductName());

        /** 新老侦测规则运营单元权重 */
        int newEocWeight = MonitorHelper.calculateWeight(newEocs.get(FieldValConstant.START_POS));
        int dispatchWeight = MonitorHelper.calculateWeight(dispatchEocs.get(FieldValConstant.START_POS));
        /** 校验入参newEocList 是否符合跨层级修改侦测规则 */
        validEocLevel(newEocs, dispatchEocs, newEocWeight, dispatchWeight);
        if (CollectionUtil.isEmpty(insModels)) {
            return newEocs.stream().map(eocMap -> new TriggerRepDto(eocMap, null)).collect(Collectors.toList());
        }
        /** dispatchWeight为空0，ins表中最多只有一条记录。 */
        if (dispatchWeight == 0) {
            return newEocs.stream().map(eocMap -> new TriggerRepDto(eocMap, insModels.get(0))).collect(Collectors.toList());
        }
        /** 返回新规则的运营单元和原侦测规则triggerIns表记录的对应关系 */
        List<TriggerRepDto> result = new ArrayList<>();
        /** 公司级别 */
        if (newEocWeight == 3) {
            newEocs.stream().forEach(newEoc -> {
                Optional<MonitorTriggerInsModel> optional = insModels.stream().filter(ins -> StringUtils.equals(newEoc.getEocCompanyId(), ins.getEocCompanyId())).findFirst();
                result.add(new TriggerRepDto(newEoc, optional.orElse(null)));
            });
        }

        /** 公司+工厂级别 */
        if (newEocWeight == 8) {
            newEocs.stream().forEach(newEoc -> {
                Optional<MonitorTriggerInsModel> optional = insModels.stream().filter(ins ->
                        (StringUtil.isBlank(ins.getEocSiteId()) || StringUtils.equals(newEoc.getEocSiteId(), ins.getEocSiteId())) && StringUtils.equals(newEoc.getEocCompanyId(), ins.getEocCompanyId()))
                        .findFirst();
                result.add(new TriggerRepDto(newEoc, optional.orElse(null)));
            });
        }

        return result;
    }

    /**
     * 校验入参newEocList 是否符合跨层级修改侦测规则
     *
     * @param newEocDtos
     * @param dispatchEocs
     * @param newEocWeight
     * @param dispatchWeight
     * @return
     */
    private void validEocLevel(List<EocDto> newEocDtos, List<EocDto> dispatchEocs, int newEocWeight, int dispatchWeight) {
        if (newEocWeight == 0) {
            throw new BusinessException(ErrorCodeEnum.CROSS_LEVEL_TENANT_LEVEL_ERR);
        }

        if (newEocWeight < dispatchWeight) {
            /**
             * 如果dispatch是工厂级别，说明多个工厂产品等不同，侦测逻辑不同，所以侦测应该是按工厂建不同规则，
             * 故规则应该是工厂级别，不存在按公司配
             */
            throw new BusinessException(ErrorCodeEnum.CROSS_LEVEL_NEW_GRETTER_OLD_ERR);
        }

        /** 如果dispatch为空，不需要再判断newEocMap和dispatch层级是否匹配。 */
        if (dispatchWeight == 0) {
            return;
        }

        /** 比对同层级的运营单元个数是否相同，如果不相同，不允许更新侦测规则 */
        Set<String> newR = new HashSet<>();
        Set<String> oldR = new HashSet<>();
        boolean isEocSiteLevel = newEocWeight == dispatchWeight && newEocWeight == 8;
        for (EocDto newEocDto : newEocDtos) {
            if (isEocSiteLevel) {
                newR.add(newEocDto.getEocSiteId());
            } else {
                newR.add(newEocDto.getEocCompanyId());
            }
        }

        for (EocDto dispatchEoc : dispatchEocs) {
            if (isEocSiteLevel) {
                oldR.add(dispatchEoc.getEocSiteId());
            } else {
                oldR.add(dispatchEoc.getEocCompanyId());
            }
        }

        if (newR.size() != oldR.size() || !newR.equals(oldR)) {
            throw new BusinessException(ErrorCodeEnum.CROSS_LEVEL_NEW_EOC_NOT_MATCH);
        }
    }
}
