package com.digiwin.athena.cdme.service.facade.rulesync.impl;

import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.pojo.dto.MonitorRuleDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.dto.RuleExtensionDto;
import com.digiwin.athena.cdme.repository.model.MonitorOptRecordModel;
import com.digiwin.athena.cdme.service.facade.rulesync.IRuleExtensionEventService;
import com.digiwin.athena.cdme.service.facade.rulesync.IRuleExtensionService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorOptRecordService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @description:
 * @author: liunansheng
 * @date: 2022/4/14 14:18
 */
@Service("cdmeRuleExtensionService")
public class RuleExtensionService implements IRuleExtensionService {

    private static Logger LOGGER = LoggerFactory.getLogger(RuleExtensionService.class);

    private final Map<String, IRuleExtensionEventService> eventMap;

    private final IMonitorOptRecordService monitorOptRecordService;

    public RuleExtensionService(Map<String, IRuleExtensionEventService> eventMap, IMonitorOptRecordService monitorOptRecordService) {
        this.eventMap = eventMap;
        this.monitorOptRecordService = monitorOptRecordService;
    }

    public ResultDto<Void> execute(MonitorRuleDto ruleDto) {
        if (null == ruleDto.getExtensionDto() || CollectionUtil.isEmpty(ruleDto.getExtensionDto().getEvents())) {
            return ResultHelper.generateSuccessResult(null);
        }
        MonitorOptRecordModel optRecordModel = new MonitorOptRecordModel(ruleDto.getTenantId(), ruleDto.getRuleId(), "RULE_EXTENSION_OP", ruleDto.getRuleEoc(), JsonUtil.getJsonString(ruleDto.getExtensionDto()));
        try {
            LOGGER.info("ruleId:{},tenantId:{},eoc:{}, 处理规则扩展配置", ruleDto.getRuleId(), ruleDto.getTenantId(), ruleDto.getRuleEoc());
            List<RuleExtensionDto.Event> events = ruleDto.getExtensionDto().getEvents();
            List<String> errMsg = new ArrayList<>();
            //开始事件处理
            for (RuleExtensionDto.Event event : events) {
                IRuleExtensionEventService eventService = eventMap.get(event.getEventType());
                if (null == eventService) {
                    errMsg.add("未知的事件" + event.getEventType());
                    continue;
                }
                ResultDto eventRs = eventService.handle(event, ruleDto);
                if (MonitorHelper.isResultFail(eventRs)) {
                    errMsg.add(event.getEventType() + "处理失败：" + eventRs.getMessage());
                }
            }
            if (CollectionUtil.isEmpty(errMsg)) {
                optRecordModel.setStatus(1);
                return ResultHelper.generateSuccessResult(null);
            } else {
                String msg = StringUtils.join(errMsg, ";");
                LOGGER.error("ruleId:{},tenantId:{},eoc:{},处理规则扩展配置失败:{}", ruleDto.getRuleId(), ruleDto.getTenantId(), ruleDto.getRuleEoc(), msg);
                return ResultHelper.generateFailResult(ErrorCodeEnum.AFTER_CREATE_EVENT_ERR);
            }
        } catch (Exception e) {
            LOGGER.error("ruleId:{},tenantId:{},eoc:{},处理规则扩展配置异常", ruleDto.getRuleId(), ruleDto.getTenantId(), ruleDto.getRuleEoc(), e);
            return ResultHelper.generateFailResult(ErrorCodeEnum.AFTER_CREATE_EVENT_ERR);
        } finally {
            monitorOptRecordService.save(optRecordModel);
        }
    }

}
