package com.digiwin.athena.cdme.core.config;

import static com.digiwin.athena.cdme.core.constant.ConfigConstant.MQTT_OPEN;

import com.digiwin.app.common.DWApplicationConfigUtils;
import com.digiwin.app.iot.mqtt.ClientType;
import com.digiwin.app.iot.mqtt.DWMqttClientV2Factory;
import com.digiwin.app.iot.mqtt.config.DWMqttConfig;
import com.digiwin.athena.cdme.core.constant.MqttConstant;
import com.digiwin.athena.cdme.core.handler.MqttClientSingle;
import com.digiwin.athena.cdme.core.util.MqttUtil;
import com.digiwin.athena.cdme.mq.consumer.mqtt.MqttConsumerCallback;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import com.digiwin.athena.cdme.repository.model.MqttServerConfigModel;
import com.digiwin.athena.cdme.service.facade.detection.IMonitorFacadeService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRuleCdcService;
import com.digiwin.athena.cdme.service.srp.db.impl.MqttServerConfigService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.paho.client.mqttv3.IMqttAsyncClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

@Component
public class CdmeMqttClientListener implements ApplicationListener<ContextRefreshedEvent> {
    private static final Logger LOGGER = LoggerFactory.getLogger(CdmeMqttClientListener.class);
    private final AtomicBoolean isInit = new AtomicBoolean(false);
    @Autowired
    private MqttServerConfigService mqttServerConfigService;
    @Autowired
    private IMonitorRuleCdcService monitorRuleCdcService;
    @Autowired
    private IMonitorFacadeService monitorFacadeService;
    @Autowired
    @Qualifier("cdmeMsgExecutor")
    private ThreadPoolTaskExecutor cdmeMsgExecutor;

    @Override
    public void onApplicationEvent(ContextRefreshedEvent contextRefreshedEvent) {
        if (!MQTT_OPEN) {
            LOGGER.info("MQTT开关关闭，不启动==================================");
            return;
        }
        try {
            //防止重复触发
            if (!isInit.compareAndSet(false, true)) {
                return;
            }
            List<MqttServerConfigModel> mqttServerEntityList = mqttServerConfigService.getServerList();
            if (CollectionUtils.isNotEmpty(mqttServerEntityList)) {
                //按照集群分组进行创建client
                Map<String, List<MqttServerConfigModel>> collect = mqttServerEntityList.stream()
                        .collect(Collectors.groupingBy(res -> res.getBusinessSources()));
                collect.forEach((groupKey, items) -> {
                    try {
                       initMqttClient(items);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                });
            }
        } catch (Exception e) {
            LOGGER.error("MQTT 启动失败，失败原因：{}==================================", e);
        }
    }

    private void initMqttClient(List<MqttServerConfigModel> mqttServerEntityList) throws Exception {
        for (MqttServerConfigModel mqttServerConfigModel : mqttServerEntityList) {
            DWMqttConfig dwMqttConfig = DWMqttConfig.getDefaultMqttConfig().clone();
            dwMqttConfig.setServerURI(mqttServerConfigModel.getServerHost());
            dwMqttConfig.setClientId("MQME_SUB_" + UUID.randomUUID().toString().replaceAll("-", ""));
            MqttConnectOptions connectionOptions = dwMqttConfig.getConnectOptions();
            if(StringUtils.isNotBlank(mqttServerConfigModel.getUsername())){
                connectionOptions.setUserName(mqttServerConfigModel.getUsername());
            }
            if(StringUtils.isNotBlank(mqttServerConfigModel.getPwd())){
                connectionOptions.setPassword(mqttServerConfigModel.getPwd().toCharArray());
            }
            dwMqttConfig.setConnectOptions(connectionOptions);
            DWMqttClientV2Factory clientFactory = new DWMqttClientV2Factory(dwMqttConfig);
            IMqttAsyncClient mqttClient = (IMqttAsyncClient)
                    clientFactory.createAndConnectClient(
                            dwMqttConfig.getClientId(), ClientType.ASYNC, new MqttConsumerCallback(monitorFacadeService, monitorRuleCdcService, cdmeMsgExecutor, mqttServerConfigService));
            LOGGER.info("客户端：{}MQTT 启动成功,ops:{}", dwMqttConfig.getClientId(),mqttServerConfigModel.getOps());
            LOGGER.info("MQTT 开始订阅主题来源为：{} ，客户端端为：{},下的topic",mqttServerConfigModel.getBusinessSources(), mqttServerConfigModel.getClientId());
            subscribeTopics(dwMqttConfig, mqttClient, mqttServerConfigModel,mqttServerConfigModel.getOps());
            MqttClientSingle.getInstance().push(mqttServerConfigModel.getBusinessSources(), mqttClient);
            MqttClientSingle.getInstance().pushConfig(mqttServerConfigModel.getBusinessSources(),mqttServerConfigModel);
        }
    }

    private void subscribeTopics(DWMqttConfig mqttConfig, IMqttAsyncClient mqttClient, MqttServerConfigModel mqttServerConfigModel,int ops) {
        List<MonitorRuleCdcModel> ruleList = monitorRuleCdcService.getByCategory(MqttConstant.CATEGORY, mqttServerConfigModel.getBusinessSources());
        if (CollectionUtils.isEmpty(ruleList)) {
            return;
        }
        StringBuffer errorMsg = new StringBuffer();
        if (CollectionUtils.isNotEmpty(ruleList)) {
            ruleList.forEach(rule -> {
                try {
                    mqttClient.subscribe(MqttUtil.getTopicByDbname(rule.getTenantSid(), rule.getTableName(),
                            mqttServerConfigModel.getBusinessSources(),mqttServerConfigModel.getZone()), ops)
                            .waitForCompletion(mqttConfig.getWaitForCompletion());
                    LOGGER.info("订阅topic, tenantId={}， topic={} 成功", rule.getTenantId(), rule.getTableName());
                } catch (MqttException e) {
                    LOGGER.warn("订阅topic失败, tenantId={}， topic={}，错误：{}", rule.getTenantId(), rule.getTableName(), e);
                    errorMsg.append(rule.getTenantId()).append(",").append(rule.getTableName()).append(";");
                }
            });
        }
        if (StringUtils.isNotBlank(errorMsg.toString())) {
            LOGGER.warn(String.format("订阅topic失败，失败的租户和topic：{}", errorMsg));
        }
    }
}