package com.digiwin.athena.cdme.repository.model;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.annotation.IdType;
import com.baomidou.mybatisplus.annotation.TableId;
import com.baomidou.mybatisplus.annotation.TableName;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.util.LocalTimeUtil;
import com.digiwin.athena.cdme.pojo.dto.CrossLevelDto;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ExpireRuleDto;
import com.digiwin.athena.cdme.pojo.dto.SyncRuleParamDto;

import java.time.LocalDateTime;

/**
 * @description: 针对MQ同步规则、手动设置侦测分页条数以及侦测回溯日期的记录表
 * @author: ZhangJun
 * @create: 2021/8/4
 */
@TableName("sd_monitor_cdc_opt_record")
public class MonitorOptRecordModel {
    /**
     * 主键ID
     */
    @TableId(type = IdType.AUTO)
    private String id;
    /**
     * 租户ID
     */
    private String tenantId;
    /**
     * 规则ID
     */
    private String ruleId;

    /**
     * eoc公司ID
     */
    private String eocCompanyId;
    /**
     * eoc工厂ID
     */
    private String eocSiteId;
    /**
     * eoc区域ID
     */
    private String eocRegionId;
    /**
     * 操作类型 SYNC、SET
     */
    private String operateType;
    /**
     * 方式，MQ、API接口
     */
    private String operateMode;
    /**
     * 具体操作类型
     * operate_type=SYNC时,取值:CREATED/MODIFIED/DELETED/DISABLED/ENABLED; operate_type=SET时,取值:TRACK_BACK_TIME/PAGE_SIZE/CROSS_LEVEL',
     */
    private String operate;
    /**
     * 请求参数
     */
    private String requestParam;
    /**
     * 状态
     */
    private Integer status;

    /**
     * 创建时间
     */
    private LocalDateTime createTime;

    public MonitorOptRecordModel() {
        this.createTime = LocalTimeUtil.now();
    }

    /**
     * 同步rule
     * 基于api参数构造MonitorOptRecordModel
     *
     * @param tenantId
     * @param ruleId
     * @param operate
     * @param eocDto
     */
    public MonitorOptRecordModel(String tenantId, String ruleId, String operate, EocDto eocDto, String requestParam) {
        this.tenantId = tenantId;
        this.ruleId = ruleId;
        if (eocDto != null) {
            this.eocCompanyId = eocDto.getEocCompanyId();
            this.eocSiteId = eocDto.getEocSiteId();
            this.eocRegionId = eocDto.getEocRegionId();
        }
        this.createTime = LocalTimeUtil.now();

        this.operate = operate;
        this.operateMode = FieldConstant.OPERATE_MODE_API;
        this.operateType = FieldConstant.OPERATE_TYPE_SYNC;
        this.requestParam = requestParam;
        this.status = 0;
    }

    /**
     * 同步rule
     * mq消息同步构造
     *
     * @param ruleDto
     */
    public MonitorOptRecordModel(SyncRuleParamDto ruleDto) {
        this.tenantId = ruleDto.getTenantId();
        this.ruleId = ruleDto.getRuleId();
        EocDto eocDto = ruleDto.getEocMap();
        if (eocDto != null) {
            this.eocCompanyId = eocDto.getEocCompanyId();
            this.eocSiteId = eocDto.getEocSiteId();
            this.eocRegionId = eocDto.getEocRegionId();
        }
        this.createTime = LocalTimeUtil.now();

        this.operate = ruleDto.getChangeType();
        this.operateMode = FieldConstant.OPERATE_MODE_MQ;
        this.operateType = FieldConstant.OPERATE_TYPE_SYNC;
        this.requestParam = ruleDto.toString();
        this.status = 0;
    }

    /**
     * 基于手动设置侦测回溯日期构造MonitorOptRecordModel
     *
     * @param tenantId
     * @param ruleId
     * @param backtrackDate
     * @param eocDto
     * @param isSuccess
     */
    public MonitorOptRecordModel(String tenantId, String ruleId, String backtrackDate, EocDto eocDto, boolean isSuccess) {
        this.tenantId = tenantId;
        this.ruleId = ruleId;
        if (eocDto != null) {
            this.eocCompanyId = eocDto.getEocCompanyId();
            this.eocSiteId = eocDto.getEocSiteId();
            this.eocRegionId = eocDto.getEocRegionId();
        }
        this.createTime = LocalTimeUtil.now();

        this.operate = FieldConstant.TRACK_BACK_TIME;
        this.operateMode = FieldConstant.OPERATE_MODE_API;
        this.operateType = FieldConstant.OPERATE_TYPE_SET;
        JSONObject requestParam = new JSONObject();
        requestParam.put("tenantId", tenantId);
        requestParam.put("ruleId", ruleId);
        requestParam.put("eocMap", eocDto);
        requestParam.put("backtrackDate", backtrackDate);
        this.requestParam = requestParam.toJSONString();
        this.status = isSuccess ? 1 : 0;
    }

    public MonitorOptRecordModel(String tenantId, String ruleId, int pageSize, EocDto eocDto, boolean isSuccess) {
        this.tenantId = tenantId;
        this.ruleId = ruleId;
        if (eocDto != null) {
            this.eocCompanyId = eocDto.getEocCompanyId();
            this.eocSiteId = eocDto.getEocSiteId();
            this.eocRegionId = eocDto.getEocRegionId();
        }
        this.createTime = LocalTimeUtil.now();

        this.operate = FieldConstant.PAGE_SIZE;
        this.operateMode = FieldConstant.OPERATE_MODE_API;
        this.operateType = FieldConstant.OPERATE_TYPE_SET;
        JSONObject requestParam = new JSONObject();
        requestParam.put("tenantId", tenantId);
        requestParam.put("ruleId", ruleId);
        requestParam.put("eocMap", eocDto);
        requestParam.put("pageSize", pageSize);
        this.requestParam = requestParam.toJSONString();
        this.status = isSuccess ? 1 : 0;
    }

    /**
     * 跨层级更新侦测规则
     *
     * @param crossLevelDto
     */
    public MonitorOptRecordModel(CrossLevelDto crossLevelDto, Integer status) {
        this.tenantId = crossLevelDto.getTenantId();
        this.ruleId = crossLevelDto.getRuleId();
        if (crossLevelDto.getEocMap() != null) {
            this.eocCompanyId = crossLevelDto.getEocMap().getEocCompanyId();
            this.eocSiteId = crossLevelDto.getEocMap().getEocSiteId();
            this.eocRegionId = crossLevelDto.getEocMap().getEocRegionId();
        }
        this.createTime = LocalTimeUtil.now();

        this.operate = FieldConstant.CROSS_LEVEL;
        this.operateMode = FieldConstant.OPERATE_MODE_MQ;
        this.operateType = FieldConstant.OPERATE_TYPE_SET;
        this.requestParam = crossLevelDto.toString();
        this.status = status;
    }

    /**
     * 产品过期，删除对应的侦测规则
     *
     * @param ruleExpireDto
     * @param status
     */
    public MonitorOptRecordModel(ExpireRuleDto ruleExpireDto, Integer status) {
        this.ruleId = "appExpire";
        this.tenantId = ruleExpireDto.getTenantId();
        this.createTime = LocalTimeUtil.now();
        this.operate = FieldConstant.RULE_EXPIRE;
        this.operateMode = FieldConstant.OPERATE_MODE_MQ;
        this.operateType = FieldConstant.OPERATE_TYPE_SYNC;
        this.requestParam = ruleExpireDto.toString();
        this.status = status;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getTenantId() {
        return tenantId;
    }

    public void setTenantId(String tenantId) {
        this.tenantId = tenantId;
    }

    public String getRuleId() {
        return ruleId;
    }

    public void setRuleId(String ruleId) {
        this.ruleId = ruleId;
    }

    public String getEocCompanyId() {
        return eocCompanyId;
    }

    public void setEocCompanyId(String eocCompanyId) {
        this.eocCompanyId = eocCompanyId;
    }

    public String getEocSiteId() {
        return eocSiteId;
    }

    public void setEocSiteId(String eocSiteId) {
        this.eocSiteId = eocSiteId;
    }

    public String getEocRegionId() {
        return eocRegionId;
    }

    public void setEocRegionId(String eocRegionId) {
        this.eocRegionId = eocRegionId;
    }

    public String getOperateType() {
        return operateType;
    }

    public void setOperateType(String operateType) {
        this.operateType = operateType;
    }

    public String getOperateMode() {
        return operateMode;
    }

    public void setOperateMode(String operateMode) {
        this.operateMode = operateMode;
    }

    public String getOperate() {
        return operate;
    }

    public void setOperate(String operate) {
        this.operate = operate;
    }

    public String getRequestParam() {
        return requestParam;
    }

    public void setRequestParam(String requestParam) {
        this.requestParam = requestParam;
    }

    public Integer getStatus() {
        return status;
    }

    public void setStatus(Integer status) {
        this.status = status;
    }

    public LocalDateTime getCreateTime() {
        return createTime;
    }

    public void setCreateTime(LocalDateTime createTime) {
        this.createTime = createTime;
    }
}
