package com.digiwin.athena.cdme.service.client.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.MonitorRuleDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerModel;
import com.digiwin.athena.cdme.service.client.AbstractExecuteClient;
import com.digiwin.athena.cdme.service.client.IScheduleClient;
import com.digiwin.athena.cdme.service.client.request.BatchCreateScheduleRequest;
import com.digiwin.athena.cdme.service.client.request.ScheduleBaseRequest;
import com.digiwin.athena.cdme.service.client.request.SyncScheduleRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/20 10:56
 */
@Service("cdmeScheduleClient")
public class ScheduleClient extends AbstractExecuteClient implements IScheduleClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(ScheduleClient.class);

    @Override
    public ResultDto createSchedule(List<MonitorRuleDto> ruleDtos, String scheduleStatus) {
        List<SyncScheduleRequest> batchParams = new ArrayList<>();
        ruleDtos.forEach(ruleDto -> {
            MonitorRuleModel ruleModel = ruleDto.getRuleModel();
            MonitorTriggerModel triggerModel = ruleDto.getTriggerModel();
            JSONArray triggerParam = JsonUtil.parseArray(triggerModel.getTriggerParam());
            SyncScheduleRequest param = new SyncScheduleRequest(triggerModel.getId(), null, ruleModel.getActionId(), triggerParam, StringUtil.trimToNull(scheduleStatus));
            batchParams.add(param);
        });
        BatchCreateScheduleRequest request = new BatchCreateScheduleRequest(ruleDtos.get(0).getTenantId(), batchParams);
        return syncSchedule(request, ConfigConstant.POST_SCHEDULE_URL, HttpMethod.POST, "创建排程");
    }

    @Override
    public ResultDto editSchedule(MonitorRuleDto ruleDto, String scheduleStatus) {
        MonitorRuleModel ruleModel = ruleDto.getRuleModel();
        MonitorTriggerModel triggerModel = ruleDto.getTriggerModel();
        JSONArray triggerParam = JsonUtil.parseArray(triggerModel.getTriggerParam());
        SyncScheduleRequest request = new SyncScheduleRequest(triggerModel.getId(), ruleModel.getTenantId(), ruleModel.getActionId(), triggerParam, StringUtil.trimToNull(scheduleStatus));
        return syncSchedule(request, ConfigConstant.POST_SCHEDULE_URL, HttpMethod.PUT, "修改排程");
    }

    @Override
    public ResultDto stopSchedule(String triggerId, String tenantId) {
        ScheduleBaseRequest request = new ScheduleBaseRequest(triggerId, tenantId);
        return syncSchedule(request, ConfigConstant.POST_SCHEDULE_URL + "/stop", HttpMethod.POST, "暂停排程");
    }

    @Override
    public ResultDto startSchedule(String triggerId, String tenantId) {
        ScheduleBaseRequest request = new ScheduleBaseRequest(triggerId, tenantId);
        return syncSchedule(request, ConfigConstant.POST_SCHEDULE_URL + "/start", HttpMethod.POST, "启用排程");
    }

    @Override
    public ResultDto deleteSchedule(String triggerId, String tenantId) {
        ScheduleBaseRequest request = new ScheduleBaseRequest(triggerId, tenantId);
        return syncSchedule(request, ConfigConstant.POST_SCHEDULE_URL, HttpMethod.DELETE, "删除排程");
    }

    @Override
    public ResultDto<String> getScheduleStatus(String triggerId, String tenantId, String token) {
        ScheduleBaseRequest request = new ScheduleBaseRequest(triggerId, tenantId);
        LOGGER.info("侦测调用排程引擎获取排程实例状态请求参数为:[{}]", JsonUtil.getJsonString(request));
        JSONObject scheduleResp = execute(ConfigConstant.POST_SCHEDULE_URL + "/query", request, token, JSONObject.class, HttpMethod.POST);
        LOGGER.info("侦测调用排程引擎获取排程实例状态返回结果:[{}]", JsonUtil.getJsonString(scheduleResp));
        JSONObject scheduleInfo = scheduleResp.getJSONObject(FieldConstant.DW_SERVICE_RESPONSE);
        if (CollectionUtil.isEmpty(scheduleInfo)) {
            return ResultHelper.generateFailResult(ErrorCodeEnum.SCHEDULE_NOT_EXIST, FieldConstant.SCHEDULE_NOT_EXIST);
        }
        return ResultHelper.generateResult(true, "", scheduleInfo.getString(FieldConstant.ENABLE_STATUS));
    }

    @Override
    public ResultDto<Boolean> isScheduleDisabled(String triggerId, String tenantId, String token) {
        /** 获取原侦测对应的排程状态*/
        ResultDto<String> statusResult = this.getScheduleStatus(triggerId, tenantId, token);
        if (MonitorHelper.isResultFail(statusResult)) {
            return ResultHelper.generateFailResult(statusResult.getMessage());
        }
        return ResultHelper.generateSuccessResult(FieldConstant.ENABLE_STATUS_N.equals(statusResult.getData()));
    }



    private ResultDto syncSchedule(Object request, String url, HttpMethod httpMethod, String opName) {
        try {
            LOGGER.info("侦测调用排程引擎请求参数:[{}]", JsonUtil.getJsonString(request));
            JSONObject scheduleResult = execute(url, request, DWServiceContext.getContext().getToken(), JSONObject.class, httpMethod);
            LOGGER.info("侦测调用排程引擎返回结果:[{}]", JsonUtil.getJsonString(scheduleResult));
            /** 如果排程引擎创建任意一个失败，则respObject为空 */
            boolean isScheduleSuccess = CollectionUtil.isNotEmpty(scheduleResult);
            /** 判断调用排程组件 更新排程是否成功 */
            if (!isScheduleSuccess) {
                LOGGER.error("侦测调用排程引擎异常，请查看！");
                return ResultHelper.generateFailResult(ErrorCodeEnum.RULE_SCHEDULE_SYNC_FAILED);
            }
            return ResultHelper.generateSuccessResult(null);
        } catch (Exception e) {
            LOGGER.error("调用排程模组{}服务调用异常", opName, e);
            return ResultHelper.generateFailResult(ErrorCodeEnum.RULE_SCHEDULE_SYNC_FAILED);
        }
    }

    @Override
    protected ErrorCodeEnum getErrorCodeByHttpStatus(String url, HttpStatus httpStatus) {
        if (null == httpStatus) {
            return ErrorCodeEnum.SCHEDULE_OTHER_ERR;
        }
        switch (httpStatus) {
            case INTERNAL_SERVER_ERROR :
                return ErrorCodeEnum.SCHEDULE_500;
            case BAD_GATEWAY :
                return ErrorCodeEnum.SCHEDULE_502;
            case SERVICE_UNAVAILABLE :
                return ErrorCodeEnum.SCHEDULE_503;
            case GATEWAY_TIMEOUT :
                return ErrorCodeEnum.SCHEDULE_504;
        }
        return ErrorCodeEnum.SCHEDULE_OTHER_ERR;
    }
}
