package com.digiwin.athena.cdme.service.client;

import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.exception.ResponseStatusException;
import com.digiwin.athena.cdme.core.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.UnknownHttpStatusCodeException;

import java.util.function.Consumer;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/19 15:49
 */
public abstract class AbstractExecuteClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractExecuteClient.class);

    @Autowired
    private CustomHttpClient httpClient;

    protected <T> T post(String url, Object serviceParam, String token, Class<T> cls) {
        return execute(url, serviceParam, token, cls, HttpMethod.POST);
    }

    protected <T> T post(String url, Object serviceParam, String token, Class<T> cls, Consumer<HttpHeaders> headerAdder) {
        return execute(url, serviceParam, token, cls, HttpMethod.POST, headerAdder);
    }

    protected <T> T execute(String url, Object serviceParam, String token, Class<T> cls, HttpMethod httpMethod, Object... uriVariables) {
        return execute(url, serviceParam, token, cls, httpMethod, null, uriVariables);
    }

    protected <T> T execute(String url, Object serviceParam, String token, Class<T> cls, HttpMethod httpMethod, Consumer<HttpHeaders> headerAdder, Object... uriVariables) {
        HttpHeaders headers = initHeader(token);
        if (isNeedAuthInfo()) {
            if (headers == null) {
                headers = new HttpHeaders();
            }
            setAuthInfo(headers);
        }
        if (headerAdder != null) {
            if (headers == null) {
                headers = new HttpHeaders();
            }
            headerAdder.accept(headers);
        }
        try {
            return httpClient.execute(url, serviceParam, headers, cls, httpMethod, uriVariables);
        } catch (HttpClientErrorException e) {
            LOGGER.error("{}请求[{}]出现客户端4XX异常！报文体内容: {}", httpMethod, url, e.getResponseBodyAsString(), e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, e.getStatusCode()), e);
        } catch (HttpServerErrorException e) {
            LOGGER.error("{}请求[{}]出现服务端5XX异常！报文体内容: {}", httpMethod, url, e.getResponseBodyAsString(), e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, e.getStatusCode()), e);
        } catch (UnknownHttpStatusCodeException e) {
            LOGGER.error("{}请求[{}]出现未知状态码异常！报文体内容: {}", httpMethod, url, e.getResponseBodyAsString(), e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, null), e);
        } catch (Exception e) {
            LOGGER.error("{}请求[{}]出现异常！报文体内容: {}", httpMethod, url, e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, null), e);
        }
    }

    protected <T> T postWithoutRouterkey(String url, Object serviceParam, String token, Class<T> cls, Consumer<HttpHeaders> headerAdder) {
        HttpHeaders headers = new HttpHeaders();
        headers.set("token", token);
        headers.set("traceId", DWServiceContext.getContext().getTraceId());
        headerAdder.accept(headers);
        try {
            return httpClient.execute(url, serviceParam, headers, cls, HttpMethod.POST);
        } catch (HttpClientErrorException e) {
            LOGGER.error("{}请求[{}]出现客户端4XX异常！报文体内容: {}", HttpMethod.POST, url, e.getResponseBodyAsString(), e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, e.getStatusCode()), e);
        } catch (HttpServerErrorException e) {
            LOGGER.error("{}请求[{}]出现服务端5XX异常！报文体内容: {}", HttpMethod.POST, url, e.getResponseBodyAsString(), e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, e.getStatusCode()), e);
        } catch (UnknownHttpStatusCodeException e) {
            LOGGER.error("{}请求[{}]出现未知状态码异常！报文体内容: {}", HttpMethod.POST, url, e.getResponseBodyAsString(), e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, null), e);
        } catch (Exception e) {
            LOGGER.error("{}请求[{}]出现异常！报文体内容: {}", HttpMethod.POST, url, e);
            throw new ResponseStatusException(getErrorCodeByHttpStatus(url, null), e);
        }
    }

    protected HttpHeaders initHeader(String token) {
        /** NOTE 不停机版更需要调用方在header中添加 routerKey:tenantId。組件間調用跟穩敏api調用有routerkey，中间件都不用传
         * 规范要求下发请求在head中区分routerKey大小写，确保K8S能够识别和路由。
         * */
        String routerKey = (String) DWServiceContext.getContext().getRequestHeader().get(FieldConstant.ROUTER_KEY);
        if (StringUtil.isEmpty(routerKey)) {
            throw new BusinessException(ErrorCodeEnum.ROUTE_KEY_EMPTY);
        }
        /** 设置header信息 */
        HttpHeaders headers = new HttpHeaders();
        headers.set(FieldConstant.ROUTER_KEY, routerKey);
        String locale = (String) DWServiceContext.getContext().getRequestHeader().get(FieldConstant.LOCALE);
        if (StringUtil.isNotBlank(locale)) {
            headers.set(FieldConstant.LOCALE, locale);
        }
        headers.set("token", token);
        headers.set("traceId", DWServiceContext.getContext().getTraceId());
        return headers;
    }

    /**
     * 获取http异常对应错误信息
     * @param url
     * @param httpStatus
     * @return
     */
    protected abstract ErrorCodeEnum getErrorCodeByHttpStatus(String url, HttpStatus httpStatus);

    /**
     * 是否需要在head中加入授权信息
     */
    protected boolean isNeedAuthInfo() {
        return false;
    }

    private void setAuthInfo(HttpHeaders headers) {
        headers.set(FieldConstant.DW_APP_TOKEN_HEADER_NAME, ConfigConstant.MULTIPLEAPPENV?ConfigConstant.MERGED_IAM_APP_TOKEN:ConfigConstant.IAM_APP_TOKEN);
        headers.set(FieldConstant.DW_TENANT_TOKEN_HEADER_NAME, DWServiceContext.getContext().getToken());
    }
}
