package com.digiwin.athena.cdme.service.facade.detection.data.impl;

import com.alibaba.fastjson.JSONArray;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.exception.RepositoryException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.dto.SyncRuleParamDto;
import com.digiwin.athena.cdme.pojo.dto.TriggerRepDto;
import com.digiwin.athena.cdme.repository.model.MonitorDupModel;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerInsModel;
import com.digiwin.athena.cdme.service.facade.detection.data.IDupDataCacheFacadeService;
import com.digiwin.athena.cdme.service.facade.rulesync.ISyncFacadeService;
import com.digiwin.athena.cdme.service.srp.cache.ICacheService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorDupService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import java.util.Set;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/19 15:33
 */
@Service("cdmeDupDataCacheFacadeService")
public class DupDataCacheFacadeService implements IDupDataCacheFacadeService {

    private static final Logger LOGGER = LoggerFactory.getLogger(DupDataCacheFacadeService.class);

    private final ICacheService redisService;

    private final IMonitorDupService dupService;

    private final ISyncFacadeService disabledService;

    public DupDataCacheFacadeService(ICacheService redisService, IMonitorDupService dupService, @Qualifier(FieldValConstant.MONITOR_RULE_DISABLED) ISyncFacadeService disabledService) {
        this.redisService = redisService;
        this.dupService = dupService;
        this.disabledService = disabledService;
    }

    @Override
    @Transactional(value = "dw-transactionManager")
    public void syncProcess(MonitorRuleModel ruleModel, EocDto eocDto, JSONArray changeData, JSONArray dupdata, String redisKey, boolean isAppend) {
        JSONArray oldCache = null;
        boolean readOldCacheSuccess = false;
        try {
            JSONArray newCache = new JSONArray();
            oldCache = redisService.get(redisKey);
            readOldCacheSuccess = true;
            /** 分页发起侦测部分成功，需要把成功的部分追加到redis和db中 */
            if (isAppend) {
                if (CollectionUtil.isNotEmpty(oldCache)){
                    newCache.addAll(oldCache);
                }
            } else if (CollectionUtil.isNotEmpty(dupdata)) {
                newCache.addAll(dupdata);
            }
            if (changeData != null) {
                newCache.addAll(changeData);
            }
            /** 更新缓存 NOTE 平台封装的redis接口都有问题！目前内部都吞了对应的连接异常, 改为应用自实现处理*/
            redisService.set(redisKey, newCache);
            LOGGER.info("..... 更新redis：redis-value：{}  by key:{}", newCache, redisKey);

            /** 更新DB */
            MonitorDupModel dupModel = new MonitorDupModel(ruleModel, eocDto, newCache.toJSONString());
            MonitorDupModel monitorDup = dupService.getByRuleIdAndTenantIdAndProdAndEocmap(ruleModel.getRuleId(), ruleModel.getTenantId(), ruleModel.getProductName(), eocDto);
            boolean cacheFlag;
            if (monitorDup == null) {
                cacheFlag = dupService.save(dupModel);
            } else {
                dupModel.setId(monitorDup.getId());
                cacheFlag = dupService.edit(dupModel);
            }

            /** NOTE 当前业务场景DB操作异常或redis操作异常后的处理方式相同：还原redis数据，回滚事务*/
            if (!cacheFlag) {
                throw new RepositoryException(ErrorCodeEnum.SAVE_DUP_FAIL);
            }
        } catch (Exception e) {
            LOGGER.error("同步更新Redis/DB数据异常", e);
            /** 还原redis数据，回滚数据库 */
            try {
                if (readOldCacheSuccess) {
                    redisService.set(redisKey, oldCache);
                }
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            } catch (Exception e1) {
                LOGGER.error("同步还原Redis/DB数据异常", e1);
            }
            processIfSyncFail(ruleModel, eocDto, changeData, redisKey, oldCache);
            throw new RepositoryException(ErrorCodeEnum.SAVE_DUP_FAIL);
        }
    }

    @Transactional(value = "dw-transactionManager")
    @Override
    public ResultDto<Void> cleanDeduplicateData(MonitorRuleModel ruleModel, EocDto eocDto) {
        /**清除db*/
        dupService.deleteByRuleIdAndTenantIdAndEocmap(ruleModel.getRuleId(), ruleModel.getTenantId(), eocDto);
        /** 删除此规则对应的redis缓存记录 */
        StringBuilder patternKey = new StringBuilder(FieldConstant.REDIS_KEY_PREFIX)
                .append(":").append(ruleModel.getTenantId())
                .append(":").append(ruleModel.getRuleId());
        if (StringUtil.isNotBlank(eocDto.getEocCompanyId())) {
            patternKey.append(":").append(eocDto.getEocCompanyId());
        }
        if (StringUtil.isNotBlank(eocDto.getEocSiteId())) {
            patternKey.append(":").append(eocDto.getEocSiteId());
        }
        Set<String> keys = redisService.keys(patternKey.append(":*").toString());
        boolean isSuccess = true;
        if (CollectionUtil.isNotEmpty(keys)) {
            isSuccess = redisService.delete(keys);
        }
        if (!isSuccess) {
            throw new RepositoryException(ErrorCodeEnum.CLEAN_DUP_FAIL);
        }
        return ResultHelper.generateSuccessResult(null);
    }

    @Transactional(value = "dw-transactionManager")
    @Override
    public ResultDto<Void> cleanDeduplicateData(MonitorRuleModel ruleModel, TriggerRepDto triggerRepDto) {
        /**清除db*/
        MonitorTriggerInsModel triggerIns = triggerRepDto.getIns();
        EocDto eocDto = new EocDto(triggerIns.getEocCompanyId(), triggerIns.getEocSiteId(), null);
        dupService.deleteByRuleIdAndTenantIdAndEocmap(ruleModel.getRuleId(), ruleModel.getTenantId(), eocDto);
        /**清除redis*/
        String redisKey = new StringBuilder(FieldConstant.REDIS_KEY_PREFIX)
                .append(":").append(ruleModel.getTenantId())
                .append(":").append(ruleModel.getRuleId())
                .append(":").append(StringUtil.nullToEmpty(triggerRepDto.getIns().getEocCompanyId()))
                .append(":").append(StringUtil.nullToEmpty(triggerRepDto.getIns().getEocSiteId()))
                .append(":").append(ruleModel.getProductName())
                .toString();
        Set<String> keys = redisService.keys(redisKey);
        boolean isSuccess = true;
        if (CollectionUtil.isNotEmpty(keys)) {
            isSuccess = redisService.delete(keys);
        }
        if (!isSuccess) {
            throw new RepositoryException(ErrorCodeEnum.CLEAN_DUP_FAIL);
        }
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public boolean deleteCacheByKey(String cacheKey) {
        Set<String> keys = redisService.keys(cacheKey);
        if (CollectionUtil.isNotEmpty(keys)) {
            return redisService.delete(keys);
        }
        return true;
    }

    @Override
    public ResultDto<Void> saveDupData(MonitorRuleModel ruleModel, EocDto eocDto, String cacheKey, JSONArray data) {
        redisService.set(cacheKey, data);
        MonitorDupModel dupModel = new MonitorDupModel(ruleModel, eocDto.getEocCompanyId(), eocDto.getEocSiteId(), data.toJSONString());
        dupService.save(dupModel);
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public ResultDto<Void> cacheDupData(String cacheKey, JSONArray data) {
        redisService.set(cacheKey, data);
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public JSONArray getDupData(String cacheKey) {
        return redisService.get(cacheKey);
    }


    /**
     * 同步处理失败的时候简单的日志打印以及回滚处理措施
     *
     * @param ruleModel
     * @param eocDto
     * @param changeData
     * @param redisKey
     * @param oldCache
     */
    private void processIfSyncFail(MonitorRuleModel ruleModel, EocDto eocDto, JSONArray changeData, String redisKey, JSONArray oldCache) {
        /** 记录错误日志信息,发送通知并停止当前侦测*/
        LOGGER.error("！！！ [同步缓存与DB数据失败],redisKey:[{}], changeData:{} ", redisKey, changeData);
        LOGGER.error("！！！ [同步缓存与DB数据失败],ruleModel:[{}] eocDto:{} ", ruleModel, JsonUtil.getJsonString(eocDto));
        LOGGER.error("！！！ [同步缓存与DB数据失败],oldCache:[{}] ", oldCache);
        LOGGER.info("同步侦测规则入参:ruleId:{},tenantId:{},changeType:{}", ruleModel.getRuleId(), ruleModel.getTenantId(), FieldValConstant.MONITOR_RULE_DISABLED);
        DWServiceContext.getContext().getRequestHeader().put(FieldConstant.ROUTER_KEY, ruleModel.getTenantId());
        SyncRuleParamDto ruleDto = new SyncRuleParamDto(ruleModel.getTenantId(), ruleModel.getRuleId(),
                FieldValConstant.MONITOR_RULE_DISABLED, eocDto, null, null);
        ResultDto synMonitorRuleResult = disabledService.syncMonitorRuleHandler(ruleDto);
        if (MonitorHelper.isResultFail(synMonitorRuleResult)) {
            LOGGER.error("停用侦测规则:ruleId:{},tenantId:{},changeType:{}失败！", ruleModel.getRuleId(), ruleModel.getTenantId(), FieldValConstant.MONITOR_RULE_DISABLED);
            throw new BusinessException(ErrorCodeEnum.DISABLED_RULE_FAIL);
        }
    }

}
