package com.digiwin.athena.cdme.mq.consumer.monitor;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.mq.producer.MessageProducer;
import com.digiwin.athena.cdme.repository.model.MonitorCdcTopicRouteModel;
import com.digiwin.athena.cdme.service.client.IRouterClient;
import com.digiwin.athena.cdme.service.facade.auth.IContextFacadeService;
import com.digiwin.athena.cdme.service.srp.cache.ICacheService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorCdcTopicRouteService;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;

/**
 * @author yangxiao
 * @description: 监听BM_MOSC_CDC Topic的侦测消息
 * @date 2023/10/10 11:10
 */
@Component("cdmeMonitorTransferCdcListener")
public class MonitorTransferCdcListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(MonitorTransferCdcListener.class);

    private final MessageProducer producer;

    private IMonitorCdcTopicRouteService  cdcTopicRouteService;

    private final ICacheService redisService;

    private final IRouterClient routerClient;

    private final IContextFacadeService contextService;

    public MonitorTransferCdcListener(IMonitorCdcTopicRouteService cdcTopicRouteService, ICacheService redisService, MessageProducer producer, IRouterClient routerClient, IContextFacadeService contextService) {
        this.cdcTopicRouteService = cdcTopicRouteService;
        this.producer = producer;
        this.redisService = redisService;
        this.routerClient = routerClient;
        this.contextService = contextService;
    }

    @KafkaListener(topics = {ConfigConstant.PRIVATIZATION_CDC_MQ_TOPIC}, groupId = ConfigConstant.CDME_MQ_GROUP, containerFactory = ConfigConstant.CDC_MQ_FACTORY, errorHandler = "cdmeConsumerAwareErrorHandler")
    public void consumeMonitorMsg(ConsumerRecord<?, ?> record, Acknowledgment ack) throws Exception {
        LOGGER.info("侦测引擎接收CDC的转发消息来自私有化0 Topic为:[{}]", record.value());
        if (!ConfigConstant.PRIVATIZATION) {
            // 如果非私有化部署，不处理该消息
            LOGGER.info("非私有化部署，不处理===========》");
            ack.acknowledge();
            return;
        }

        JSONObject recordJsonObject = JSON.parseObject(String.valueOf(record.value()));
        String redisKey = MonitorHelper.buildCdcTopicRouteKey(ConfigConstant.PRIVATIZATION_CDC_MQ_TOPIC, ConfigConstant.CDME_MQ_GROUP);

        if(redisService.hasKey(redisKey)){
            recordJsonObject.put(FieldConstant.TENANTID, redisService.get(redisKey));
            //producer.sendMessage(JSON.toJSONString(recordJsonObject));
        } else {
            MonitorCdcTopicRouteModel cdcTopicRouteModel = cdcTopicRouteService.getByTopicAndGroup(ConfigConstant.PRIVATIZATION_CDC_MQ_TOPIC, ConfigConstant.CDME_MQ_GROUP);
            if(null == cdcTopicRouteModel){
                LOGGER.error("未配置租户和Topic的关系，topic={}, group={}", ConfigConstant.PRIVATIZATION_CDC_MQ_TOPIC, ConfigConstant.CDME_MQ_GROUP);
            } else {
                recordJsonObject.put(FieldConstant.TENANTID, cdcTopicRouteModel.getTenantId());
                //producer.sendMessage(JSON.toJSONString(recordJsonObject));
                // 将租户ID放入缓存
                redisService.set(redisKey, cdcTopicRouteModel.getTenantId());
            }
        }

        String tenantId = recordJsonObject.getString(FieldConstant.TENANTID);

        if(StringUtils.isNotEmpty(tenantId)){
            try {
                contextService.constructContext(tenantId);
                routerClient.deliver(JSON.toJSONString(recordJsonObject), tenantId);
            } catch (Throwable e) {
                LOGGER.error("处理私有化0的消息构建上下文失败，tenantId={}，异常:[{}]", recordJsonObject.getString(FieldConstant.TENANTID), e);
               throw e;
            }
        }

        ack.acknowledge();
    }
}
