package com.digiwin.athena.cdme.provider;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.aop.ExceptionRetry;
import com.digiwin.athena.cdme.core.aop.RouterKey;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.enums.MonitorOpsEnum;
import com.digiwin.athena.cdme.core.exception.ArgumentValidException;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.LocalTimeUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.core.util.SecurityUtil;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.dto.RuleOpsDto;
import com.digiwin.athena.cdme.pojo.vo.MonitorOpsDetailVo;
import com.digiwin.athena.cdme.pojo.vo.MonitorOpsRuleVo;
import com.digiwin.athena.cdme.pojo.vo.SyncRuleInfoVo;
import com.digiwin.athena.cdme.provider.converter.EocDtoConverter;
import com.digiwin.athena.cdme.provider.converter.MonitorOpsVoConverter;
import com.digiwin.athena.cdme.provider.converter.RuleVoConverter;
import com.digiwin.athena.cdme.repository.model.MonitorOptRecordModel;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerModel;
import com.digiwin.athena.cdme.service.client.IIamClient;
import com.digiwin.athena.cdme.service.client.IScheduleClient;
import com.digiwin.athena.cdme.service.facade.auth.ITokenFacadeService;
import com.digiwin.athena.cdme.service.facade.ops.IRuleTriggerOperatorFacadeService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorOptRecordService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRuleService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorTriggerService;
import javax.xml.ws.ServiceMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.springframework.stereotype.Service;

/**
 * @description: 侦测运维相关接口
 * @author: dongwh
 * @date: 2020/11/4 10:53
 */
@Service("cdmeMonitorOpsService")
public class MonitorOpsService implements IMonitorOpsService {
    private static final Logger LOGGER = LoggerFactory.getLogger(MonitorOpsService.class);

    @Autowired
    private IMonitorRuleService ruleService;

    @Autowired
    private IRuleTriggerOperatorFacadeService ruleTriggerOperatorFacadeService;

    @Autowired
    private IMonitorTriggerService triggerService;

    @Autowired
    private IMonitorOptRecordService optRecordService;

    @Autowired
    protected IScheduleClient scheduleClient;

    @Autowired
    private IIamClient iamClient;

    @Autowired
    private ITokenFacadeService tokenFacadeService;

    @Override
    @RouterKey
    @ExceptionRetry(times = 3, waitTime = 300)
    public JSONArray postMonitorInfo(String ruleId, String tenantId, Map<String, String> eocMap, String secure) {
        LOGGER.info("操作postMonitorInfo的入参:ruleId:{},tenantId:{},eocMap:{}", ruleId, tenantId, eocMap);

        /** 入参非空校验*/
        if (StringUtil.isBlank(tenantId)) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 判断租户是否是当前环境的测试租户，如果是测试租户不需要校验secure，非测试租户需要验证secure是否正确 */
        boolean isTestTenant = iamClient.isTest(tenantId);
        if (!isTestTenant && !SecurityUtil.verifySecure(secure, FieldConstant.SECURE_TYPE_MONITOROPS)) {
            throw new ArgumentValidException(ErrorCodeEnum.SECURE_ERR);
        }
        /** 将入参map转换为Dto对象 */
        EocDto eocDto = EocDtoConverter.convertEocDtoByCamelCaseMap(eocMap);
        /** 入参只有租户ID时,返回该租户下的所有侦测规则概要信息*/
        if (StringUtil.isBlank(ruleId)) {
            /*查询该租户信息*/
            List<MonitorRuleModel> rules = ruleService.listByTenantIdAndEoc(tenantId, eocDto);
            if (CollectionUtil.isEmpty(rules)) {
                return new JSONArray();
            }
            /** JSONARRAY强转是为了不让所有字段都返回处理 */
            return JsonUtil.getJsonArray(rules.stream().map(one -> MonitorOpsVoConverter.convertModelToOpsSummaryVo(one))
                    .collect(Collectors.toList()));
        }

        /** 入参有租户ID和规则ID时，查询侦测规则详情 **/
        List<RuleOpsDto> details = ruleTriggerOperatorFacadeService.listRuleDetailByRuleIdAndEoc(ruleId, tenantId, eocDto);
        if (CollectionUtil.isEmpty(details)) {
            return new JSONArray();
        }
        List<MonitorOpsDetailVo> voList = details.stream().map(dto -> {
            MonitorOpsDetailVo vo = MonitorOpsVoConverter.convertOpsDetailVoByDto(dto);
            if (null != dto.getTriggerModel()) {
                /** 获取当前租户对应的token*/
                String token = tokenFacadeService.getTenantToken(tenantId);
                /** 查询排程实例状态 **/
                ResultDto resultSchedule = isScheduleDisabled(dto.getTriggerModel().getId(), tenantId, token);
                /** 排程实例状态逻辑处理 **/
                String scheduleInstanceStatus = MonitorOpsEnum.SCHEDULE_ISENABLE_N.getDesc();
                if (MonitorHelper.isResultFail(resultSchedule)) {
                    scheduleInstanceStatus = FieldConstant.SCHEDULE_IS_ENABLED.equals(resultSchedule.getData()) ? MonitorOpsEnum.SCHEDULE_ISENABLE_Y.getDesc() : resultSchedule.getMessage();
                }
                vo.setScheduleInstanceStatus(scheduleInstanceStatus);
            }
            return vo;
        }).collect(Collectors.toList());

        return JsonUtil.getJsonArray(voList);
    }

    @Override
    @RouterKey
    @Deprecated
    public String postBacktrackDate(String ruleId, String tenantId, Map<String, String> eocMap, String backtrackDate) {
        return notice();
    }

    @Override
    @RouterKey
    @Deprecated
    public String postBacktrackDate(String ruleId, String tenantId, Map<String, String> eocMap, String backtrackDate, String secure) {
        return notice();
    }

    /**
     * 不做处理，返回公告信息
     */
    @Deprecated
    private String notice() {
        return "当前接口已废弃，请使用规则同步接口：/restful/standard/MonitorEngine/rule/sync(changeType为ENABLED)；\n详情可参考doclever文档：http://athena-devops-doclever.digiwincloud.com.cn/html/web/controller/share/share.html#5eb275575f9aae000c969cac";
    }

    /**
     * 检查排程状态是否停用。
     * note 调用此接口需确认传入token是当前租户对应的token
     *
     * @param triggerId
     * @param tenantId
     * @param token
     * @return
     */
    private ResultDto isScheduleDisabled(String triggerId, String tenantId, String token) {
        /** 获取原侦测对应的排程状态*/
        ResultDto<String> statusResult = scheduleClient.getScheduleStatus(triggerId, tenantId, token);
        if (MonitorHelper.isResultFail(statusResult)) {
            return statusResult;
        }
        if (!FieldConstant.ENABLE_STATUS_N.equals(statusResult.getData())) {
            return ResultHelper.generateFailResult(ErrorCodeEnum.SCHEDULE_IS_ENABLED);
        }
        return ResultHelper.generateResult(true, "");
    }

    /**
     * 根据测试租户id获取租户token
     *
     * @param tenantId
     * @return
     */
    @Override
    @RouterKey
    @ExceptionRetry(times = 3, waitTime = 300)
    public String getTestUserToken(String tenantId) {
        LOGGER.info("根据测试租户id获取租户token入参:tenantId:{},时间:{}", tenantId, LocalTimeUtil.getCurrentDateTime());
        if (StringUtil.isBlank(tenantId)) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 获取token **/
        String token = tokenFacadeService.getTenantToken(tenantId);
        /** 将token添加到上下文中，查询是否测试租户需要使用 **/
        DWServiceContext.getContext().setToken(token);
        /** 当前租户非测试租户不允许手动获取token*/
        boolean isTestTenant = iamClient.isTest(tenantId);
        if (!isTestTenant) {
            throw new BusinessException(ErrorCodeEnum.NOT_TEST_TENANT);
        }
        return token;
    }

    /**
     * 根据secure及租户id获取租户token
     * 验证secure合法后返回正确的token
     *
     * @param tenantId
     * @return
     */
    @Override
    @RouterKey
    public String getSecureUserToken(String tenantId, String secure) {
        LOGGER.info("根据secure及租户id获取租户token入参:tenantId:{},时间:{}", tenantId, LocalTimeUtil.getCurrentDateTime());
        if (StringUtil.isBlank(tenantId) || StringUtil.isBlank(secure)) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 验证当前加密串是否正确*/
        boolean checkSecure = SecurityUtil.verifySecure(secure, FieldConstant.SECURE_TYPE_REDISOPS);
        if (!checkSecure) {
            throw new ArgumentValidException(ErrorCodeEnum.SECURE_ERR);
        }
        /** 获取token并返回 **/
        return tokenFacadeService.getTenantToken(tenantId);

    }

    @Override
    @RouterKey
    @ExceptionRetry(times = 3, waitTime = 300)
    public String postSetPageSize(String ruleId, String tenantId, Map<String, String> eocMap, Integer pageSize) {
        LOGGER.info("设置测试租户分页下发执行的侦测数量入参:ruleId:{},tenantId:{},eocMap:{},pageSize:{}, 时间:{}", ruleId, tenantId, eocMap, pageSize, LocalTimeUtil.getCurrentDateTime());
        /** 校验入参合法性 */
        boolean paramValid = StringUtil.isBlank(ruleId) || StringUtil.isBlank(tenantId) || pageSize <= 0;
        if (paramValid) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 校验是否是测试租户 */
        boolean isTestTenant = iamClient.isTest(tenantId);
        if (!isTestTenant) {
            throw new BusinessException(ErrorCodeEnum.NOT_TEST_TENANT);
        }
        return setPageSize(ruleId, tenantId, eocMap, pageSize);
    }

    @Override
    @RouterKey
    public String postSetPageSize(String ruleId, String tenantId, Map<String, String> eocMap, Integer pageSize, String secure) {
        LOGGER.info("设置分页下发执行的侦测数量入参:ruleId:{},tenantId:{},eocMap:{},pageSize:{}, 时间:{}", ruleId, tenantId, eocMap, pageSize, LocalTimeUtil.getCurrentDateTime());
        /** 校验入参合法性 */
        boolean paramValid = StringUtil.isBlank(ruleId) || StringUtil.isBlank(tenantId) || pageSize <= 0;
        if (paramValid) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 验证当前加密串是否正确*/
        boolean checkSecure = SecurityUtil.verifySecure(secure, FieldConstant.SECURE_TYPE_MONITOROPS);
        if (!checkSecure) {
            throw new ArgumentValidException(ErrorCodeEnum.SECURE_ERR);
        }
        return setPageSize(ruleId, tenantId, eocMap, pageSize);
    }

    private String setPageSize(String ruleId, String tenantId, Map<String, String> eocMap, Integer pageSize) {
        /** 更改侦测分页数据量 */
        EocDto eocDto = EocDtoConverter.convertEocDtoByCamelCaseMap(eocMap);
        boolean isSuccess = ruleService.editPageSizeByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto, pageSize);
        optRecordService.save(new MonitorOptRecordModel(tenantId, ruleId, pageSize, eocDto, isSuccess));
        return isSuccess ? "SUCCESS" : ResultHelper.generateFailResult(ErrorCodeEnum.RULE_NOT_EXISTS).getMessage();
    }

    @Override
    @RouterKey
    public Integer postGetPageSize(String ruleId, String tenantId, Map<String, String> eocMap) {
        LOGGER.info("获取分页下发执行数据量入参:ruleId:{},tenantId:{},eocMap:{}, 时间:{}", ruleId, tenantId, eocMap, LocalTimeUtil.getCurrentDateTime());
        EocDto eocDto = EocDtoConverter.convertEocDtoByCamelCaseMap(eocMap);
        MonitorRuleModel ruleModel = ruleService.getByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto);
        if (null == ruleModel) {
            throw new BusinessException(ErrorCodeEnum.RULE_NOT_EXISTS);
        }
        return ruleModel.getPageSize();
    }

    @Override
    @RouterKey
    @ExceptionRetry(times = 3, waitTime = 300)
    public SyncRuleInfoVo postGetMonitorRule(String ruleId, String tenantId, Map<String, String> eocMap) {
        LOGGER.info("检查测试租户同步结果是否成功入参:ruleId:{},tenantId:{},eocMap:{}, 时间:{}", ruleId, tenantId, eocMap, LocalTimeUtil.getCurrentDateTime());
        /** 校验入参合法性 */
        boolean paramValid = StringUtil.isBlank(ruleId) || StringUtil.isBlank(tenantId);
        if (paramValid) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 当前租户非测试租户不允许手动发起侦测流程*/
        boolean isTestTenant = iamClient.isTest(tenantId);
        if (!isTestTenant) {
            throw new ArgumentValidException(ErrorCodeEnum.NOT_TEST_TENANT);
        }
        return getMonitorRule(ruleId, tenantId, eocMap);
    }

    @Override
    @RouterKey
    public SyncRuleInfoVo postGetMonitorRule(String ruleId, String tenantId, Map<String, String> eocMap, String secure) {
        LOGGER.info("检查同步结果是否成功入参:ruleId:{},tenantId:{},eocMap:{}, 时间:{}", ruleId, tenantId, eocMap, LocalTimeUtil.getCurrentDateTime());
        /** 校验入参合法性 */
        boolean paramValid = StringUtil.isBlank(ruleId) || StringUtil.isBlank(tenantId) || StringUtil.isBlank(secure);
        if (paramValid) {
            throw new ArgumentValidException(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /** 验证当前加密串是否正确*/
        boolean checkSecure = SecurityUtil.verifySecure(secure, FieldConstant.SECURE_TYPE_MONITOROPS);
        if (!checkSecure) {
            throw new ArgumentValidException(ErrorCodeEnum.SECURE_ERR);
        }
        return getMonitorRule(ruleId, tenantId, eocMap);
    }

    private SyncRuleInfoVo getMonitorRule(String ruleId, String tenantId, Map<String, String> eocMap) {
        EocDto eocDto = EocDtoConverter.convertEocDtoByCamelCaseMap(eocMap);
        MonitorRuleModel monitorRule = ruleService.getByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto);
        if (null == monitorRule) {
            LOGGER.error("入参ruleId:{},tenantId:{},eocMap:{}未查询到对应的侦测规则，请查看！", ruleId, tenantId, eocMap);
            return null;
        }
        MonitorOpsRuleVo monitorOpsRuleVo = RuleVoConverter.convertModelToVo(monitorRule);
        /** monitorTrigger不存在 */
        if (FieldValConstant.CATEGORY_REPORT.equals(monitorRule.getCategory())) {
            return new SyncRuleInfoVo(new JSONObject(), monitorOpsRuleVo, false);
        } else {
            MonitorTriggerModel monitorTrigger = triggerService.getByTriggerId(monitorOpsRuleVo.getTriggerId());
            JSON standardPollingRule = JsonUtil.parse(monitorTrigger.getTriggerParam());
            /** 获取当前租户对应的token*/
            String token = tokenFacadeService.getTenantToken(tenantId);
            /** 获取当前侦测规则的排程启停状态 */
            ResultDto resultSchedule = isScheduleDisabled(monitorTrigger.getId(), tenantId, token);
            /** 排程实例状态逻辑处理 **/
            boolean scheduleInstanceStatus = false;
            if (MonitorHelper.isResultFail(resultSchedule)) {
                scheduleInstanceStatus = FieldConstant.SCHEDULE_IS_ENABLED.equals(resultSchedule.getData());
            }
            return new SyncRuleInfoVo(standardPollingRule, monitorOpsRuleVo, scheduleInstanceStatus);
        }
    }
}