package com.digiwin.athena.cdme.mq.consumer.mqtt;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.iot.mqtt.callback.DWMqttCallback;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.constant.MqttConstant;
import com.digiwin.athena.cdme.core.thread.MqttMsgThread;
import com.digiwin.athena.cdme.core.util.ActionParamsUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import com.digiwin.athena.cdme.repository.model.MqttServerConfigModel;
import com.digiwin.athena.cdme.service.facade.detection.IMonitorFacadeService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRuleCdcService;
import com.digiwin.athena.cdme.service.srp.db.IMqttServerConfigService;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author yang.xiao
 * @version V1.0
 * @Description
 * @date 2024/6/24 17:01
 * @Copyright 鼎捷软件股份有限公司
 */
public class MqttConsumerCallback extends DWMqttCallback {
    private static final Logger LOGGER = LoggerFactory.getLogger(MqttConsumerCallback.class);
    private final IMonitorFacadeService monitorFacadeService;
    private final IMonitorRuleCdcService monitorRuleCdcService;
    private final ThreadPoolTaskExecutor cdmeMsgExecutor;
    private final IMqttServerConfigService mqttServerConfigService;


    public MqttConsumerCallback(IMonitorFacadeService monitorFacadeService, IMonitorRuleCdcService monitorRuleCdcService,
                                ThreadPoolTaskExecutor cdmeMsgExecutor,IMqttServerConfigService mqttServerConfigService) {
        this.monitorFacadeService = monitorFacadeService;
        this.monitorRuleCdcService = monitorRuleCdcService;
        this.cdmeMsgExecutor = cdmeMsgExecutor;
        this.mqttServerConfigService=mqttServerConfigService;
    }


    public void messageArrived(String topic, MqttMessage message) {
        LOGGER.info("接收消息主题 : {}, 接收消息内容 : {}",topic, new String(message.getPayload()));
        if(StringUtil.isEmpty(topic)){
            LOGGER.warn("主题为空，消息不处理");
            return;
        }
        String[] topicArray = topic.split("/");
        if(topicArray.length < 6){
            LOGGER.warn("主题格式不符合规范，消息不处理，topic={}", topic);
            return;
        }

        StringBuffer topicName = new StringBuffer();
        topicName.append(topicArray[5]);
        for(int i = 6; i < topicArray.length; i++){
            topicName.append("/").append(topicArray[i]);
        }
        JSONObject msgJsonObjec = JSON.parseObject(new String(message.getPayload()));
        String tenantId = msgJsonObjec.getString(MqttConstant.MSG_TENANT_ID);
        LOGGER.info("接收消息的业务主题：{}，租户：{}", topicName, tenantId);
        if(StringUtils.isBlank(tenantId)){
            LOGGER.warn("消息报文中缺少租户信息，tenantId={}", tenantId);
            return;
        }
        List<JSONObject> uniJsonObjectList = converter(topicArray[0],tenantId, topicName.toString(), msgJsonObjec);

        if(CollectionUtils.isEmpty(uniJsonObjectList)){
            LOGGER.info("消息没有匹配的侦测规则，不做后续处理");
            return;
        }
        LOGGER.info("转化后的消息条数：{}",uniJsonObjectList.size());
        uniJsonObjectList.forEach(uniJsonObject->{
            cdmeMsgExecutor.execute(new MqttMsgThread(monitorFacadeService, uniJsonObject, tenantId, topicName.toString()));
        });

    }

    private List<JSONObject> converter(String topicArea,String tenantId, String topicName, JSONObject msgJsonObjec) {
        List<JSONObject> uniJsonObjectList = Lists.newArrayList();
        List<MonitorRuleCdcModel> byTenantIdAndTableAndChangeType =
                monitorRuleCdcService.getByTenantIdAndTableAndChangeType(tenantId, topicName, MonitorHelper.getChangeType(MqttConstant.UNI_OP_VAL));
        if(CollectionUtils.isNotEmpty(byTenantIdAndTableAndChangeType)){
            List<String> collect = byTenantIdAndTableAndChangeType.stream().map(res -> res.getDbName()).collect(Collectors.toList());
            List<MqttServerConfigModel> dbNameList = mqttServerConfigService.getServerList();
            if(CollectionUtils.isEmpty(dbNameList)){
                return uniJsonObjectList;
            }
            List<MqttServerConfigModel> collect1 = dbNameList.stream().filter(res -> collect.contains(res.getBusinessSources())
                    && StringUtils.equals(res.getZone(), topicArea)).collect(Collectors.toList());
            collect1.forEach(mqttServerConfigModel->{
                List<MonitorRuleCdcModel> monitorRuleCdcModelList = new ArrayList<>();
                monitorRuleCdcService.queryCdcRuleByTenantIdAndTableAndOp(monitorRuleCdcModelList, tenantId, mqttServerConfigModel.getBusinessSources(), topicName, MonitorHelper.getChangeType(MqttConstant.UNI_OP_VAL));
                if(CollectionUtils.isNotEmpty(monitorRuleCdcModelList)){
                    monitorRuleCdcModelList.forEach(cdcRule -> {
                        if(FieldValConstant.RULE_DISABLE.equals(cdcRule.getValid())){
                            LOGGER.info("侦测:{} 侦测规则禁用，无法发起侦测，tenantId={}，topicName={}",cdcRule.getRuleId(), tenantId, topicName);
                            return;
                        }
                        if (StringUtils.isEmpty(cdcRule.getActionParams())) {
                            LOGGER.error("侦测:{} 侦测规则没有配置数据映射，无法发起侦测，tenantId={}，topicName={}",cdcRule.getRuleId(), tenantId, topicName);
                            return;
                        }
                        JSONArray actionParams = JSON.parseArray(cdcRule.getActionParams());
                        ActionParamsUtil.parse(actionParams,msgJsonObjec).forEach(res->{
                            JSONObject uniJsonObject = new JSONObject();
                            uniJsonObject.put(MqttConstant.UNI_OP, MqttConstant.UNI_OP_VAL);
                            ((JSONObject)res).put(MqttConstant.UNI_TENANTID, tenantId);
                            ((JSONObject)res).put(MqttConstant.RULE_ID, cdcRule.getRuleId());
                            uniJsonObject.put(MqttConstant.UNI_AFTER, res);
                            JSONObject source = new JSONObject();
                            source.put(MqttConstant.UNI_DB, mqttServerConfigModel.getBusinessSources());
                            source.put(MqttConstant.UNI_TABLE, topicName);
                            uniJsonObject.put(MqttConstant.UNI_SOURCE, source);
                            uniJsonObject.put(MqttConstant.UNI_TS_MS, System.currentTimeMillis());
                            uniJsonObjectList.add(uniJsonObject);
                            LOGGER.info("侦测{}:统一格式消息：{}",cdcRule.getRuleId(), uniJsonObject.toJSONString());
                        });
                    });
                }
            });
        }
        return uniJsonObjectList;
    }
}