package com.digiwin.athena.cdme.service.client.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.digiwin.app.container.exceptions.DWRuntimeException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import com.digiwin.athena.cdme.core.constant.MqttConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.*;
import com.digiwin.athena.cdme.pojo.dto.*;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.repository.model.MonitorTriggerModel;
import com.digiwin.athena.cdme.service.client.AbstractExecuteClient;
import com.digiwin.athena.cdme.service.client.IThemeMapClient;
import com.digiwin.athena.cdme.service.client.request.ActionMetaDataReq;
import com.digiwin.athena.cdme.service.client.request.ExpireRuleReq;
import com.digiwin.athena.cdme.service.client.request.SyncRuleCallBackReq;
import com.digiwin.athena.cdme.service.client.response.ActionMetaDataRes;
import com.digiwin.athena.cdme.service.client.response.ExpireRuleRes;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/19 17:16
 */
@Service("cdmeThemeMapClient")
public class ThemeMapClient extends AbstractExecuteClient implements IThemeMapClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(ThemeMapClient.class);

    /**
     * LicenseKey的requestFrom
     */
    public static final String LICENSEKEY_REQUEST_FROM = "requestFrom";
    /**
     * LicenseKey的timestamp
     */
    public static final String LICENSEKEY_TIMESTAMP = "timestamp";
    /**
     * LicenseKey的key
     */
    public static final String LICENSEKEY_KEY = "key";

    private static final String COMPONENT = "component";

    @Override
    public String getTenantSecretKey(String tenantId) {
        Map<String, Object> licenseKeyParam = licenseKeyParam(tenantId);
        JSONObject httpRespOjb = post(ConfigConstant.POST_THEMEMAP_LICENSEKEY_URL, licenseKeyParam, DWServiceContext.getContext().getToken(), JSONObject.class);
        if (CollectionUtil.isNotEmpty(httpRespOjb) && FieldValConstant.RESPONSE_SUCCESS_CODE == httpRespOjb.getIntValue(FieldConstant.DW_SERVICE_STATUS)) {
            LOGGER.info("侦测调用ThemeMap获取LicenseKey成功！");
            String data = httpRespOjb.getString(FieldConstant.DW_SERVICE_RESPONSE);
            if (StringUtil.isEmpty(data)) {
                LOGGER.info("侦测调用ThemeMap获取LicenseKey为空！");
                throw new BusinessException(ErrorCodeEnum.KG_400);
            }
            return SecurityUtil.decodeAES(data);
        } else {
            LOGGER.error("侦测调用ThemeMap获取LicenseKey异常，返回结果为[{}]", httpRespOjb);
            throw new BusinessException(ErrorCodeEnum.KG_400);
        }
    }

    /**
     * 求取LicenseKey入参
     *
     * @param tenantId
     * @return
     */
    private Map<String, Object> licenseKeyParam(String tenantId) {
        Map<String, Object> param = new HashMap<>(8);
        long timestamp = System.currentTimeMillis();
        StringBuilder sb = new StringBuilder();
        String key = SecurityUtil.getMd5Hex(sb.append(tenantId).append(ConfigConstant.THEMEMAP_LICENSEKEY_REQUEST_FROM).append(timestamp).append(ConfigConstant.THEMEMAP_LICENSEKEY_MD5_KEY).toString());
        param.put(FieldConstant.TENANTID, tenantId);
        param.put(LICENSEKEY_REQUEST_FROM, ConfigConstant.THEMEMAP_LICENSEKEY_REQUEST_FROM);
        param.put(LICENSEKEY_TIMESTAMP, timestamp);
        param.put(LICENSEKEY_KEY, key);
        return param;
    }

    @Override
    public MonitorRuleDto getRuleInfo(String tenantId, EocDto eocDto, String ruleId) {
        Map<String, Object> config = new HashMap<>(4);
        config.put(FieldConstant.TENANTID, tenantId);
        config.put(FieldConstant.RULEID, ruleId);
        config.put(FieldConstant.EOCMAP, eocDto);
        Map<String, Object> param = new HashMap<>(1);
        param.put(FieldConstant.CONFIG, config);
        LOGGER.info("侦测调用ThemeMap获取侦测规则请求参数:[{}]", JsonUtil.getJsonString(param));
        JSONObject responseObj = post(ConfigConstant.THEMEMAP_MONITORRULE_URL,
                param, DWServiceContext.getContext().getToken(), JSONObject.class);
        LOGGER.info("侦测调用ThemeMap获取侦测规则返回结果:[{}]", JsonUtil.getJsonString(responseObj));
        if (CollectionUtil.isEmpty(responseObj)) {
            LOGGER.error("请求参数为：{},thememap侦测规则数据返回为空！", JsonUtil.getJsonString(param));
            throw new BusinessException(ErrorCodeEnum.KG_400);
        }

        JSONObject ruleObj = responseObj.getJSONObject(FieldConstant.DW_SERVICE_RESPONSE);
        if (CollectionUtil.isEmpty(ruleObj)) {
            LOGGER.error("请求参数为：{},thememap侦测规则返回response节点为空！", JsonUtil.getJsonString(param));
            throw new BusinessException(ErrorCodeEnum.KG_400);
        }

        return parseRuleObj(ruleObj);
    }

    /**
     * 获取侦测规则
     *
     * @param tenantId
     * @param eocDto
     * @param ruleId
     * @return
     */
    @Override
    public MonitorCdcRuleDto getCdcRuleInfo(String tenantId, EocDto eocDto, String ruleId) {
        Map<String, Object> config = new HashMap<>(4);
        config.put(FieldConstant.TENANTID, tenantId);
        config.put(FieldConstant.RULEID, ruleId);
        config.put(FieldConstant.EOCMAP, eocDto);
        Map<String, Object> param = new HashMap<>(1);
        param.put(FieldConstant.CONFIG, config);
        LOGGER.info("侦测调用ThemeMap获取侦测规则请求参数:[{}]", JsonUtil.getJsonString(param));
        JSONObject responseObj = post(ConfigConstant.THEMEMAP_MONITORRULE_URL,
                param, DWServiceContext.getContext().getToken(), JSONObject.class);
        LOGGER.info("侦测调用ThemeMap获取侦测规则返回结果:[{}]", JsonUtil.getJsonString(responseObj));
        if (CollectionUtil.isEmpty(responseObj)) {
            LOGGER.error("请求参数为：{},thememap侦测规则数据返回为空！", JsonUtil.getJsonString(param));
            throw new BusinessException(ErrorCodeEnum.KG_400);
        }


        JSONObject ruleObj = responseObj.getJSONObject(FieldConstant.DW_SERVICE_RESPONSE);
        if (CollectionUtil.isEmpty(ruleObj)) {
            LOGGER.error("请求参数为：{},thememap侦测规则返回response节点为空！", JsonUtil.getJsonString(param));
            throw new BusinessException(ErrorCodeEnum.KG_400);
        }

        return parseCdcRuleObj(ruleObj);
    }

    private MonitorCdcRuleDto parseCdcRuleObj(JSONObject ruleObj) {
        MonitorRuleCdcModel ruleModel = new MonitorRuleCdcModel(ruleObj);
        String category = ruleObj.getString(FieldConstant.CATEGORY);
        ruleObj.fluentRemove(FieldConstant.RULE_ID).fluentRemove(FieldConstant.TENANT_ID).fluentRemove(FieldConstant.ACTION_TYPE)
                .fluentRemove(FieldConstant.ACTION_ID).fluentRemove(FieldConstant.PRODUCT_NAME)
                .fluentRemove(FieldConstant.STANDARD_POLLING_RULES).fluentRemove(FieldConstant.MONITOR_ACTION_ID)
                .fluentRemove(FieldConstant.CATEGORY).fluentRemove(FieldConstant.PRODUCT_UID)
                .fluentRemove(FieldConstant.RULE_EXTENSION);

        ruleModel.setActionParams(ruleObj.getJSONArray("action_params").toJSONString());
        ruleModel.setChangeType(ruleObj.getString("change_type"));
        ruleModel.setEocCompanyId(StringUtil.emptyToNull(ruleObj.getString("eoc_company_id")));
        ruleModel.setEocSiteId(StringUtil.emptyToNull(ruleObj.getString("eoc_site_id")));
        ruleModel.setStartTime(LocalTimeUtil.parseTime(ruleObj.getString("start_time")));
        if(MqttConstant.CATEGORY.equals(category)){
            ruleModel.setTableName(ruleObj.getString(FieldConstant.MQTT_TOPIC));
            ruleModel.setDbName(ruleObj.getString(FieldConstant.MQTT_SOURCE));
            if(CollectionUtil.isNotEmpty(ruleObj.getJSONArray(FieldConstant.CONVERSION_FILTER_CONDITION))){
                ruleModel.setConversionFilterCondition(ruleObj.getJSONArray(FieldConstant.CONVERSION_FILTER_CONDITION).toJSONString());
            }else{
                ruleModel.setConversionFilterCondition(null);
            }

            if(Objects.isNull(ruleModel.getStartTime())){
                ruleModel.setStartTime(LocalTimeUtil.now());
            }
            if(Objects.isNull(ruleObj.getJSONArray("filter_condition"))){
                ruleModel.setFilterCondition(null);
            }else{
                ruleModel.setFilterCondition(ruleObj.getJSONArray("filter_condition").toJSONString());
            }
            ruleModel.setCategory(category);
        } else {
            ruleModel.setTableName(ruleObj.getString("table_name"));
            ruleModel.setChangeFields(ruleObj.getJSONArray("change_fields").toJSONString());
            ruleModel.setFilterCondition(ruleObj.getJSONArray("filter_condition").toJSONString());
        }

        return new MonitorCdcRuleDto(ruleModel);
    }

    private MonitorRuleDto parseRuleObj(JSONObject ruleObj) {
        MonitorRuleModel ruleModel = new MonitorRuleModel(ruleObj);
        MonitorTriggerModel triggerModel = new MonitorTriggerModel();
        if (!(FieldValConstant.CATEGORY_REPORT.equals(ruleModel.getCategory()) || FieldValConstant.CATEGORY_MQTT.equals(ruleModel.getCategory()))) {
            /** 排程周期 */
            JSONArray triggerParam = ruleObj.getJSONArray(FieldConstant.STANDARD_POLLING_RULES);
            if (CollectionUtil.isEmpty(triggerParam)) {
                throw new BusinessException(ErrorCodeEnum.RULE_SCHEDULE_EMPTY);
            }
            triggerModel = new MonitorTriggerModel(ruleModel.getRuleId(), ruleModel.getTenantId(), triggerParam.toJSONString());
        }
        JSONObject extensionConfig = ruleObj.getJSONObject(FieldConstant.RULE_EXTENSION);
        RuleExtensionDto extensionDto = CollectionUtil.isEmpty(extensionConfig) ? new RuleExtensionDto() : extensionConfig.toJavaObject(RuleExtensionDto.class);
        ruleObj.fluentRemove(FieldConstant.RULE_ID).fluentRemove(FieldConstant.TENANT_ID).fluentRemove(FieldConstant.ACTION_TYPE)
                .fluentRemove(FieldConstant.ACTION_ID).fluentRemove(FieldConstant.PRODUCT_NAME)
                .fluentRemove(FieldConstant.STANDARD_POLLING_RULES).fluentRemove(FieldConstant.MONITOR_ACTION_ID)
                .fluentRemove(FieldConstant.CATEGORY).fluentRemove(FieldConstant.PRODUCT_UID)
                .fluentRemove(FieldConstant.RULE_EXTENSION);
        ruleModel.setMonitorRule(JsonUtil.getJsonString(ruleObj, SerializerFeature.WriteMapNullValue));
        return new MonitorRuleDto(ruleModel, triggerModel, extensionDto);
    }

    @Override
    public ActionMetaDataRes getActionMetaData(String actionId) {
        ActionMetaDataReq requestParam = new ActionMetaDataReq(actionId);
        LOGGER.info("侦测调用ThemeMap获取action元数据请求参数:{}", requestParam);
        JSONObject responseObj = post(ConfigConstant.THEMEMAP_ACTIONMETADATA_URL,
                requestParam, DWServiceContext.getContext().getToken(), JSONObject.class);
        LOGGER.info("侦测调用ThemeMap获取action元数据返回结果:{}", responseObj);
        ResultDto<ActionMetaDataRes> rs = DWResponseUtil.convertResponse(responseObj, ActionMetaDataRes.class);
        if (!rs.getSuccess()) {
            throw new BusinessException(ErrorCodeEnum.KG_400);
        }
        return rs.getData();
    }

    @Override
    public ResultDto<Void> syncRuleCallBack(String transactionId, String ruleId, boolean syncSuccess, String errorMsg) {
        try {
            String success = syncSuccess ? SyncRuleCallBackReq.SUCCESS : SyncRuleCallBackReq.FAIL;
            SyncRuleCallBackReq req = new SyncRuleCallBackReq(transactionId, ruleId, success, errorMsg);
            Map<String, SyncRuleCallBackReq> param = new HashMap<>();
            param.put("syncResult", req);
            LOGGER.info("侦测调用ThemeMap回调同步规则结果请求参数:{}", req);
            post(ConfigConstant.THEMEMAP_SYNC_RULE_CALLBACK_URL, param, DWServiceContext.getContext().getToken(), JSONObject.class);
            LOGGER.info("侦测调用ThemeMap回调同步规则结果成功");
        } catch (DWRuntimeException e) {
            LOGGER.error("侦测调用ThemeMap回调同步规则结果失败", e);
            return ResultHelper.generateFailResult(e.getErrorCode(), null);
        } catch (Exception e) {
            LOGGER.error("侦测调用ThemeMap回调同步规则结果失败", e);
            return ResultHelper.generateFailResult(ErrorCodeEnum.KG_OTHER_ERR);
        }
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public ExpireRuleRes getExpireRule(ExpireRuleReq expireRuleReq) {
        Map<String, Object> param = new HashMap<>(8);
        param.put(FieldConstant.TENANTID, expireRuleReq.getTenantId());
        param.put(FieldConstant.APPCODE, expireRuleReq.getAppCode());
        param.put(FieldConstant.EVENT_TYPE, expireRuleReq.getEventType());
        param.put(COMPONENT, Arrays.asList(FieldConstant.EXPIRE_COMPONENT));
        LOGGER.info("侦测调用ThemeMap获取过期应用规则请求参数:[{}]", JsonUtil.getJsonString(param));
        JSONObject responseObj = post(ConfigConstant.THEMEMAP_EXPIRE_RULE_URL,
                param, DWServiceContext.getContext().getToken(), JSONObject.class);
        LOGGER.info("侦测调用ThemeMap获取过期应用规则返回结果:[{}]", JsonUtil.getJsonString(responseObj));
        ResultDto<ExpireRuleRes> rs = DWResponseUtil.convertResponse(responseObj, ExpireRuleRes.class);
        if (!rs.getSuccess()) {
            throw new BusinessException(ErrorCodeEnum.KG_400);
        }
        return rs.getData();
    }

    @Override
    protected ErrorCodeEnum getErrorCodeByHttpStatus(String url, HttpStatus httpStatus) {
        if (null == httpStatus) {
            return ErrorCodeEnum.KG_OTHER_ERR;
        }
        switch (httpStatus) {
            case INTERNAL_SERVER_ERROR :
                return ErrorCodeEnum.KG_500;
            case BAD_GATEWAY :
                return ErrorCodeEnum.KG_502;
            case SERVICE_UNAVAILABLE :
                return ErrorCodeEnum.KG_503;
            case GATEWAY_TIMEOUT :
                return ErrorCodeEnum.KG_504;
        }
        return ErrorCodeEnum.KG_OTHER_ERR;
    }
}
