package com.digiwin.athena.cdme.service.facade.auth.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.service.client.IIamClient;
import com.digiwin.athena.cdme.service.client.IThemeMapClient;
import com.digiwin.athena.cdme.service.facade.auth.IContextFacadeService;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/20 10:37
 */
@Service("cdmeContextFacadeService")
public class ContextFacadeService implements IContextFacadeService {

    private final IThemeMapClient themeMapClient;

    private final IIamClient iamClient;

    public ContextFacadeService(IThemeMapClient themeMapClient, IIamClient iamClient) {
        this.themeMapClient = themeMapClient;
        this.iamClient = iamClient;
    }


    @Override
    public void constructContext(String tenantId) throws Exception {
        addRouterKeyToContext(tenantId);
        addTenantSidAndTokenToContext(tenantId);
    }

    @Override
    public void constructIntegrationContext(String tenantId) throws Exception {
        addRouterKeyToContext(tenantId);
        addIntegrationToContext(tenantId);
    }

    private void addIntegrationToContext(String tenantId) {
        JSONObject integration = iamClient.getIntegration(tenantId);
        DWServiceContext.getContext().setToken(integration.getString(FieldConstant.USER_TOKEN));
    }

    /**
     * 将routerKey添加到上下文中
     *
     * @param tenantId
     */
    private void addRouterKeyToContext(String tenantId) {
        DWServiceContext.getContext().getRequestHeader().put(FieldConstant.ROUTER_KEY, tenantId);
    }

    /**
     * 将tenantSid和token添加到上下文中
     *
     * @param tenantId
     * @throws Exception
     */
    private void addTenantSidAndTokenToContext(String tenantId) throws Exception {
        JSONObject iamResult = getIamResult(tenantId);
        Map<String, Object> profile = new HashMap<>(1);
        profile.put(FieldConstant.TENANTSID, iamResult.getLong(FieldConstant.TENANTSID));
        /** profile在服务中只有多租户信息之类的会用到 */
        DWServiceContext.getContext().setProfile(profile);
        /** 不通过平台，跨容器间的调用需要手动塞token，否则会出现不允许匿名调用异常 */
        DWServiceContext.getContext().setToken(iamResult.getString(FieldConstant.USER_TOKEN));
    }

    /**
     * 调用iam接口获取tenantSid和token
     *
     * @param tenantId
     * @return
     * @throws Exception
     */
    private JSONObject getIamResult(String tenantId) throws Exception {
        String secretKey = themeMapClient.getTenantSecretKey(tenantId);
        JSONObject iamResult = iamClient.getLoginResponse(secretKey);
        if (iamResult.isEmpty()) {
            throw new BusinessException(ErrorCodeEnum.IAM_GET_TOKEN_EMPTY);
        }
        return iamResult;
    }
}
