package com.digiwin.athena.cdme.service.facade.ops.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.config.CdmeApplicationProp;
import com.digiwin.athena.cdme.core.constant.MqttConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.handler.MqttClientSingle;
import com.digiwin.athena.cdme.core.util.*;
import com.digiwin.athena.cdme.pojo.dto.*;
import com.digiwin.athena.cdme.provider.MqttOpsService;
import com.digiwin.athena.cdme.repository.model.*;
import com.digiwin.athena.cdme.service.facade.ops.IRuleTriggerOperatorFacadeService;
import com.digiwin.athena.cdme.service.srp.cache.ICacheService;
import com.digiwin.athena.cdme.service.srp.db.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.paho.client.mqttv3.IMqttAsyncClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @description: 规则变更操作
 * @author: liunansheng
 * @date: 2021/12/3 10:56
 */
@Service("cdmeRuleTriggerOperatorFacadeService")
public class RuleTriggerOperatorFacadeService implements IRuleTriggerOperatorFacadeService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RuleTriggerOperatorFacadeService.class);


    private final IMonitorRuleService ruleService;

    private final IMonitorRuleCdcService ruleCdcService;

    private final IMonitorTriggerService triggerService;

    private final IMonitorTriggerInsService triggerInsService;

    private final IMqttServerConfigService mqttServerConfigService;

    private final IMonitorRecordService recordService;

    private CdmeApplicationProp applicationProp;

    private final ICacheService redisService;

    @Autowired
    @Qualifier("cdmeMqttOpsService")
    private MqttOpsService mqttOpsService;




    public RuleTriggerOperatorFacadeService(IMonitorRuleService ruleService, IMonitorTriggerService triggerService,
                                            IMonitorRuleCdcService ruleCdcService,
                                            IMonitorTriggerInsService triggerInsService, IMqttServerConfigService mqttServerConfigService,
                                            CdmeApplicationProp applicationProp, IMonitorRecordService recordService,
                                            ICacheService redisService) {
        this.ruleService = ruleService;
        this.ruleCdcService = ruleCdcService;
        this.triggerService = triggerService;
        this.triggerInsService = triggerInsService;
        this.mqttServerConfigService = mqttServerConfigService;
        this.applicationProp = applicationProp;
        this.recordService = recordService;
        this.redisService = redisService;
    }

    @Override
    public MonitorRuleDto queryRuleDetailByRuleId(String ruleId, String tenantId, EocDto eocDto) {
        MonitorRuleDto ruleDto = new MonitorRuleDto();
        MonitorRuleModel ruleModel = ruleService.getByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto);
        if (null == ruleModel) {
            return ruleDto;
        }
        ruleDto.setRuleModel(ruleModel);
        if (StringUtil.isNotBlank(ruleModel.getTriggerId())) {
            MonitorTriggerModel triggerModel = triggerService.getByTriggerId(ruleModel.getTriggerId());
            ruleDto.setTriggerModel(triggerModel);
        }
        return ruleDto;
    }

    @Override
    public MonitorCdcRuleDto queryCdcRuleDetailByRuleId(String ruleId, String tenantId, EocDto eocDto) {
        MonitorCdcRuleDto ruleDto = new MonitorCdcRuleDto();
        MonitorRuleCdcModel ruleModel = ruleCdcService.getByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto);
        if (null == ruleModel) {
            return ruleDto;
        }
        ruleDto.setRuleModel(ruleModel);
        return ruleDto;
    }
    @Override
    public MonitorCdcRuleDto getByRuleIdAndTenantIdAndEocUnStatus(String ruleId, String tenantId, EocDto eocDto) {
        MonitorCdcRuleDto ruleDto = new MonitorCdcRuleDto();
        MonitorRuleCdcModel ruleModel = ruleCdcService.getByRuleIdAndTenantIdAndEocUnStatus(ruleId, tenantId, eocDto);
        if (null == ruleModel) {
            return ruleDto;
        }
        ruleDto.setRuleModel(ruleModel);
        return ruleDto;
    }

    @Override
    public MonitorCdcRuleDto queryCdcRuleDetailByRuleIdAndTenantId(String ruleId, String tenantId, EocDto eocDto) {
        MonitorCdcRuleDto ruleDto = new MonitorCdcRuleDto();
        MonitorRuleCdcModel ruleModel = ruleCdcService.getByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto);
        if (null == ruleModel) {
            return ruleDto;
        }
        ruleDto.setRuleModel(ruleModel);
        return ruleDto;
    }

    @Override
    public MonitorCdcRuleDto queryRuleDetailByRuleIdIgnoreIsEnable(String ruleId, String tenantId, EocDto eocDto) {
        MonitorCdcRuleDto ruleDto = new MonitorCdcRuleDto();
        MonitorRuleCdcModel ruleModel = ruleCdcService.queryCdcRuleDetailByRuleIdAndTenantId(ruleId, tenantId, eocDto);
        if (null == ruleModel) {
            return ruleDto;
        }
        ruleDto.setRuleModel(ruleModel);

        return ruleDto;
    }

    @Override
    @Transactional(value = "dw-transactionManager")
    public ResultDto<Void> handleBacktrackDate(String ruleId, String tenantId, EocDto eocDto, String backtrackDate) {
        /** 入参非空校验 */
        if (StringUtil.isBlank(ruleId) || StringUtil.isBlank(tenantId) || StringUtil.isBlank(backtrackDate)) {
            return ResultHelper.generateFailResult(ErrorCodeEnum.PARAM_EMPTY_ERR);
        }
        /**查询规则详情*/
        MonitorRuleDto ruleDto = this.queryRuleDetailByRuleId(ruleId, tenantId, eocDto);
        if (null == ruleDto.getRuleModel()) {
            return ResultHelper.generateFailResult(ErrorCodeEnum.RULE_NOT_EXISTS);
        }
        if (null == ruleDto.getTriggerModel()) {
            return ResultHelper.generateFailResult(ErrorCodeEnum.TRIGGER_NOT_EXISTS);
        }
        LocalDateTime backTrackDateTime = LocalTimeUtil.parseTime(backtrackDate);
        /**回溯时间只设置存放在trigger表，侦测时取trigger和ins表中最大的时间*/
        MonitorTriggerModel triggerModel = ruleDto.getTriggerModel();
        MonitorTriggerModel triggerUpdateBean = new MonitorTriggerModel();
        triggerUpdateBean.setId(triggerModel.getId());
        triggerUpdateBean.setKid(triggerModel.getKid());
        triggerUpdateBean.setLastMonitorTime(backTrackDateTime);
        triggerUpdateBean.setUpdateTime(LocalDateTime.now());
        triggerService.edit(triggerUpdateBean);
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public ResultDto<Void> saveCdcRule(MonitorRuleDto monitorRuleDto, EocDto eocDto) {
        MonitorRuleModel ruleModel = monitorRuleDto.getRuleModel();
        MonitorRuleCdcModel ruleCdcModel = new MonitorRuleCdcModel();
        JSONObject monitorRule = JsonUtil.getObject(ruleModel.getMonitorRule());
        ruleCdcModel.setCreateTime(LocalDateTime.now());
        ruleCdcModel.setStartTime(LocalTimeUtil.parseTime(monitorRule.getString("start_time")));
        ruleCdcModel.setActionId(ruleModel.getActionId());
        ruleCdcModel.setActionParams(monitorRule.getJSONArray("action_params").toJSONString());
        ruleCdcModel.setActionType(ruleModel.getActionType());
        ruleCdcModel.setChangeType(monitorRule.getString("change_type"));
        ruleCdcModel.setEocCompanyId(StringUtil.emptyToNull(eocDto.getEocCompanyId()));
        ruleCdcModel.setEocSiteId(StringUtil.emptyToNull(eocDto.getEocSiteId()));
        ruleCdcModel.setProductName(ruleModel.getProductName());
        ruleCdcModel.setRuleId(ruleModel.getRuleId());
        ruleCdcModel.setTenantId(ruleModel.getTenantId());
        ruleCdcModel.setTenantSid(String.valueOf(DWServiceContext.getContext().getProfile().get(FieldConstant.TENANTSID)));
        ruleCdcModel.setCategory(ruleModel.getCategory());

        if(MqttConstant.CATEGORY.equals(ruleModel.getCategory())){
            ruleCdcModel.setTableName(monitorRule.getString(FieldConstant.MQTT_TOPIC));
            ruleCdcModel.setDbName(monitorRule.getString(FieldConstant.MQTT_SOURCE));
            if(CollectionUtils.isNotEmpty(monitorRule.getJSONArray(FieldConstant.CONVERSION_FILTER_CONDITION))){
                ruleCdcModel.setConversionFilterCondition(monitorRule.getJSONArray(FieldConstant.CONVERSION_FILTER_CONDITION).toJSONString());
            }
            if(Objects.isNull(ruleCdcModel.getStartTime())){
                ruleCdcModel.setStartTime(LocalTimeUtil.now());
            }
            if(Objects.isNull(ruleCdcModel.getChangeType())){
                ruleCdcModel.setChangeType(FieldConstant.CHANGE_TYPE_CREATE);
            }
            if(!Objects.isNull(monitorRule.getJSONArray("filter_condition"))){
                ruleCdcModel.setFilterCondition(monitorRule.getJSONArray("filter_condition").toJSONString());
            }
        } else {
            ruleCdcModel.setTableName(monitorRule.getString("table_name"));
            ruleCdcModel.setChangeFields(monitorRule.getJSONArray("change_fields").toJSONString());
            ruleCdcModel.setFilterCondition(monitorRule.getJSONArray("filter_condition").toJSONString());
        }
        ruleCdcModel.setValid(StringUtil.isEmpty(monitorRule.getString("valid")) ? FieldValConstant.RULE_ENABLE : monitorRule.getString("valid"));
        boolean result = ruleCdcService.save(ruleCdcModel);
        if(result){
            String cdcRuleKey = MonitorHelper.buildCdcRuleKey(ruleCdcModel.getTenantId(), null == ruleCdcModel.getDbName() ? StringUtils.EMPTY : ruleCdcModel.getDbName(), ruleCdcModel.getTableName(), ruleCdcModel.getChangeType());
            String subKey = MonitorHelper.buildCdcRuleSubKey(ruleCdcModel);
            redisService.hPut(cdcRuleKey, subKey, JsonUtil.getJsonString(ruleCdcModel));

            if(MqttConstant.CATEGORY.equals(ruleModel.getCategory())){
                try {
                    IMqttAsyncClient mqttAsyncClient = MqttClientSingle.getInstance().pull(ruleCdcModel.getDbName());
                    MqttServerConfigModel mqttServerConfigModel = MqttClientSingle.getInstance().pullConfig(ruleCdcModel.getDbName());
                    if(!Objects.isNull(mqttAsyncClient)){
                        mqttAsyncClient.subscribe(MqttUtil.getTopicByDbname(ruleCdcModel.getTenantSid(), ruleCdcModel.getTableName(),ruleCdcModel.getDbName(),mqttServerConfigModel.getZone()), mqttServerConfigModel.getOps());
                    }else{
                        mqttOpsService.initMqttByBusinessSource(ruleCdcModel.getDbName());
                        IMqttAsyncClient mqttAsyncClient1 = MqttClientSingle.getInstance().pull(ruleCdcModel.getDbName());
                        MqttServerConfigModel mqttServerConfigModel1 = MqttClientSingle.getInstance().pullConfig(ruleCdcModel.getDbName());
                        if(!Objects.isNull(mqttAsyncClient1)) {
                            mqttAsyncClient1.subscribe(MqttUtil.getTopicByDbname(ruleCdcModel.getTenantSid(), ruleCdcModel.getTableName(), ruleCdcModel.getDbName(), mqttServerConfigModel1.getZone()), mqttServerConfigModel1.getOps());
                         }
                        }
                } catch (Exception e) {
                    LOGGER.error("创建侦测规则时订阅Topic失败，RuleId=[{}]，TenantId=[{}]，Topic=[{}]，异常=[{}]", ruleCdcModel.getRuleId(), ruleCdcModel.getTenantId(), ruleCdcModel.getTableName(), e);
                }
            }
        }
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public ResultDto<Void> modifyRule(MonitorRuleDto monitorRuleDto) {
        /** 控制表的插入 删除 更新顺序，放置并发情况下数据库行锁竞争产生死锁 */
        if (!FieldValConstant.CATEGORY_REPORT.equals(monitorRuleDto.getRuleModel().getCategory())
                && monitorRuleDto.getTriggerModel() != null) {
            triggerService.edit(monitorRuleDto.getTriggerModel());
        }
        MonitorRuleModel ruleModel = monitorRuleDto.getRuleModel();
        if (ruleModel.getPageSize() == null || ruleModel.getPageSize() < 1
                || ruleModel.getPageSize() > applicationProp.getMonitorPageSize()) {
            //不满足限制，则不更改pageSize
            ruleModel.setPageSize(null);
        }
        ruleService.edit(ruleModel);
        return ResultHelper.generateSuccessResult(null);
    }

    /**
     * 修改规则
     *
     * @param monitorRuleDto
     * @return
     */
    @Override
    public ResultDto<Void> modifyCdcRule(MonitorCdcRuleDto monitorRuleDto) {
        boolean result = ruleCdcService.edit(monitorRuleDto.getRuleModel());
        if (result) {
            MonitorRuleCdcModel ruleCdcModel = monitorRuleDto.getRuleModel();
            List<String> dbNames = getDbNames();
            String cdcRuleKey = dbNames.contains(ruleCdcModel.getDbName()) ? MonitorHelper.buildCdcRuleKey(ruleCdcModel.getTenantId(), ruleCdcModel.getDbName(), ruleCdcModel.getTableName(), ruleCdcModel.getChangeType()) : MonitorHelper.buildCdcRuleKey(ruleCdcModel.getTenantId(), StringUtils.EMPTY, ruleCdcModel.getTableName(), ruleCdcModel.getChangeType());
            String subKey = MonitorHelper.buildCdcRuleSubKey(ruleCdcModel);
            redisService.hPut(cdcRuleKey, subKey, JsonUtil.getJsonString(ruleCdcModel));
        }

        //deleteCdcCache(monitorRuleDto);
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public List<MonitorRuleModel> queryRuleByRuleIdAndTenantId(String ruleId, String tenantId) {
        return ruleService.listByRuleIdAndTenantId(ruleId, tenantId);
    }

    @Override
    public ResultDto<Void> deleteRule(MonitorRuleDto ruleDto, EocDto eocDto) {
        ruleService.deleteByRuleIdAndTenantIdAndEoc(ruleDto.getRuleModel().getRuleId(), ruleDto.getRuleModel().getTenantId(), eocDto);
        return ResultHelper.generateSuccessResult(null);
    }

    /**
     * 删除rule
     *
     * @param ruleDto
     * @param eocDto
     * @return
     */
    @Override
    public ResultDto<Void> deleteCdcRule(MonitorCdcRuleDto ruleDto, EocDto eocDto) {
        boolean result = ruleCdcService.deleteByRuleIdAndTenantIdAndEoc(ruleDto.getRuleModel().getRuleId(), ruleDto.getRuleModel().getTenantId(), eocDto);
        if (result) {
            MonitorRuleCdcModel ruleCdcModel = ruleDto.getRuleModel();
            List<String> dbNames = getDbNames();
            String cdcRuleKey = dbNames.contains(ruleCdcModel.getDbName()) ? MonitorHelper.buildCdcRuleKey(ruleCdcModel.getTenantId(), ruleCdcModel.getDbName(), ruleCdcModel.getTableName(), ruleCdcModel.getChangeType()) : MonitorHelper.buildCdcRuleKey(ruleCdcModel.getTenantId(), StringUtils.EMPTY, ruleCdcModel.getTableName(), ruleCdcModel.getChangeType());
            String subKey = MonitorHelper.buildCdcRuleSubKey(ruleCdcModel);
            redisService.hDelete(cdcRuleKey, subKey);

            if(MqttConstant.CATEGORY.equals(ruleCdcModel.getCategory())){
                List<MonitorRuleCdcModel> cdcRules = ruleCdcService.getByTenantIdAndTableAndChangeType(ruleCdcModel.getTenantId(), ruleCdcModel.getTableName(), FieldConstant.CHANGE_TYPE_CREATE);
                if(CollectionUtils.isEmpty(cdcRules)){
                    try {
                        IMqttAsyncClient mqttAsyncClient = MqttClientSingle.getInstance().pull(ruleCdcModel.getDbName());
                        MqttServerConfigModel mqttServerConfigModel = MqttClientSingle.getInstance().pullConfig(ruleCdcModel.getDbName());

                        if(!Objects.isNull(mqttAsyncClient)) {
                            mqttAsyncClient.unsubscribe(MqttUtil.getTopicByDbname(ruleCdcModel.getTenantSid(), ruleCdcModel.getTableName(), ruleCdcModel.getDbName(),mqttServerConfigModel.getZone()));
                          }
                        } catch (Exception e) {
                        LOGGER.error("删除侦测规则时取消订阅Topic失败，RuleId=[{}]，TenantId=[{}]，Topic=[{}]，异常=[{}]", ruleCdcModel.getRuleId(), ruleCdcModel.getTenantId(), ruleCdcModel.getTableName(), e);
                    }
                }
            }
        }
        return ResultHelper.generateSuccessResult(null);
    }

    private List<String> getDbNames() {
        return mqttServerConfigService.getbusinessSourcesList();
    }

    @Override
    public ResultDto<Void> saveTriggerIns(MonitorTriggerInsModel insModel) {
        triggerInsService.save(insModel);
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public List<MonitorTriggerInsModel> queryTriggerInsByTriggerId(String triggerId) {
        return triggerInsService.listByTriggerId(triggerId);
    }

    @Override
    public List<RuleOpsDto> listRuleDetailByRuleIdAndEoc(String ruleId, String tenantId, EocDto eocDto) {
        List<MonitorRuleModel> ruleModelList = ruleService.listByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto);
        if (CollectionUtil.isEmpty(ruleModelList)) {
            return null;
        }
        List<RuleOpsDto> ruleOpsDtoList = new ArrayList<>();
        for (MonitorRuleModel ruleModel : ruleModelList) {
            MonitorTriggerModel triggerModel = null;
            if (!FieldValConstant.CATEGORY_REPORT.equals(ruleModel.getCategory())) {
                triggerModel = triggerService.getByTriggerId(ruleModel.getTriggerId());
            }
            if (null == triggerModel) {
                ruleOpsDtoList.add(buildRuleOpsDto(ruleModel, null, null));
                continue;
            }
            List<MonitorTriggerInsModel> insList = triggerInsService.listByTriggerId(triggerModel.getId());
            if (CollectionUtil.isEmpty(insList)) {
                ruleOpsDtoList.add(buildRuleOpsDto(ruleModel, triggerModel, null));
                continue;
            }
            for (MonitorTriggerInsModel ins : insList) {
                ruleOpsDtoList.add(buildRuleOpsDto(ruleModel, triggerModel, ins));
            }
        }
        return ruleOpsDtoList;
    }

    private RuleOpsDto buildRuleOpsDto(MonitorRuleModel rule, MonitorTriggerModel trigger, MonitorTriggerInsModel ins) {
        RuleOpsDto ruleOpsDto = new RuleOpsDto(rule, trigger, ins);
        fillMonitorRecordInfo(ruleOpsDto);
        return ruleOpsDto;
    }

    private void fillMonitorRecordInfo(RuleOpsDto ruleOpsDto) {
        MonitorRuleModel rule = ruleOpsDto.getRuleModel();
        String eocCompanyId = rule.getEocCompanyId();
        String eocSiteId = rule.getEocSiteId();
        if (null != ruleOpsDto.getIns()) {
            eocCompanyId = ruleOpsDto.getIns().getEocCompanyId();
            eocSiteId = ruleOpsDto.getIns().getEocSiteId();
        }
        MonitorRecordModel recordModel = recordService.getLastByRule(rule.getRuleId(), rule.getTenantId(), eocCompanyId, eocSiteId);
        ruleOpsDto.setRecordModel(recordModel);
    }

}
