package com.digiwin.athena.cdme.mq.consumer.rule;

import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.dto.SyncRuleParamDto;
import com.digiwin.athena.cdme.repository.model.MonitorOptRecordModel;
import com.digiwin.athena.cdme.service.facade.auth.IContextFacadeService;
import com.digiwin.athena.cdme.service.facade.rulesync.RuleSyncFacadeProxy;
import com.digiwin.athena.cdme.service.srp.db.IMonitorOptRecordService;
import com.rabbitmq.client.Channel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.Exchange;
import org.springframework.amqp.rabbit.annotation.Queue;
import org.springframework.amqp.rabbit.annotation.QueueBinding;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;

/**
 * @description: themeMap rabbitmq规则消息监听
 * @author: dongwh
 * @date: 2020/6/5 16:37
 */
@Component("cdmeMonitorRuleSynListener")
public class MonitorRuleSynListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(MonitorRuleSynListener.class);

    @Autowired
    private RuleSyncFacadeProxy ruleSyncFacadeProxy;

    @Autowired
    private RuleSyncCallBack ruleSyncCallBack;

    @Autowired
    private IMonitorOptRecordService optRecordService;

    @Autowired
    private IContextFacadeService contextService;


    @RabbitListener(bindings = {@QueueBinding(
            value = @Queue(value = "MonitorRuleConfigQueueCDC",
                    durable = "true"),
            exchange = @Exchange(value = "themeMap", ignoreDeclarationExceptions = "true"),
            key = {"MonitorRuleConfigCDC"}
    )
    }, errorHandler = "cdmeConsumeExceptionHandle")
    public void onMessage(Message message, Channel channel) throws Exception {
        MonitorOptRecordModel monitorOptRecordModel = null;
        /** MQ接收到的侦测规则内容*/
        String queueMsg = new String(message.getBody(), StandardCharsets.UTF_8);
        LOGGER.info("侦测引擎接收队列MonitorRuleConfigQueue的消息为:[{}]", queueMsg);
        /** MQ同步规则类型 */
        SyncRuleParamDto ruleDto = parseMessage(queueMsg);
        try {
            /** 构造上下文 */
            contextService.constructContext(ruleDto.getTenantId());
            //构建optRecordModel
            monitorOptRecordModel = new MonitorOptRecordModel(ruleDto);
            /** 同步侦测规则 */
            synMonitorRule(ruleDto);
            monitorOptRecordModel.setStatus(1);
            /** 确认消息 */
            channel.basicAck(message.getMessageProperties().getDeliveryTag(), true);
            ruleSyncCallBack.successCallBack(ruleDto.getTransactionId(), ruleDto.getRuleId());
        } catch (Throwable e) {
            ruleSyncCallBack.failCallBack(ruleDto.getTransactionId(), ruleDto.getRuleId(), e, message);
            throw e;
        } finally {
            //保存侦测操作记录
            if (monitorOptRecordModel != null) {
                optRecordService.save(monitorOptRecordModel);
            }
        }
    }


    /**
     * 同步mq侦测规则
     *
     * @param ruleDto
     * @throws Exception
     */
    private ResultDto synMonitorRule(SyncRuleParamDto ruleDto) {
        ResultDto synRuleResult = ruleSyncFacadeProxy.syncMonitorRuleHandler(ruleDto);
        /** FIXME 当前使用success判断无法满足场景，所以又使用了message当作判断条件。思考一下后续如何处理*/
        if (MonitorHelper.isResultFail(synRuleResult)) {
            LOGGER.error("侦测引擎同步侦测规则:{}失败！", ruleDto);
            String resultMessage = synRuleResult.getMessage();
            /** note 侦测规则已经存在或没有对应正在侦测的规则信息不会再进行重试*/
            if (FieldConstant.MONITOR_RULE_EXIST.equals(resultMessage) || FieldConstant.MONITOR_RULE_NOT_EXIST.equals(resultMessage)) {
                throw new BusinessException(resultMessage);
            }
        }
        LOGGER.info("侦测引擎同步侦测规则入参:{}", ruleDto);
        return synRuleResult;
    }

    /**
     * 获取mq侦测规则
     *
     * @param tmRule
     * @return
     */
    private SyncRuleParamDto parseMessage(String tmRule) {
        if (StringUtil.isEmpty(tmRule) || CollectionUtil.isEmpty(JsonUtil.getObject(tmRule))) {
            throw new IllegalArgumentException("队列MonitorRuleConfigQueue收到的消息为空，请查看！");
        }
        return JsonUtil.getObject(tmRule, SyncRuleParamDto.class);
    }
}
