package com.digiwin.athena.cdme.service.client.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.esp.sdk.Invoker;
import com.digiwin.athena.esp.sdk.enums.InvokeTypeEnum;
import com.digiwin.athena.esp.sdk.model.RequestModel;
import com.digiwin.athena.esp.sdk.model.ResponseModel;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.config.CdmeEaiProp;
import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.core.util.ServiceResponseUtil;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.service.client.AbstractExecuteClient;
import com.digiwin.athena.cdme.service.client.IEspClient;
import com.digiwin.athena.cdme.service.client.request.EspSdkReq;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/19 16:10
 */
@Service("cdmeEspClient")
public class EspClient extends AbstractExecuteClient implements IEspClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(EspClient.class);

    /**
     * esp返回的运营单元节点
     */
    public static final String OPERATION_MANAGER = "operation_manager";

    private final CdmeEaiProp eaiProp;

    public EspClient(CdmeEaiProp eaiProp) {
        this.eaiProp = eaiProp;
    }

    @Override
    public List<EocDto> getDispatchEoc(String tenantId, String productName) {
        JSONArray dispatchArray = getDispatchData(tenantId, productName);
        if(Objects.isNull(dispatchArray)){
            return Collections.singletonList(new EocDto());
        }

        if (CollectionUtil.isEmpty(dispatchArray)) {
            return Collections.singletonList(new EocDto());
        }
        List<EocDto> eocList = new ArrayList<>();
        /** 遍历获取operation_manager节点信息并转为对象集合 */
        for (int i = 0; i < dispatchArray.size(); i++) {
            JSONObject dispatchEoc = dispatchArray.getJSONObject(i).getJSONObject(OPERATION_MANAGER);
            if (CollectionUtil.isNotEmpty(dispatchEoc)) {
                eocList.add(new EocDto(dispatchEoc.getString(FieldConstant.EOC_COMPANY_ID),
                        dispatchEoc.getString(FieldConstant.EOC_SITE_ID), dispatchEoc.getString(FieldConstant.EOC_REGION_ID)));
            } else {
                eocList.add(new EocDto());
            }
        }
        return eocList;
    }

    /**
     * 调用esp dispatch,
     *
     * @param tenantId
     * @param prodName
     * @return
     */
    private JSONArray getDispatchData(String tenantId, String prodName) {
        Map<String, Object> dispatchParam = new HashMap<>(4);
        dispatchParam.put(FieldConstant.TENANT_ID, tenantId);
        dispatchParam.put(FieldConstant.PROD_NAME, prodName);
        LOGGER.info("侦测调用ESP获取Dispatch请求参数:[{}]", JsonUtil.getJsonString(dispatchParam));
        JSONObject jsonObject = post(ConfigConstant.ESP_DISPATCH_URL,
                dispatchParam, DWServiceContext.getContext().getToken(), JSONObject.class);
        LOGGER.info("侦测调用ESP获取Dispatch返回结果:[{}]", JsonUtil.getJsonString(jsonObject));
        if (CollectionUtil.isEmpty(jsonObject)) {
            throw new BusinessException(ErrorCodeEnum.GET_DISPATCH_400);
        }
        JSONObject dataObj = jsonObject.getJSONObject(FieldConstant.DATA);
        if (CollectionUtil.isEmpty(dataObj)) {
            return null;
        }
        return dataObj.getJSONArray(FieldConstant.DISPATCH);
    }

    @Override
    protected ErrorCodeEnum getErrorCodeByHttpStatus(String url, HttpStatus httpStatus) {
        if (null == httpStatus) {
            return ErrorCodeEnum.GET_DISPATCH_OTHER_ERR;
        }
        switch (httpStatus) {
            case INTERNAL_SERVER_ERROR :
                return ErrorCodeEnum.GET_DISPATCH_500;
            case BAD_GATEWAY :
                return ErrorCodeEnum.GET_DISPATCH_502;
            case SERVICE_UNAVAILABLE :
                return ErrorCodeEnum.GET_DISPATCH_503;
            case GATEWAY_TIMEOUT :
                return ErrorCodeEnum.GET_DISPATCH_504;
        }
        return ErrorCodeEnum.GET_DISPATCH_OTHER_ERR;
    }

    @Override
    public ResponseModel invokeRestSync(EspSdkReq reqParam) throws Exception {
        Map<String, String> header = new HashMap<>(4);
        header.put(FieldConstant.USER_TOKEN, DWServiceContext.getContext().getToken());
        /** NOTE 不停机版更需要调用方在header中添加 routerKey:tenantId。組件間調用跟穩敏api調用有routerkey，中间件都不用传
         * 规范要求下发请求在head中区分routerKey大小写，确保K8S能够识别和路由。
         * */
        header.put(FieldConstant.ROUTER_KEY, (String) DWServiceContext.getContext().getRequestHeader().get(FieldConstant.ROUTER_KEY));
        header.put(FieldConstant.DW_APP_TOKEN_HEADER_NAME, ConfigConstant.MULTIPLEAPPENV?ConfigConstant.MERGED_IAM_APP_TOKEN:ConfigConstant.IAM_APP_TOKEN);
        //DWServiceChainUtils.beforeInvokeOutterAPI(header::put);
        RequestModel requestModel = convertToRequestModel(reqParam, header);
        requestModel.setLanguage((String) DWServiceContext.getContext().getRequestHeader().get(FieldConstant.LOCALE));
        return Invoker.invokeRestSync(requestModel);
    }

    private RequestModel convertToRequestModel(EspSdkReq reqParam, Map<String, String> header) {
        RequestModel requestModel = new RequestModel();
        requestModel.setInvokeType(InvokeTypeEnum.SYNC);
        requestModel.setHostProd(eaiProp.getEaiHostProd());
        requestModel.setHostVer(reqParam.getHostVer());
        requestModel.setHostId(eaiProp.getEaiHostId());
        requestModel.setHostAcct(reqParam.getHostAcct());
        requestModel.setTenantId(reqParam.getTenantId());
        requestModel.setServiceProd(reqParam.getServiceProd());
        requestModel.setServiceName(reqParam.getServiceName());
        requestModel.setDatakeyMap(reqParam.getDatakey());
        requestModel.setHeaderMap(header);
        requestModel.setBodyJsonString(reqParam.getBodyJsonString());
        requestModel.setEocMap(reqParam.getEocDto().toUnderLineMap());
        return requestModel;
    }

    /**
     * 是否需要在头部加入租户、app授权信息
     *
     * @return
     */
    @Override
    protected boolean isNeedAuthInfo() {
        return true;
    }

    /**
     * 同步调用esp
     *
     * @param espRequest
     * @return
     */
    private ResultDto<JSONObject> callEspSync(EspSdkReq espRequest) {
        LOGGER.info("[{}-{}]侦测调用ESP请求入参:[{}]", espRequest.getDatakey().get(FieldConstant.RULEID), espRequest.getTenantId(), espRequest);
        /*调用esp返回结果*/
        ResponseModel responseModel;
        try {
            responseModel = this.invokeRestSync(espRequest);
        } catch (Exception e) {
            LOGGER.error("侦测调用ESP接口异常", e);
            //todo 等esp提供不同的异常状态，暂按500错误处理
            throw new BusinessException(ErrorCodeEnum.DATA_CHANGE_GET_500);
        }
        String respContent = responseModel.getBodyJsonString();
        String reqId = responseModel.getReqid();
        LOGGER.info("[{}-{}]侦测调用ESP返回结果:[{}]", espRequest.getDatakey().get(FieldConstant.RULEID), espRequest.getTenantId(), respContent);
        /** 获取esp返回对象*/
        JSONObject dataObj = getEspRespData(respContent);
        /** 获取ESP返回状态*/
        boolean status = isSuccessResp(dataObj);
        if (!status) {
            LOGGER.error("ESP 返回状态为 [false],调用失败！失败原因：{}", getEspRespDesc(dataObj, respContent, reqId));
            return ResultHelper.generateFailResult(ErrorCodeEnum.DATA_CHANGE_GET_500, new JSONObject());
        }
        JSONObject espRespParameter = dataObj.getJSONObject(FieldConstant.ESP_PARAMETER);
        if (CollectionUtil.isEmpty(espRespParameter)) {
            LOGGER.warn("ESP调用成功,但返回parameter节点数据为空！");
            return ResultHelper.generateFailResult(ErrorCodeEnum.DATA_CHANGE_GET_RES_EMPTY, new JSONObject());
        }
        return ResultHelper.generateResult(true, "ESP调用成功！", espRespParameter);
    }

    /**
     * 获取response_objects节点信息
     *
     * @param espRequest
     * @return
     */
    public ResultDto<JSONObject> getChangeDataAndMetaInfo(EspSdkReq espRequest) {
        ResultDto<JSONObject> espRespParameter = callEspSync(espRequest);
        if (MonitorHelper.isResultFail(espRespParameter)) {
            return espRespParameter;
        }
        JSONArray respObjects = espRespParameter.getData().getJSONArray(FieldConstant.ESP_RESPONSE_OBJECTS);
        if (CollectionUtil.isEmpty(respObjects)) {
            LOGGER.warn("ESP调用成功,但返回response_objects节点数据为空！");
            return ResultHelper.generateFailResult(ErrorCodeEnum.DATA_CHANGE_GET_RES_EMPTY, new JSONObject());
        }
        return ResultHelper.generateResult(true, "ESP调用成功！", respObjects.getJSONObject(0));
    }

    public ResultDto<JSONArray> getChangeObjects(EspSdkReq espRequest) {
        ResultDto<JSONObject> changeData = getChangeDataAndMetaInfo(espRequest);
        if (MonitorHelper.isResultFail(changeData)) {
            return ResultHelper.generateFailResult(changeData.getCode(), changeData.getMessage(), new JSONArray());
        }
        JSONArray changeObjects = changeData.getData().getJSONArray(FieldConstant.ESP_CHANGE_OBJECTS);
        return ResultHelper.generateResult(true, "ESP调用成功！", changeObjects);
    }


    /**
     * 获取ESP返回状态
     *
     * @param respObj
     * @return
     */
    private boolean isSuccessResp(JSONObject respObj) {
        if (CollectionUtil.isNotEmpty(respObj)) {
            return 0 == respObj.getJSONObject(FieldConstant.ESP_EXECUTION).getIntValue(FieldConstant.ESP_CODE);
        }
        return false;
    }

    /**
     * 获取esp返回对象
     *
     * @param respContent
     * @return
     */
    private JSONObject getEspRespData(String respContent) {
        JSONObject respJsonObject = JsonUtil.getObject(respContent);
        if (CollectionUtil.isNotEmpty(respJsonObject)) {
            JSONObject respObj = respJsonObject.getJSONObject(FieldConstant.ESP_STD_DATA);
            return ServiceResponseUtil.checkObjEmpty(respObj);
        }
        return new JSONObject();
    }

    /**
     * 获取ESP返回结果描述信息
     *
     * @param respObj
     * @return
     */
    private String getEspRespDesc(JSONObject respObj, String respContent, String reqId) {
        if (CollectionUtil.isEmpty(respObj)) {
            return new StringBuilder("ESP返回异常结果为：[").append(respContent).append("]，请检查！")
                    .append("reqId：[").append(reqId).append("]").toString();
        }
        return new StringBuilder(respObj.getJSONObject(FieldConstant.ESP_EXECUTION).getString(FieldConstant.ESP_DESCRIPTION))
                .append("reqId：[").append(reqId).append("]").toString();
    }
}
