package com.digiwin.athena.cdme.core.config;

import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.AcknowledgeMode;
import org.springframework.amqp.core.Address;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.DirectExchange;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.annotation.EnableRabbit;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.SendRetryContextAccessor;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.support.RetryTemplate;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * @description: RabbitMQ 配置类
 * @author: dongwh
 * @date: 2021/6/10 15:06
 */
@Configuration
@EnableRabbit
public class CdmeRabbitMQConfig {
    private static final Logger LOGGER = LoggerFactory.getLogger(CdmeRabbitMQConfig.class);

    @Autowired
    private CdmeRabbitMQProp property;

    @Bean("cdmeRabbitTemplate")
    public RabbitTemplate rabbitTemplate(@Qualifier("cdmeConnectionFactory") ConnectionFactory rabbitConnectionFactory) {
        RabbitTemplate rabbitTemplate = new RabbitTemplate(rabbitConnectionFactory);
        rabbitTemplate.setMessageConverter(new Jackson2JsonMessageConverter());
        rabbitTemplate.setConfirmCallback((correlationData, ack, cause) -> {
            if (!ack) {
                LOGGER.error("消息发送失败,ID为: {},错误信息: {}", correlationData.getId(), cause);
            }
        });
        return rabbitTemplate;
    }

    @Bean("cdmeConnectionFactory")
    public ConnectionFactory connectionFactory() {
        CachingConnectionFactory connectionFactory = new CachingConnectionFactory();
        connectionFactory.setHost(property.getHost());
        connectionFactory.setPort(property.getPort());
        connectionFactory.setUsername(property.getUsername());
        connectionFactory.setPassword(property.getPassword());
        connectionFactory.setVirtualHost(property.getVirtualHost());
        return connectionFactory;
    }

    @Bean("cdmeMmqpAdmin")
    public RabbitAdmin cdmeMmqpAdmin(@Qualifier("cdmeConnectionFactory") ConnectionFactory connectionFactory) {
        return new RabbitAdmin(connectionFactory);
    }

    @Bean("cdmeRabbitListenerContainerFactory")
    public SimpleRabbitListenerContainerFactory cdmeRabbitListenerContainerFactory(@Qualifier("cdmeConnectionFactory") ConnectionFactory connectionFactory) {
        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(connectionFactory);
        factory.setAcknowledgeMode(AcknowledgeMode.MANUAL);
        /** 设置消费者数量 */
        factory.setConcurrentConsumers(property.getSimpleConcurrency());
        /** 设置最大消费者数量 */
        factory.setMaxConcurrentConsumers(property.getSimpleMaxConcurrency());
        /** 单个消费者每次获取消息数量 */
        factory.setPrefetchCount(property.getSimplePrefetch());
        /** 设置消费之后ack返回broker的reply的重试机制 */
        RetryTemplate retryTemplate = new RetryTemplate();
        ExponentialBackOffPolicy backOffPolicy = new ExponentialBackOffPolicy();
        backOffPolicy.setMultiplier(property.getSimpleMultiplier());
        backOffPolicy.setInitialInterval(property.getSimpleInitialInterval());
        backOffPolicy.setMaxInterval(property.getSimpleMaxInterval());
        retryTemplate.setBackOffPolicy(backOffPolicy);
        factory.setRetryTemplate(retryTemplate);
        /** 设置重试机制耗尽以后，如何进行回调处理 */
        factory.setReplyRecoveryCallback(ctx -> {
            Message failed = SendRetryContextAccessor.getMessage(ctx);
            Address replyTo = SendRetryContextAccessor.getAddress(ctx);
            Throwable t = ctx.getLastThrowable();
            StringBuilder message = new StringBuilder("消息为：");
            message.append(new String(failed.getBody(), StandardCharsets.UTF_8));
            message.append(", 响应回给exchangeName：");
            message.append(replyTo.getExchangeName());
            message.append(", routingKey：");
            message.append(replyTo.getRoutingKey());
            message.append(" 的相关操作重试后还是失败！请注意！");
            LOGGER.error(message.toString(), t);
            return null;
        });
        return factory;
    }

    @Bean("cdmeGetDirectExchange")
    public DirectExchange cdmeGetDirectExchange() {
        return new DirectExchange(ConfigConstant.CDC_MQ_EXCHANGE, true, false, null);
    }

    @Bean("cdmeGetQueue")
    public Queue cdmeGetQueue() {
        Map<String, Object> arguments = new HashMap<>(4);
        return new Queue(ConfigConstant.CDC_MQ_QUEUE, true, false, false, arguments);
    }

    @Bean("cdmeBinding")
    public Binding cdmeBinding() {
        return BindingBuilder.bind(cdmeGetQueue()).to(cdmeGetDirectExchange()).with(ConfigConstant.CDC_MQ_ROUTINGKEY);
    }
}
