package com.digiwin.athena.cdme.mq.consumer.rule;

import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.aop.TraceId;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.ExpireRuleDto;
import com.digiwin.athena.cdme.repository.model.MonitorOptRecordModel;
import com.digiwin.athena.cdme.service.facade.auth.IContextFacadeService;
import com.digiwin.athena.cdme.service.facade.rulesync.IExpireRuleFacadeService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorOptRecordService;
import com.rabbitmq.client.Channel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.ExchangeTypes;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.Exchange;
import org.springframework.amqp.rabbit.annotation.Queue;
import org.springframework.amqp.rabbit.annotation.QueueBinding;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;
import java.util.Objects;

/**
 * @author yang.xiao
 * @version V1.0
 * @Description  监听租户应用到期消息
 * @date 2023/11/10 10:15
 * @Copyright 鼎捷软件股份有限公司
 */
@Component("cdmeMonitorRuleExpireListener")
public class MonitorRuleExpireListener {
    public static final Logger LOGGER = LoggerFactory.getLogger(MonitorRuleExpireListener.class);

    @Autowired
    private IContextFacadeService contextService;

    @Autowired
    private IExpireRuleFacadeService expireRuleFacadeService;

    @Autowired
    private IMonitorOptRecordService optRecordService;

    @RabbitListener(bindings = {@QueueBinding(
            value = @Queue(value = "appExpireChangeMonitorCDC",
                    durable = "true"),
            exchange = @Exchange(value = "appExpireChange", ignoreDeclarationExceptions = "true", type = ExchangeTypes.FANOUT)
    )
    },containerFactory = "cdmeConnectionFactory", errorHandler = "cdmeConsumeExceptionHandle")
    @TraceId
    public void consumeMonitorMsg(Message message, Channel channel) throws Exception {
        String queueMsg = new String(message.getBody(), StandardCharsets.UTF_8);
        LOGGER.info("侦测引擎接收队列appExpireChangeMonitorCDC的消息为:[{}]", queueMsg);
        ExpireRuleDto expireRuleDto = parseMessage(queueMsg);
        MonitorOptRecordModel optRecord = new MonitorOptRecordModel(expireRuleDto, 0);
        if (Objects.equals(FieldConstant.EVENT_TYPE_EXPIRE, expireRuleDto.getEventType()) || Objects.equals(FieldConstant.EVENT_TYPE_CLEAN, expireRuleDto.getEventType())) {
            try {
                /** 构造上下文 */
                contextService.constructIntegrationContext(expireRuleDto.getTenantId());
                /** 删除过期应用所属的侦测规则 */
                expireRuleFacadeService.processedExpireRule(expireRuleDto);
                /** 确认消息 */
                channel.basicAck(message.getMessageProperties().getDeliveryTag(), true);
                optRecord.setStatus(1);
            } catch (Throwable e) {
                LOGGER.error("删除过期应用对应的侦测规则失败，入参为:{}！", expireRuleDto);
                throw e;
            } finally {
                optRecordService.save(optRecord);
            }
        }
        channel.basicAck(message.getMessageProperties().getDeliveryTag(), true);
    }

    /**
     * 解析消息内容
     *
     * @param kmMessage
     * @return
     */
    private ExpireRuleDto parseMessage(String kmMessage) {
        if (StringUtil.isEmpty(kmMessage)) {
            throw new IllegalArgumentException("队列appExpireChangeMonitorCDC收到的消息为空，请查看！");
        }
        return JsonUtil.getObject(kmMessage, ExpireRuleDto.class);
    }
}