package com.digiwin.athena.cdme.service.srp.db.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.repository.dao.CdmeMonitorRuleCdcMapper;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import com.digiwin.athena.cdme.service.srp.cache.ICacheService;
import com.digiwin.athena.cdme.service.srp.db.AbstractRepositorService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRuleCdcService;
import com.digiwin.athena.cdme.service.srp.db.IMqttServerConfigService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * @author zhangww
 * @description:
 * @date 2023/3/26 18:18
 */
@Service("cdmeMonitorRuleCdcService")
public class MonitorRuleCdcService extends AbstractRepositorService<CdmeMonitorRuleCdcMapper, MonitorRuleCdcModel> implements IMonitorRuleCdcService {

    private static final Logger LOGGER = LoggerFactory.getLogger(MonitorRuleCdcService.class);

    public MonitorRuleCdcService(CdmeMonitorRuleCdcMapper mapper) {
        super(mapper);
    }

    @Autowired
    private ICacheService cacheService;
    @Autowired
    private IMqttServerConfigService mqttServerConfigService;

    @Override
    public List<MonitorRuleCdcModel> getByTenantIdAndTableAndChangeType(String tenantId, String tableName, String changeType) {
        return mapper.selectList(Wrappers.<MonitorRuleCdcModel>lambdaQuery()
                .eq(MonitorRuleCdcModel::getTenantId, tenantId)
                .eq(MonitorRuleCdcModel::getTableName, tableName)
                .eq(MonitorRuleCdcModel::getChangeType, changeType));
    }

    public List<MonitorRuleCdcModel> getByTenantSidAndTableAndChangeType(String tenantSid, String tableName, String changeType) {
        return mapper.selectList(Wrappers.<MonitorRuleCdcModel>lambdaQuery()
                .eq(MonitorRuleCdcModel::getTenantSid, tenantSid)
                .eq(MonitorRuleCdcModel::getTableName, tableName)
                .eq(MonitorRuleCdcModel::getChangeType, changeType));
    }

    @Override
    public List<MonitorRuleCdcModel> listByRuleIdAndTenantId(String ruleId, String tenantId) {
        return mapper.selectList(queryWrapperByRuleIdAndTenantId(ruleId, tenantId));
    }

    @Override
    public List<MonitorRuleCdcModel> getAllRules() {
        return mapper.selectList(Wrappers.query());
    }

    @Override
    public List<MonitorRuleCdcModel> getByCategory(String category, String dbName) {
        return mapper.selectList(Wrappers.<MonitorRuleCdcModel>lambdaQuery()
                .eq(MonitorRuleCdcModel::getCategory, category)
                .eq(MonitorRuleCdcModel::getDbName, dbName));
    }
    @Override
    public List<MonitorRuleCdcModel> getListByCategory(String category) {
        return mapper.selectList(Wrappers.<MonitorRuleCdcModel>lambdaQuery()
                .eq(MonitorRuleCdcModel::getCategory, category));
    }
    @Override
    public List<MonitorRuleCdcModel> getDbNameListByCategory(String category) {
        QueryWrapper<MonitorRuleCdcModel> queryWrapper=new QueryWrapper<>();
        queryWrapper.select("distinct db_name");
        LambdaQueryWrapper<MonitorRuleCdcModel> distinctDbName =
                queryWrapper.lambda()
                        .eq(MonitorRuleCdcModel::getCategory, category);
        return mapper.selectList(distinctDbName);
    }
    @Override
    public void queryCdcRuleByTenantIdAndTableAndOp(List<MonitorRuleCdcModel> monitorRuleCdcModelList, String tenantId, String db, String tableName, String changeType) {

        String cdcRuleKey = mqttServerConfigService.getbusinessSourcesList().contains(db) ? MonitorHelper.buildCdcRuleKey(tenantId, db, tableName, changeType) : MonitorHelper.buildCdcRuleKey(tenantId, StringUtils.EMPTY, tableName, changeType);
        if (cacheService.hasKey(cdcRuleKey)) {
            LOGGER.info("有缓存，从缓存中获取规则，tenantId={}, tableName={}, changeType={}，cdcRuleKey={}", tenantId, tableName, changeType, cdcRuleKey);
            Map<Object, Object> cdcRuleList = cacheService.hGetAll(cdcRuleKey);
            for(Map.Entry<Object, Object> entry:cdcRuleList.entrySet()){
                MonitorRuleCdcModel monitorRuleCdcModel = JsonUtil.getObject(String.valueOf(entry.getValue()), MonitorRuleCdcModel.class);
                monitorRuleCdcModelList.add(monitorRuleCdcModel);
            }
        } else {
            LOGGER.info("没有缓存，从数据库中获取规则，，tenantId={}, tableName={}, changeType={}", tenantId, tableName, changeType);
            monitorRuleCdcModelList = getByTenantIdAndTableAndChangeType(tenantId, tableName, changeType);
            // 将查询到的侦测规则放入缓存（规则新增修改也要处理）
            if(CollectionUtils.isNotEmpty(monitorRuleCdcModelList)){
                monitorRuleCdcModelList.forEach(rule -> {
                    String subKey = MonitorHelper.buildCdcRuleSubKey(rule);
                    cacheService.hPut(cdcRuleKey, subKey, JsonUtil.getJsonString(rule));
                });
            }
        }
    }

    /**
     * 根据 ruleId tenantId eoc删除规则
     *
     * @param ruleId
     * @param tenantId
     * @param eocDto
     * @return
     */
    @Override
    public boolean deleteByRuleIdAndTenantIdAndEoc(String ruleId, String tenantId, EocDto eocDto) {
        int result = mapper.delete(queryWrapperByRuleIdAndTenantIdAndEoc(ruleId, tenantId, eocDto));
        if (result < 1) {
            LOGGER.error("sd_monitor_rule_cdc表删除失败，删除条件ruleId:{},tenantId:{},eoc:{}", ruleId, tenantId, eocDto);
            return false;
        }
        return true;
    }

    @Override
    public MonitorRuleCdcModel getByRuleIdAndTenantIdAndEoc(String ruleId, String tenantId, EocDto eocDto) {
        String eocCompanyId = eocDto.getEocCompanyId();
        String eocSiteId = eocDto.getEocSiteId();
        LambdaQueryWrapper<MonitorRuleCdcModel> wrapper = queryWrapperByRuleIdAndTenantId(ruleId, tenantId);
        if (StringUtil.isBlank(eocCompanyId)) {
            wrapper.isNull(MonitorRuleCdcModel::getEocCompanyId);
        } else {
            wrapper.eq(MonitorRuleCdcModel::getEocCompanyId, eocCompanyId);
        }
        if (StringUtil.isBlank(eocSiteId)) {
            wrapper.isNull(MonitorRuleCdcModel::getEocSiteId);
        } else {
            wrapper.eq(MonitorRuleCdcModel::getEocSiteId, eocSiteId);
        }
        wrapper.eq(MonitorRuleCdcModel::getValid, FieldValConstant.RULE_ENABLE);
        return mapper.selectOne(wrapper);
    }
    @Override
    public MonitorRuleCdcModel getByRuleIdAndTenantIdAndEocUnStatus(String ruleId, String tenantId, EocDto eocDto){
        String eocCompanyId = eocDto.getEocCompanyId();
        String eocSiteId = eocDto.getEocSiteId();
        LambdaQueryWrapper<MonitorRuleCdcModel> wrapper = queryWrapperByRuleIdAndTenantId(ruleId, tenantId);
        if (StringUtil.isBlank(eocCompanyId)) {
            wrapper.isNull(MonitorRuleCdcModel::getEocCompanyId);
        } else {
            wrapper.eq(MonitorRuleCdcModel::getEocCompanyId, eocCompanyId);
        }
        if (StringUtil.isBlank(eocSiteId)) {
            wrapper.isNull(MonitorRuleCdcModel::getEocSiteId);
        } else {
            wrapper.eq(MonitorRuleCdcModel::getEocSiteId, eocSiteId);
        }
        return mapper.selectOne(wrapper);
    }
    @Override
    public MonitorRuleCdcModel queryCdcRuleDetailByRuleIdAndTenantId(String ruleId, String tenantId, EocDto eocDto) {
        String eocCompanyId = eocDto.getEocCompanyId();
        String eocSiteId = eocDto.getEocSiteId();
        LambdaQueryWrapper<MonitorRuleCdcModel> wrapper = queryWrapperByRuleIdAndTenantId(ruleId, tenantId);
        if (StringUtil.isBlank(eocCompanyId)) {
            wrapper.isNull(MonitorRuleCdcModel::getEocCompanyId);
        } else {
            wrapper.eq(MonitorRuleCdcModel::getEocCompanyId, eocCompanyId);
        }
        if (StringUtil.isBlank(eocSiteId)) {
            wrapper.isNull(MonitorRuleCdcModel::getEocSiteId);
        } else {
            wrapper.eq(MonitorRuleCdcModel::getEocSiteId, eocSiteId);
        }
        return mapper.selectOne(wrapper);
    }

    /**
     * 查询条件
     *
     * @param ruleId
     * @param tenantId
     * @param eocDto
     * @return
     */
    private LambdaQueryWrapper<MonitorRuleCdcModel> queryWrapperByRuleIdAndTenantIdAndEoc(String ruleId, String tenantId, EocDto eocDto) {
        String eocCompanyId = eocDto.getEocCompanyId();
        String eocSiteId = eocDto.getEocSiteId();
        return queryWrapperByRuleIdAndTenantId(ruleId, tenantId)
                .eq(StringUtil.isNotBlank(eocCompanyId), MonitorRuleCdcModel::getEocCompanyId, eocCompanyId)
                .eq(StringUtil.isNotBlank(eocSiteId), MonitorRuleCdcModel::getEocSiteId, eocSiteId);
    }

    /**
     * 查询条件
     *
     * @param ruleId
     * @param tenantId
     * @return
     */
    private LambdaQueryWrapper<MonitorRuleCdcModel> queryWrapperByRuleIdAndTenantId(String ruleId, String tenantId) {
        return Wrappers.<MonitorRuleCdcModel>lambdaQuery()
                .eq(MonitorRuleCdcModel::getRuleId, ruleId)
                .eq(MonitorRuleCdcModel::getTenantId, tenantId);
    }



}