package com.digiwin.athena.cdme.core.util;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.constant.FieldValConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.pojo.dto.TimeRangeDto;
import com.digiwin.athena.cdme.pojo.dto.parse.DynamicParamDto;
import com.digiwin.athena.cdme.pojo.dto.parse.DynamicRuleDto;
import com.digiwin.athena.cdme.pojo.dto.parse.StaticParamDto;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import com.digiwin.athena.cdme.repository.model.MonitorRuleModel;
import com.digiwin.athena.cdme.service.client.request.DataChangeSqlGetParameter;
import com.digiwin.athena.cdme.service.client.request.EspSdkReq;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * @author zhangzhi@digiwin.com
 * @date 2020/9/27
 */
public final class MonitorHelper {

    private MonitorHelper() {
    }

    /**
     * 是否允许返回为空的场景下继续执行后续业务流程
     *
     * @param monitorRule
     * @return
     */
    public static boolean isGoOn(String category, String monitorRule) {
        if (FieldValConstant.CATEGORY_SCHEDULE.equals(category)) {
            return true;
        }
        return "true".equals(JsonUtil.getObject(monitorRule).getString(FieldConstant.IS_CONTINUE));
    }

    public static boolean isPollingIds(String monitorRule) {
        return "true".equals(JsonUtil.getObject(monitorRule).getString(FieldConstant.IS_POLLING_IDS));
    }

    public static boolean isResultFail(ResultDto result) {
        return !result.getSuccess();
    }

    /**
     * 组装调用执行引擎参数
     *
     * @param ruleModel
     * @param eocDto      运营单元
     * @param monitorData 侦测数据
     * @param allParams   侦测规则参数
     * @param buildType   执行引擎类型：monitorAction、action
     * @return
     */
    public static Map<String, Object> buildExecutionParam(MonitorRuleModel ruleModel, EocDto eocDto,
                                                          JSONArray monitorData, Map<String, JSONObject> allParams, String buildType) {
        /** NOTE sysParam参数为了解决执行引擎action阶段也需要类似datetime_s、datetime_e这样的参数而设定，这种参数在执行引擎中$根路径无法匹配 */
        Map<String, Object> sysParam = new HashMap<>(4);
        String actionId = "";
        Object paras = new Object();
        if (FieldValConstant.EXECUTE_TYPE_MONITOR_ACTION.equals(buildType)) {
            actionId = ruleModel.getMonitorActionId();
            /** 动态以及静态参数解析之后需要丢入apiParams中 */
            Map<String, String> parameter = new HashMap<>(8);
            /** 循环所有参数信息,将target是action的数据作为执行入参传递下去 */
            allParams.forEach((key, value) -> {
                boolean isContains = Arrays.asList(value.getString(FieldConstant.TARGET).split(",")).contains(FieldValConstant.TARGET_MONITOR_ACTION);
                if (isContains) {
                    parameter.put(key, value.getString(FieldConstant.VALUE));
                }
            });
            paras = Collections.singletonList(parameter);
        } else if (FieldValConstant.EXECUTE_TYPE_ACTION.equals(buildType)) {
            /** 循环所有参数信息,将target是action的数据作为执行入参传递下去 */
            allParams.forEach((key, value) -> {
                boolean isContains = Arrays.asList(value.getString(FieldConstant.TARGET).split(",")).contains(FieldValConstant.TARGET_ACTION);
                if (isContains) {
                    sysParam.put(key, value.getString(FieldConstant.VALUE));
                }
            });
            actionId = ruleModel.getActionId();
            paras = monitorData;
        }
        /** 调用执行引擎入参*/
        Map<String, Object> executionParams = new HashMap<>(8);
        executionParams.put(FieldConstant.EXECUTION_SYS_PARAM, sysParam);
        executionParams.put(FieldConstant.ACTIONID, actionId);
        executionParams.put(FieldConstant.EXECUTION_PARAMS, paras);
        executionParams.put(FieldConstant.TENANTID, ruleModel.getTenantId());
        executionParams.put(FieldConstant.EXECUTION_EOC_MAP, eocDto.toUnderLineMap());
        return executionParams;
    }

    public static Map<String, Object> buildCdcExecutionParam(MonitorRuleCdcModel monitorRuleCdcModel, EocDto eocDto,
                                                             JSONArray monitorData) {
        /** 调用执行引擎入参*/
        Map<String, Object> executionParams = new HashMap<>(8);
        executionParams.put(FieldConstant.ACTIONID, monitorRuleCdcModel.getActionId());
        executionParams.put(FieldConstant.EXECUTION_PARAMS, monitorData);
        executionParams.put(FieldConstant.TENANTID, monitorRuleCdcModel.getTenantId());
        executionParams.put(FieldConstant.EXECUTION_EOC_MAP, eocDto.toUnderLineMap());
        return executionParams;
    }

    /**
     * 生成redis的key，用于侦测数据去重时使用
     *
     * @param ruleModel: 侦测规则定义
     * @param eocDto:    运用组织信息
     * @return
     */
    public static String buildRedisKey(MonitorRuleModel ruleModel, EocDto eocDto) {
        return buildRedisKey(ruleModel.getTenantId(), ruleModel.getRuleId(),
                eocDto.getEocCompanyId(), eocDto.getEocSiteId(), ruleModel.getProductName());
    }

    /**
     * redissonClient单独增加前缀
     */
    public static String buildRedisKey(String tenantId, String ruleId, EocDto eocDto) {
        return buildRedisKey(tenantId, ruleId,
                eocDto.getEocCompanyId(), eocDto.getEocSiteId(), "");
    }

    /**
     * redis兼容老key
     */
    public static String buildOldRedisKey(String tenantId, String ruleId, EocDto eocDto) {
        return FieldConstant.REDIS_KEY_PREFIX +
                "-" + tenantId + "-" + ruleId +
                "-" + StringUtil.nullToEmpty(eocDto.getEocCompanyId()) +
                "-" + StringUtil.nullToEmpty(eocDto.getEocSiteId()) +
                "-";
    }

    public static String buildRedisKey(String tenantId, String ruleId, String eocCompanyId, String eocSiteId, String productName) {
        return FieldConstant.REDIS_KEY_PREFIX +
                ":" + tenantId +
                ":" + ruleId +
                ":" + StringUtil.nullToEmpty(eocCompanyId) +
                ":" + StringUtil.nullToEmpty(eocSiteId) +
                ":" + productName;
    }

    /**
     * 生成redis的key，用于跨层级更改侦测规则复制缓存时使用
     *
     * @param ruleModel: 侦测规则定义
     * @param eocDto:    运用组织信息
     * @return
     */
    public static String buildCrossLevelRedisKey(MonitorRuleModel ruleModel, EocDto eocDto) {
        return buildRedisKey(ruleModel.getTenantId(), ruleModel.getRuleId(),
                eocDto.getEocCompanyId(), eocDto.getEocSiteId(), ruleModel.getProductName());
    }

    /**
     * 生成分布式锁的key
     *
     * @param ruleModel
     * @return
     */
    public static String buildLockKey(MonitorRuleModel ruleModel) {
        return ruleModel.getTenantId() +
                ":" + ruleModel.getRuleId() +
                ":" + StringUtil.nullToEmpty(ruleModel.getEocCompanyId()) +
                ":" + StringUtil.nullToEmpty(ruleModel.getEocSiteId()) +
                ":" + ruleModel.getProductName();
    }

    /**
     * 兼容老key
     */
    public static String buildOldLockKey(MonitorRuleModel ruleModel) {
        return ruleModel.getTenantId() +
                "-" + ruleModel.getRuleId() +
                "-" + StringUtil.nullToEmpty(ruleModel.getEocCompanyId()) +
                "-" + StringUtil.nullToEmpty(ruleModel.getEocSiteId()) +
                "-" + ruleModel.getProductName();
    }

    /**
     * 生成频次限制的key
     *
     * @param ruleModel
     * @return
     */
    public static String buildIntervelLimitKey(MonitorRuleModel ruleModel) {
        return FieldConstant.INTERVAL_KEY_PREFIX +
                ":" + ruleModel.getTenantId() +
                ":" + ruleModel.getRuleId() +
                ":" + StringUtil.nullToEmpty(ruleModel.getEocCompanyId()) +
                ":" + StringUtil.nullToEmpty(ruleModel.getEocSiteId()) +
                ":" + ruleModel.getProductName();
    }

    public static String buildCdcRuleKey(String tenantId, String db, String tableName, String changeType) {
        return FieldConstant.REDIS_CDC_KEY_PREFIX +
                ":" + tenantId +
                ":" + db +
                ":" + tableName +
                ":" + changeType;
    }

    public static String buildCdcRuleSubKey(MonitorRuleCdcModel ruleCdcModel) {
        return ruleCdcModel.getRuleId() +
                ":" + StringUtil.nullToEmpty(ruleCdcModel.getEocCompanyId()) +
                ":" + StringUtil.nullToEmpty(ruleCdcModel.getEocSiteId());
    }

    public static String buildCdcTopicRouteKey(String topicName, String groupName) {
        return FieldConstant.REDIS_CDC_TOPIC_ROUTE_KEY_PREFIX +
                ":" + topicName +
                ":" + groupName;
    }

    public static String buildCdcTsid(String tenantsid) {
        return FieldConstant.REDIS_CDC_TENANTSID_PREFIX +
                ":" + tenantsid;
    }

    /**
     * 获取下划线格式类型的年月日期
     *
     * @param dateTime
     * @return
     */
    public static String getDate(String dateTime) {
        return dateTime.substring(0, 7).replace('-', '_');
    }

    /**
     * 获取eoc级别
     *
     * @return
     */
    public static int calculateWeight(EocDto eocDto) {
        return eocDto == null ? 0 : eocDto.calculateWeight();
    }

    /**
     * 解析动态时间区间计算和映射规则(dynamic_params)
     *
     * @param ruleObject
     * @param timeRange
     * @return
     */
    public static Map<String, JSONObject> parseDynamicParams(JSONObject ruleObject, String category, TimeRangeDto timeRange) {
        Map<String, JSONObject> resultMap = new HashMap();
        JSONArray dynamicParams = ruleObject.getJSONArray(FieldConstant.DYNAMIC_PARAMS);
        if (CollectionUtil.isEmpty(dynamicParams)) {
            return resultMap;
        }
        List<DynamicParamDto> dynamicParamDtos = dynamicParams.toJavaList(DynamicParamDto.class);
        for (DynamicParamDto paramDto : dynamicParamDtos) {
            String sourceName = paramDto.getSourceName();
            if (StringUtil.isBlank(sourceName)) {
                throw new BusinessException(ErrorCodeEnum.DYNAMIC_RULE_CONFIG_ERR);
            }
            LocalDateTime referTime;
            if (sourceName.equals(FieldConstant.LAST_MONITOR_TIME)) {
                referTime = timeRange.getStartTime();
            } else if (sourceName.equals(FieldConstant.MONITOR_TIME)) {
                referTime = timeRange.getEndTime();
            } else {
                throw new BusinessException(ErrorCodeEnum.DYNAMIC_RULE_CONFIG_ERR);
            }
            DynamicRuleDto ruleDto = paramDto.getRule();
            JSONObject valObject = new JSONObject();
            if (null != ruleDto && StringUtil.isNotBlank(ruleDto.getPeriod())) {
                String dateType = paramDto.getParamType() + "_" + ruleDto.getTimeType();
                String tempVal = LocalTimeUtil.getDateTime(referTime, ruleDto.getPeriod(), ruleDto.getRange(), ruleDto.getDay(), ruleDto.getAddDay(), dateType);
                tempVal = formatDateByConfig(tempVal, ruleDto.getFomart());
                valObject.put(FieldConstant.VALUE, StringUtil.nullToEmpty(tempVal));
            } else if (null != ruleDto && StringUtil.isNotBlank(ruleDto.getFomart())) {
                valObject.put(FieldConstant.VALUE, formatDateByConfig(referTime, ruleDto.getFomart()));
            } else {
                String tmp = null == referTime ? null : referTime.format(LocalTimeUtil.DATETIME_FMT_BY_DATE_AND_TIME_UNTIL_MS);
                valObject.put(FieldConstant.VALUE, tmp);
            }
            valObject.put(FieldConstant.TARGET, parseParamTarget(category, paramDto.getParamTarget()));

            if (FieldValConstant.CATEGORY_SCAN.equals(category)
                    || FieldValConstant.CATEGORY_SQL_SCRIPT.equals(category)) {
                valObject.put(FieldConstant.TYPE, paramDto.getParamType());
            }
            resultMap.put(paramDto.getParamName(), valObject);
        }
        return resultMap;
    }

    /**
     * 解析API侦测类型下静态节点的值
     *
     * @param ruleObject
     * @return
     */
    public static Map<String, JSONObject> parseStaticParams(JSONObject ruleObject, String category) {
        Map<String, JSONObject> resultMap = new HashMap();
        JSONArray staticParams = ruleObject.getJSONArray(FieldConstant.STATIC_PARAMS);
        if (CollectionUtil.isEmpty(staticParams)) {
            return resultMap;
        }
        staticParams.toJavaList(StaticParamDto.class).forEach(dto -> {
            JSONObject tempObject = new JSONObject();
            tempObject.put(FieldConstant.VALUE, dto.getParamValue());
            tempObject.put(FieldConstant.TARGET, parseParamTarget(category, dto.getParamTarget()));
            resultMap.put(dto.getParamName(), tempObject);
        });
        return resultMap;
    }

    private static String formatDateByConfig(String time, String format) {
        if (StringUtil.isBlank(format) || StringUtil.isBlank(time)) {
            return time;
        }
        DateTimeFormatter dateTimeFormatter = time.length() > 10 ? LocalTimeUtil.DATETIME_FMT_BY_DATE_AND_TIME_UNTIL_MS : LocalTimeUtil.DATE_FORMATTER;
        LocalDateTime temp = LocalDateTime.parse(time, dateTimeFormatter);
        return temp.format(DateTimeFormatter.ofPattern(format));
    }

    private static String formatDateByConfig(LocalDateTime time, String format) {
        if (null == time) {
            return null;
        }
        return time.format(DateTimeFormatter.ofPattern(format));
    }

    private static String parseParamTarget(String category, String paramTarget) {
        if (StringUtil.isNotBlank(paramTarget)) {
            return paramTarget;
        }

        if (FieldValConstant.CATEGORY_API.equals(category)) {
            return FieldValConstant.TARGET_MONITOR_ACTION;
        }

        if (FieldValConstant.CATEGORY_SCAN.equals(category)) {
            return FieldValConstant.TARGET_SCAN;
        }

        return "";
    }

    public static JSONArray transformToDynamicCondition(Map<String, JSONObject> allParam) {
        JSONArray condition = new JSONArray();
        allParam.forEach((k, v) -> {
                    JSONObject valObject = new JSONObject();
                    valObject.put("name", k);
                    valObject.put("type", v.getString("type"));
                    valObject.put("value", v.getString("value"));
                    condition.add(valObject);
                }
        );
        return condition;
    }

    public static EspSdkReq buildDataChangeSqlGetRequest(String ruleId, String tenantId, String productName, JSONArray dynamicConditions,
                                                         JSONObject rule, EocDto eocDto) {
        DataChangeSqlGetParameter sqlGetParameter = new DataChangeSqlGetParameter();
        JSONArray ruleArray = new JSONArray();
        JSONObject ruleParameter = new JSONObject();
        rule.fluentRemove(FieldConstant.DYNAMIC_PARAMS).fluentRemove(FieldConstant.STATIC_PARAMS).fluentRemove(FieldConstant.IS_CONTINUE)
                .fluentRemove(FieldConstant.IS_POLLING_IDS).fluentRemove(FieldConstant.IS_DEDUPLICATION).fluentRemove(FieldConstant.BK_INFO)
                .fluentRemove(FieldConstant.RETURN_COLUMNS).fluentRemove(FieldConstant.EOC_LEVEL).fluentRemove(FieldConstant.JOIN_PARAMS)
                .fluentRemove(FieldConstant.ALIAS).fluentRemove(FieldConstant.TABLE).fluentRemove(FieldConstant.MONITOR_TYPE)
                .fluentRemove(FieldConstant.DYNAMIC_CONDITION);
        ruleParameter.putAll(rule);
        ruleParameter.put(FieldConstant.SCRIPT, Base64.getEncoder().encodeToString(rule.getString(FieldConstant.SCRIPT).getBytes()));
        ruleParameter.put(FieldConstant.RULE_ID, ruleId);
        ruleParameter.put(FieldConstant.DYNAMIC_CONDITION, dynamicConditions);
        ruleArray.add(ruleParameter);
        sqlGetParameter.setRules(ruleArray);
        return new EspSdkReq(tenantId, eocDto, productName, sqlGetParameter, ruleId, FieldValConstant.DATA_CHANGE_SQL_GET);
    }

    /**
     * 根据op获取changeType 数据库数据变动类型
     *
     * @param op
     * @return
     */
    public static String getChangeType(String op) {
        if (FieldConstant.OP_UPDATE.equals(op)) {
            return FieldConstant.CHANGE_TYPE_UPDATE;
        }
        if (FieldConstant.OP_CREATE.equals(op)) {
            return FieldConstant.CHANGE_TYPE_CREATE;
        }
        return FieldConstant.CHANGE_TYPE_DELETE;
    }
}
