package com.digiwin.athena.cdme.service.srp.cache.impl;

import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.service.srp.cache.ICacheService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.stereotype.Service;

import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/19 15:28
 */
@Service("cdmeCacheService")
public class CacheService implements ICacheService {

    private static final Logger LOGGER = LoggerFactory.getLogger(CacheService.class);
    private static final String PATTERN_ALL_VAL = "*";

    private static final String PREFIX = "cdme:";

    @Autowired
    @Qualifier("cdmeRedisTemplate")
    private RedisTemplate<String, Object> redisTemplate;

    @Override
    public boolean hasKey(String key) {
        return redisTemplate.hasKey(key);
    }

    @Override
    public <T> T get(String key) {
        return key == null ? null : (T) redisTemplate.opsForValue().get(key);
    }

    @Override
    public void set(String key, Object value) {
        this.redisTemplate.opsForValue().set(key, value);
    }

    @Override
    public void set(String key, Object value, int seconds) {
        this.redisTemplate.opsForValue().set(key, value, seconds, TimeUnit.SECONDS);
    }

    @Override
    public boolean delete(Collection keys) {
        long n;
        try {
            n = this.redisTemplate.delete(keys);
        } catch (Exception e) {
            LOGGER.error("CacheOperation删除redis缓存异常，请查看！", e);
            return false;
        }
        if (n == 0) {
            LOGGER.warn("未清除redis缓存数据，请查看！");
            return false;
        }
        if (keys.size() != n) {
            LOGGER.warn("redis缓存清除丢失，清除数据为[{}], 总共清除数量[{}], 实际清除数量[{}]", JsonUtil.getJsonString(keys), keys.size(), n);
        }
        return true;
    }

    /**
     * scan不参与序列化,此处额外增加序列化及反序列化逻辑
     */
    @Override
    public Set<String> keys(String keyPattern) {
        if (StringUtil.isBlank(keyPattern) || PATTERN_ALL_VAL.equals(keyPattern)) {
            return new HashSet<>();
        }

        return this.redisTemplate.execute((RedisCallback<Set<String>>) connection -> {
            Set<String> keysTmp = new HashSet<>();
            try (Cursor<byte[]> cursor = connection.scan(new ScanOptions.ScanOptionsBuilder()
                    .match(this.serialize(keyPattern)).count(1000).build())) {
                while (cursor.hasNext()) {
                    keysTmp.add(this.deserialize(new String(cursor.next(), StandardCharsets.UTF_8)));
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            return keysTmp;
        });
    }

    @Override
    public Object hGet(String key, String field) {
        return this.redisTemplate.opsForHash().get(key, field);
    }

    @Override
    public Map<Object, Object> hGetAll(String key) {
        return this.redisTemplate.opsForHash().entries(key);
    }

    @Override
    public void hPut(String key, Object hashKey, Object value) {
        this.redisTemplate.opsForHash().put(key, hashKey, value);
    }

    @Override
    public Long hDelete(String key, Object hashKey) {
        return this.redisTemplate.opsForHash().delete(key, hashKey);
    }


    /**
     * 序列化
     */
    private String deserialize(String s) {
        assert s != null;
        int index = s.indexOf(PREFIX);
        if (index != -1) {
            return s.substring(PREFIX.length());
        }
        return s;
    }

    /**
     * 反序列化
     */
    private String serialize(String s) {
        if (s == null) {
            return null;
        }
        return PREFIX + s;
    }
}
