package com.digiwin.athena.cdme.service.client.impl;

import com.digiwin.app.service.DWServiceChainContext;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.core.util.CollectionUtil;
import com.digiwin.athena.cdme.core.util.LoggerFormatUtil;
import com.digiwin.athena.cdme.core.util.ResultHelper;
import com.digiwin.athena.cdme.pojo.dto.ResultDto;
import com.digiwin.athena.cdme.service.client.IMonitorResultLogClient;
import com.digiwin.athena.cdme.service.client.request.BaseMonitorResultLogReq;
import com.digiwin.athena.cdme.service.client.request.MonitorFailResultLogReq;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.connection.CorrelationData;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.UUID;

/**
 * @description: 侦测结果埋点
 * @author: liunansheng
 * @date: 2021/12/15 9:43
 */
@Service("cdmeMonitorResultLogClient")
public class MonitorResultLogClient implements IMonitorResultLogClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(MonitorResultLogClient.class);

    private static final String EXCHANGE = "it.sys.exchange";

    private static final String ROUTING_KEY = "mon.monitor";
    private static final String ERR_ROUTING_KEY_ERR = "mon.error";

    private RabbitTemplate template;

    public MonitorResultLogClient(@Qualifier("cdmeRabbitTemplate")RabbitTemplate template) {
        this.template = template;
    }

    @Override
    public ResultDto<Void> logMonitorResult(List<BaseMonitorResultLogReq> resultLogReqList) {
        if (CollectionUtil.isEmpty(resultLogReqList)) {
            return ResultHelper.generateSuccessResult(null);
        }
        for (BaseMonitorResultLogReq resultLogReq : resultLogReqList) {
            logMonitorResult(resultLogReq);
        }
        return ResultHelper.generateSuccessResult(null);
    }

    @Override
    public ResultDto<Void> logMonitorResult(BaseMonitorResultLogReq resultLogReq) {
        try {
            if (DWServiceChainContext.getContext() != null) {
                resultLogReq.setParentSessionId(DWServiceChainContext.getContext().getParentId());
                resultLogReq.setSessionId(DWServiceChainContext.getContext().getId());
                resultLogReq.setAppId(DWServiceChainContext.getContext().getAppId());
                if (DWServiceChainContext.getContext().getTraceElement() != null
                        && DWServiceChainContext.getContext().getTraceElement().getRoot() != null) {
                    resultLogReq.setTraceId(DWServiceChainContext.getContext().getTraceElement().getRoot().getId());
                }
            }
            resultLogReq.setCreatedTime(LocalDateTime.now());
            //成功和失败发不同队列
            String routingKey = resultLogReq instanceof MonitorFailResultLogReq ? ERR_ROUTING_KEY_ERR : ROUTING_KEY;
            String msgId = UUID.randomUUID().toString();
            template.convertAndSend(EXCHANGE, routingKey, JsonUtil.getJsonString(resultLogReq), new CorrelationData(msgId));
            LOGGER.info("推送给运维模组成功,traceId:{},ruleId:{},eoc:{},msgId:{},类型:{}", resultLogReq.getTraceId(),
                    resultLogReq.getRuleCode(), resultLogReq.getEocId(), msgId, resultLogReq.getClass().getSimpleName());
            return ResultHelper.generateSuccessResult(null);
        } catch (Exception e) {
            LOGGER.error(LoggerFormatUtil.formatMessage("埋点侦测结果异常, req:{}", resultLogReq), e);
            return ResultHelper.generateFailResult("埋点侦测结果异常");
        }
    }
}
