package com.digiwin.athena.cdme.mq.consumer.monitor;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.aop.TraceId;
import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import com.digiwin.athena.cdme.core.constant.MqttConstant;
import com.digiwin.athena.cdme.core.exception.BusinessException;
import com.digiwin.athena.cdme.core.util.FilterUtil;
import com.digiwin.athena.cdme.core.util.MonitorHelper;
import com.digiwin.athena.cdme.core.util.StringUtil;
import com.digiwin.athena.cdme.pojo.dto.EocDto;
import com.digiwin.athena.cdme.repository.model.MonitorRuleCdcModel;
import com.digiwin.athena.cdme.service.facade.auth.IContextFacadeService;
import com.digiwin.athena.cdme.service.facade.detection.IMonitorFacadeService;
import com.digiwin.athena.cdme.service.srp.cache.ICacheService;
import com.digiwin.athena.cdme.service.srp.db.IMonitorRuleCdcService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.rabbitmq.client.Channel;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.Exchange;
import org.springframework.amqp.rabbit.annotation.Queue;
import org.springframework.amqp.rabbit.annotation.QueueBinding;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @description: 监听发起侦测的MQ消息
 * @author: dongwh
 * @date: 2021/6/7 9:37
 */
@Component("cdmeMonitorExecuteCdcListener")
public class MonitorExecuteCdcListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(MonitorExecuteCdcListener.class);

    private final IMonitorRuleCdcService ruleCdcService;

    private final IMonitorFacadeService monitorFacadeService;

    private final ICacheService redisService;

    private final IContextFacadeService contextService;

    public MonitorExecuteCdcListener(IMonitorRuleCdcService ruleCdcService, IMonitorFacadeService monitorFacadeService, ICacheService redisService, IContextFacadeService contextService) {
        this.ruleCdcService = ruleCdcService;
        this.monitorFacadeService = monitorFacadeService;
        this.redisService = redisService;
        this.contextService = contextService;
    }

    @RabbitListener(bindings = {@QueueBinding(
            value = @Queue(value = ConfigConstant.CDC_MQ_QUEUE,
                    durable = "true"),
            exchange = @Exchange(value = ConfigConstant.CDC_MQ_EXCHANGE, ignoreDeclarationExceptions = "true"),
            key = {ConfigConstant.CDC_MQ_ROUTINGKEY}
    )
    },containerFactory = "cdmeRabbitListenerContainerFactory", errorHandler = "cdmeConsumeExceptionHandle")
    @TraceId
    public void consumeMonitorMsg(Message message, Channel channel) throws Exception {
        String queueMsg = new String(message.getBody(), StandardCharsets.UTF_8);
        LOGGER.info("侦测引擎接收CDC的处理消息为:[{}]", queueMsg);

        JSONObject data = JsonUtil.getObject(queueMsg);
        JSONObject after = ObjectUtils.isEmpty(data.getJSONObject(FieldConstant.DATA_AFTER)) ? data.getJSONObject(FieldConstant.DATA_BEFORE) : data.getJSONObject(FieldConstant.DATA_AFTER);
        String tenantId = StringUtils.isEmpty(data.getString(FieldConstant.TENANTID)) ? after.getString(FieldConstant.TENANT_ID) : data.getString(FieldConstant.TENANTID);
        String tenantSid = after.getString(FieldConstant.TENANT_SID);
        String tableName = data.getJSONObject(FieldConstant.DATA_SOURCE).getString(FieldConstant.DATA_TABLE);
        String db = data.getJSONObject(FieldConstant.DATA_SOURCE).getString(FieldConstant.DATA_DB);
        String op = data.getString(FieldConstant.DATA_OP);
        String changeType = MonitorHelper.getChangeType(op);
        String ruleId = after.getString(MqttConstant.RULE_ID);
        List<MonitorRuleCdcModel> monitorRuleCdcModelList = new ArrayList<>();
        if (StringUtil.isEmpty(tenantId)) {
            monitorRuleCdcModelList = ruleCdcService.getByTenantSidAndTableAndChangeType(tenantSid, tableName, changeType);
            if (CollectionUtils.isEmpty(monitorRuleCdcModelList)) {
                LOGGER.error("过滤原因，没有配置该CDC侦测规则，tenantSid={}, tableName={}, changeType={}", tenantSid, tableName, changeType);
                throw new BusinessException("没有匹配的CDC侦测信息！");
            }

            // 已确认一个tenantSid只对应唯一的tenantId
            tenantId = monitorRuleCdcModelList.get(0).getTenantId();
        }

        monitorRuleCdcModelList.clear();

        ruleCdcService.queryCdcRuleByTenantIdAndTableAndOp(monitorRuleCdcModelList, tenantId, db, tableName, changeType);


        if (CollectionUtils.isEmpty(monitorRuleCdcModelList)) {
            LOGGER.error("过滤原因，没有配置该CDC侦测规则，缓存和数据库都查不到，tenantId={}, tableName={}, changeType={}", tenantId, tableName, changeType);
            throw new BusinessException("没有匹配的CDC侦测信息！");
        }
        if(StringUtils.isNotBlank(ruleId)){
            monitorRuleCdcModelList=monitorRuleCdcModelList.stream().filter(res->StringUtils.equals(res.getRuleId(),ruleId)).collect(Collectors.toList());
        }
        if (CollectionUtils.isEmpty(monitorRuleCdcModelList)) {
            LOGGER.error("过滤原因，没有配置该CDC侦测规则，缓存和数据库都查不到，tenantId={}, tableName={}, changeType={}", tenantId, tableName, changeType);
            throw new BusinessException("没有匹配的CDC侦测信息！");
        }
        // 根据侦测规则配置的过滤条件过滤规则
        Integer size = monitorRuleCdcModelList.size();
        for (; size - 1 >= 0; size--) {
            if (FilterUtil.filterData(data, op, monitorRuleCdcModelList.get(size - 1))) {
                monitorRuleCdcModelList.remove(size - 1);
            }
        }
        if(CollectionUtils.isEmpty(monitorRuleCdcModelList)){
            LOGGER.error("通过规则配置的过滤条件进行过滤没有匹配的CDC侦测信息，tenantId={}, tableName={}, changeType={}", tenantId, tableName, changeType);
            throw new BusinessException("通过规则配置的过滤条件进行过滤没有匹配的CDC侦测信息！");
        }

        LOGGER.info("过滤后的规则是，tenantId={}, tableName={}, changeType={}，rule={}", tenantId, tableName, changeType, JSON.toJSONString(monitorRuleCdcModelList));

        // 按租户规则集发起侦测
        monitorRuleCdcModelList.forEach(rule ->{
            EocDto eocDto = new EocDto(rule.getEocCompanyId(), rule.getEocSiteId(), null);
            JSONArray monitorData = new JSONArray();
            if(MqttConstant.CATEGORY.equals(rule.getCategory())){
                monitorData.add(after);
                monitorData.getJSONObject(0).remove(MqttConstant.UNI_TENANTID);
                monitorData.getJSONObject(0).remove(MqttConstant.RULE_ID);
            } else {
                monitorData.add(getData(after, rule));
            }
            try {
                contextService.constructContext(rule.getTenantId());
            } catch (Exception e) {
                LOGGER.error("设置上下文失败:{}", e);
                throw new BusinessException("设置上下文失败", e);
            }
            monitorFacadeService.executeCdc(rule, eocDto, monitorData);
        });


        /** 正常ack消息 */
        channel.basicAck(message.getMessageProperties().getDeliveryTag(), true);
    }

    private JSONObject getData(JSONObject after, MonitorRuleCdcModel monitorRuleCdcModel) {
        JSONArray actionParams = JsonUtil.parseArray(monitorRuleCdcModel.getActionParams());
        JSONObject afterData = new JSONObject();
        for (Object object : actionParams) {
            JSONObject param = (JSONObject) object;
            afterData.put(param.getString(FieldConstant.DATA_VALUE), after.get(param.getString(FieldConstant.DATA_NAME)));
        }
        return afterData;
    }
}
