package com.digiwin.athena.cdme.service.client.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.common.DWApplicationConfigUtils;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.app.service.restful.DWRequestMethod;
import com.digiwin.athena.cdme.JsonUtil;
import com.digiwin.athena.cdme.constant.FieldConstant;
import com.digiwin.athena.cdme.core.constant.ConfigConstant;
import com.digiwin.athena.cdme.core.enums.ErrorCodeEnum;
import com.digiwin.athena.cdme.core.exception.ResponseStatusException;
import com.digiwin.athena.cdme.service.client.AbstractExecuteClient;
import com.digiwin.athena.cdme.service.client.IIamClient;
import com.digiwin.iam.HttpResponseModel;
import com.digiwin.iam.IAMService;
import com.digiwin.iam.ServiceModel;
import org.apache.commons.lang3.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * @description:
 * @author: dongwh
 * @date: 2021/10/19 17:05
 */
@Service("cdmeIamClient")
public class IamClient extends AbstractExecuteClient implements IIamClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(IamClient.class);

    private static final String TENANT_KEY = "testTenant";

    private static final String INTEGRATION_ACCOUNT = "integration";
    private static final String PASSWORD_HASH = DWApplicationConfigUtils.getProperty("cdme.IamClient.PASSWORD_HASH");

    @Override
    public boolean isTest(String tenantId) {
        /** 组装body信息 */
        Map<String, Object> param = new HashMap<>(1);
        /** 调用iam入参*/
        param.put("id", tenantId);
        /** 设置header信息 */
        /** NOTE 传入非测试租户iam接口会抛出 org.springframework.web.client.HttpServerErrorException: 500 异常，由于此异常过大无法单独进行捕获，需注意*/
        LOGGER.info("侦测调用IAM查看是否是测试租户请求参数:[{}]", JsonUtil.getJsonString(param));
        Map<String, Object> resp = post(ConfigConstant.IAM_TENANT_SIMPLE_URL, param, null, Map.class);
        LOGGER.info("侦测调用IAM查看是否是测试租户返回结果:[{}]", JsonUtil.getJsonString(resp));
        if (resp.containsKey(TENANT_KEY)) {
            return (boolean) resp.get(TENANT_KEY);
        }
        return false;
    }


    /**
     * 解析token判断tenantId是否一致
     *
     * @param tenantId
     * @return
     */
    @Override
    public boolean isTenantIdMatchByToken(String tenantId) {
        /** 解析token获取租户的信息 **/
        JSONObject userInfo = getUserInfoByAnalyzeToken();
        if (userInfo.isEmpty()) {
            return false;
        }
        /** 判断解析token出来的tenantId与入参中的tenantId是否一致 **/
        return Objects.equals(tenantId, userInfo.getString(FieldConstant.TENANTID));
    }

    /**
     * 解析token获取租户的信息
     *
     * @return
     */
    private JSONObject getUserInfoByAnalyzeToken() {
        return post(ConfigConstant.IAM_USER_TOKEN_ANALYZE_URL, null, null, JSONObject.class);
    }

    /**
     * 获取登录用户信息
     *
     * @return
     */
    @Override
    public JSONObject getLoginResponse(String secretKey) {
        Map<String, Object> paramMap = new HashMap<>(4);
        paramMap.put(FieldConstant.SECRETKEY, secretKey);
        paramMap.put(FieldConstant.IDENTITYTYPE, "secretKey");

        ServiceModel iamModel = new ServiceModel();
        iamModel.setInvokeURL(ConfigConstant.IAM_USERLOGINURL);
        iamModel.setParams(paramMap);
        iamModel.setRequestMethod(DWRequestMethod.POST);
        try {
            HttpResponseModel responseModel = (HttpResponseModel) IAMService.invoke(iamModel);
            /** IAMService.invoke 内部已经做了!200的处理：抛出DWExternalAPIHttpFailedException，因此可以直接取返回值 */
            return JsonUtil.getObject(responseModel.getResponseBody());
        } catch (Exception e) {
            throw new ResponseStatusException(ErrorCodeEnum.IAM_500, e);
        }

    }

    @Override
    public String getDefaultLocale() {
        ServiceModel iamModel = new ServiceModel();
        iamModel.setInvokeURL("/api/iam/v2/tenant/metadata/language");
        iamModel.setToken(DWServiceContext.getContext().getToken());
        iamModel.setRequestMethod(DWRequestMethod.GET);
        LOGGER.info("获取租户默认语言别，token为：{}", iamModel.getToken());
        try {
            HttpResponseModel responseModel = (HttpResponseModel) IAMService.invoke(iamModel);
            /** IAMService.invoke 内部已经做了!200的处理：抛出DWExternalAPIHttpFailedException，因此可以直接取返回值 */
            JSONObject result = JsonUtil.getObject(responseModel.getResponseBody());
            LOGGER.info("获取租户默认语言别返回内容：{}", responseModel.getResponseBody());
            if (BooleanUtils.isNotTrue(result.getBoolean("success"))) {
                throw new ResponseStatusException(ErrorCodeEnum.IAM_500);
            }
            return result.getString("data");
        } catch (Exception e) {
            throw new ResponseStatusException(ErrorCodeEnum.IAM_500, e);
        }
    }

    @Override
    protected ErrorCodeEnum getErrorCodeByHttpStatus(String url, HttpStatus httpStatus) {
        if (null == httpStatus) {
            return ErrorCodeEnum.IAM_OTHER_ERR;
        }
        switch (httpStatus) {
            case INTERNAL_SERVER_ERROR :
                return ErrorCodeEnum.IAM_500;
            case BAD_GATEWAY :
                return ErrorCodeEnum.IAM_502;
            case SERVICE_UNAVAILABLE :
                return ErrorCodeEnum.IAM_503;
            case GATEWAY_TIMEOUT :
                return ErrorCodeEnum.IAM_504;
        }
        return ErrorCodeEnum.IAM_OTHER_ERR;
    }

    /**
     * 是否需要在头部加入租户、app授权信息
     * @return
     */
    @Override
    protected boolean isNeedAuthInfo() {
        return true;
    }

    @Override
    public JSONObject getIntegration(String tenantId) {
        // 组装body
        Map<String, String> requestBody = new HashMap<>(4);
        requestBody.put("tenantId", tenantId);
        requestBody.put("userId", INTEGRATION_ACCOUNT);
        requestBody.put(getMapKey(), PASSWORD_HASH);
        LOGGER.info("调用iam获取集成帐号租户为：{}", tenantId);
        JSONObject result = post(ConfigConstant.IAM_INTEGRATION_URL, requestBody, null, JSONObject.class);
        LOGGER.info("调用iam获取集成帐号信息为：{}", JsonUtil.getJsonString(result));
        return result;
    }

    @Override
    public JSONObject getTenantId(String tenantSid) {
        // 组装body
        Map<String, String> requestBody = new HashMap<>(4);
        requestBody.put("tenantSid", tenantSid);
        requestBody.put("userId", INTEGRATION_ACCOUNT);
        requestBody.put(getMapKey(), PASSWORD_HASH);
        LOGGER.info("调用iam获取租户ID请求参数为：{}", tenantSid);
        Consumer<HttpHeaders> headerAdder = headers -> headers.set(FieldConstant.DW_APP_TOKEN_HEADER_NAME, ConfigConstant.MULTIPLEAPPENV?ConfigConstant.MERGED_IAM_APP_TOKEN:ConfigConstant.IAM_APP_TOKEN);

        JSONObject result = postWithoutRouterkey(ConfigConstant.IAM_INTEGRATION_URL, requestBody, null, JSONObject.class, headerAdder);
        LOGGER.info("调用iam获取帐号信息为：{}", JsonUtil.getJsonString(result));
        return result;
    }

    private String getMapKey(){
        return  "pas"+"swordHash";
    }
}
