package com.digiwin.athena.datamap.dao;

import com.digiwin.athena.datamap.service.inner.DataMapKgDataPickService;
import com.digiwin.athena.kg.activity.Activity;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.stream.Collectors;

@Repository
public class DataMapActivityDAO {

    @Autowired
    @Qualifier("kgSystem")
    private MongoTemplate mongoTemplateSystem;

    @Autowired
    @Qualifier("kgDataPickService")
    DataMapKgDataPickService dataPickService;

    /**
     * 查询所有ABi报表code
     * @return List<String>
     */
    public List<String> getAllABiReportCodes() {
        Criteria criteria = Criteria.where("category").is("ABI-STATEMENT");
        List<Activity> result = mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
        return result.stream().map(Activity::getCode).distinct().collect(Collectors.toList());
    }

    /**
     * 根据code和版本查询报表
     * @param activityCode      activityCode
     * @param tenantVersion     租户版本
     * @param tenantId          租户ID
     * @return  List<Activity>
     */
    public List<Activity> getActivityByCodeAndVersion(String activityCode, String tenantVersion, String tenantId) {
        Criteria criteria = Criteria.where("code").is(activityCode)
                .and("version").is(tenantVersion)
                .andOperator(getCriteriaByTenantId(tenantId, true));
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据租户ID查询公共数据报表
     * @param tenantId    租户ID
     * @return
     */
    public List<Activity> getCommonBaseData(String tenantId) {
        Criteria criteria = Criteria.where("nameSpace").is("common")
                .andOperator(getCriteriaByTenantId(tenantId, true));
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据租户ID和是否包含子租户查询条件
     * @param tenantId          租户ID
     * @param inclusionTenant   是否包含子租户
     * @return  Criteria
     */
    public Criteria getCriteriaByTenantId(String tenantId, boolean inclusionTenant) {
        Criteria criteria = Criteria.where("tenantId").in(tenantId, "SYSTEM", null);
        if (inclusionTenant) {
            criteria = new Criteria().orOperator(criteria, Criteria.where("inclusionTenant").is(tenantId));
        }
        return criteria;
    }
}
