/*
 * FileName: ProcessVariableUtil
 * Author:   DONGSK
 * Datetime: 2020/12/7 11:13
 * Description: 項目变量工具
 * History:
 * 作者姓名 --修改时间 --版本号--描述
 */
package com.digiwin.athena.datamap.engine.service.utils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ProcessVariableUtil
 * 項目变量工具
 *
 * @author DONGSK
 * @date 2020/12/7
 * @since 1.0.0
 */
public class ProjectVariableUtil {

    // private static final String _REGEX = "\\$\\(\\w+\\)";
    /**
     * 正则匹配
     * $(complexArray)
     * $(complexArray[0])
     * $(complexArray[0].name)
     * $(complexArray[1].user)
     * $(complexArray[1].user.id)
     * $(complexObject.type)
     * $(complexObject.user)
     * $(complexObject.user.name)
     * $(complexObject.user.dept[0]),
     * $(complexObject.user.dept[1].name)
     */
    private static final String _REGEX = "\\$\\(\\w+((\\[[0-9]+\\]|[\\.]|\\w+)+)+\\)";

    private static void appendLog(String logStr) {

    }

    private static boolean isJSONString(String str, String start, String end) {
        if (Objects.isNull(str)) {
            return false;
        }
        String _v = str.trim();
        return _v.startsWith(start) && _v.endsWith(end);
    }

    public static boolean isJSONObject(String str) {
        return isJSONString(str, "{", "}");
    }

    public static boolean isJSONArray(String str) {
        return isJSONString(str, "[", "]");
    }

    public static boolean isProcessVariableKey(String key) {
        if (Objects.isNull(key)) {
            return false;
        }
        String _key = key.trim();
        return _key.startsWith("$(") && _key.endsWith(")");
    }

    /*
     * public static String processKey(String key) { String _key = key.trim(); if (isProcessVariableKey(key)) { _key = _key.substring(2, _key.length() - 1); } return _key; }
     */

    /**
     * 取得項目变量值
     * 推荐使用
     *
     * @param key       键值 $(key)
     * @param variables 所有項目变量
     * @return 返回值 不存在返回null
     */
    public static String getValueString(String key, Map<String, Object> variables) {
        // 是否为項目变量
        if (!isProcessVariableKey(key)) {
            return null;
        }
        String _key = key.substring(2, key.length() - 1);
        String returnValue = getValueQuery(_key, variables);
        if (Objects.isNull(returnValue)) {
            if (variables.containsKey(_key)) {

                if (variables.get(_key) instanceof Map) {
                    Map tTmpValue = (Map) variables.get(_key);
                    returnValue = (new JSONObject(tTmpValue)).toString();
                } else if (variables.get(_key) instanceof List) {
                    List tTmpValue = (List) variables.get(_key);
                    returnValue = (new JSONArray(tTmpValue)).toString();
                } else {
                    returnValue = variables.get(_key) + "";
                }

            } else {
                appendLog(String.format("the  key[%s] is not found", key));
            }
        }
        return returnValue;
    }

    /**
     * 支持query查找
     * 支持以下key：
     * $(complexArray[0])
     * $(complexArray[0].name)
     * $(complexArray[1].user)
     * $(complexArray[1].user.id)
     * $(complexObject.type)
     * $(complexObject.user)
     * $(complexObject.user.name)
     * $(complexObject.user.dept[0]),
     * $(complexObject.user.dept[1].name)
     *
     * @param key       a.b或者a[0].c或者a.b[0]或者a[0]或者a.b[0].c
     * @param variables 項目变量
     * @return 返回值
     */
    private static String getValueQuery(String key, Map<String, Object> variables) {
        if (!(key.contains(".") || (key.contains("[") && key.contains("]")))) {
            return null;
        }
        boolean isArray;
        int dotIndex = key.indexOf('.');
        int bracketIndex = key.indexOf('[');
        if (dotIndex > 0 && bracketIndex > 0) {
            isArray = dotIndex >= bracketIndex;
        } else {
            isArray = dotIndex <= 0;
        }
        String rKey = isArray ? key.substring(0, bracketIndex) : key.substring(0, dotIndex);
        String nextValue = getValueString(String.format("$(%s)", rKey), variables);
        if (Objects.isNull(nextValue)) {
            return null;
        }
        JSONObject jsonObject = new JSONObject();
        if (isArray) {
            jsonObject.put(rKey, new JSONArray(nextValue));
        } else {
            jsonObject.put(rKey, new JSONObject(nextValue));
        }
        String path = "/" + key.replace('.', '/')
                .replace('[', '/')
                .replace("]", "").trim();
        Object val = jsonObject.query(path);
        if (Objects.isNull(val)) {
            return null;
        }
        return val.toString();
    }

    /**
     * 取得項目变量值
     *
     * @param key       键值 $(key)
     * @param variables 所有項目变量
     * @return 返回值 不存在返回null
     */
    public static Object getValueObject(String key, Map<String, Object> variables) {
        String returnValue = getValueString(key, variables);
        if (isJSONObject(returnValue)) {
            return new JSONObject(returnValue);
        } else if (isJSONArray(returnValue)) {
            return new JSONArray(returnValue);
        } else {
            return returnValue;
        }
    }

    /**
     * 获取項目变量值(Json{...}结构)，并将结果转换为Map
     *
     * @param key       键值 $(key)
     * @param variables 所有項目变量
     * @return 若值不是Json{...}结构，则返回空Map
     */
    public static Map<String, String> geValueToMap(String key, Map<String, Object> variables) {
        Map<String, String> result = new HashMap<>();
        String value = getValueString(key, variables);
        if (isJSONObject(value)) {
            JSONObject jsonObject = new JSONObject(value);
            jsonObject.keySet().forEach(s -> {
                result.put(s, jsonObject.get(s).toString());
            });
        } else {
            appendLog(String.format("the value[%s] of key[%s] is not JSONObject String", value, key));
        }
        return result;
    }

    /**
     * 处理文本中的項目变量
     *
     * @param context   文本
     * @param variables 所有項目变量
     * @return 返回替换后的文本
     */
    public static String parseContext(String context, Map<String, Object> variables) {

        if (null == variables || variables.isEmpty()) {
            return context;
        }

        if (Objects.isNull(context) || "".equals(context)) {
            return context;
        }

        String returnValue = context;
        // 匹配 $(key)
        Pattern pattern = Pattern.compile(_REGEX);
        Matcher matcher = pattern.matcher(context);
        while (matcher.find()) {
            String variablePlaceholder = matcher.group();
            // String variableKey = variablePlaceholder.substring(2, variablePlaceholder.length() - 1);
            String variableValue = getValueString(variablePlaceholder, variables);
            if (!Objects.isNull(variableValue)) {
                returnValue = returnValue.replace(variablePlaceholder, variableValue);
            }
        }
        return returnValue;
    }

    /**
     * 处理Json{...}结构文本中的項目变量，并将文本转换为Map<String, String>
     *
     * @param context   Json{...}结构文本
     * @param variables 所有項目变量
     * @return 若值不是Json{...}结构文本，则返回空Map
     */
    public static Map<String, String> parseContextToMap(String context, Map<String, Object> variables) {
        Map<String, String> result = new HashMap<>();
        if (Objects.isNull(context) || "".equals(context)) {
            return result;
        }
        String _context = parseContext(context, variables);
        if (isJSONObject(_context)) {
            JSONObject jsonObject = new JSONObject(_context);
            jsonObject.keySet().forEach(s -> {
                result.put(s, jsonObject.get(s).toString());
            });
        }
        return result;
    }

    /**
     * 将 JSONObject 转换为 Map<String, String>
     *
     * @param jsonObject 被转换对象
     * @return
     */
    public static Map<String, String> JSONToMapString(JSONObject jsonObject) {
        Map<String, String> result = new HashMap<>();
        jsonObject.keySet().forEach(s -> {
            result.put(s, jsonObject.get(s).toString());
        });
        return result;
    }

    /**
     * 将 JSONObject 转换为 Map<String, Object>
     *
     * @param jsonObject 被转换对象
     * @return
     */
    public static Map<String, Object> JSONToMapObject(JSONObject jsonObject) {
        Map<String, Object> result = new HashMap<>();
        jsonObject.keySet().forEach(s -> {
            result.put(s, jsonObject.get(s));
        });
        return result;
    }

    /**
     * 項目变量转换为json字符串
     *
     * @param variables 項目变量集合
     * @return 項目变量Json
     */
    public static String ConvertToJsonString(Map<String, String> variables) {
        return ConvertToJSONObject(variables).toString();
    }

    /**
     * 項目变量转换为JSONObject 对象
     *
     * @param variables 項目变量集合
     * @return 項目变量JSONObject
     */
    public static JSONObject ConvertToJSONObject(Map<String, String> variables) {
        JSONObject jsonObject = new JSONObject();
        variables.forEach((k, v) -> {
            if (isJSONObject(v)) {
                // 对象
                jsonObject.put(k, new JSONObject(v));
            } else if (isJSONArray(v)) {
                // 数组
                jsonObject.put(k, new JSONArray(v));
            } else {
                // 其他类型
                jsonObject.put(k, v);
            }
        });
        return jsonObject;
    }

    /**
     * 從眾多變量中，取出要保留的那些並回傳
     * @param pVariables 所有變量
     * @param pKeepedVarKeys 要保留的變量keys
     * @return
     */
    public static Map<String, Object> keepVariables(Map<String, Object> pVariables, List<String> pKeepedVarKeys) {
        Map<String, Object> tRtn = new HashMap<>();
        if (Objects.isNull(pKeepedVarKeys)) {
            tRtn = pVariables;
        } else {
            for (String tTmpVarKey : pKeepedVarKeys) {
                if (pVariables.containsKey(tTmpVarKey)) {
                    tRtn.put(tTmpVarKey, pVariables.get(tTmpVarKey));
                }
            }
        }
        return tRtn;
    }
}
