package com.digiwin.athena.datamap.mechanism.convert.parsers.fieldValueType;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.datamap.mechanism.convert.CustomFuncResult;
import com.digiwin.athena.datamap.mechanism.convert.MechanismParserFactory;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.common.MechanismVariable;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Description 布尔类型解析器
 * @Author liugl
 * @Date 2024/7/22 10:29
 **/
@Service
@Slf4j
public class BooleanParser implements FieldValueTypeParser {

    @Autowired
    MechanismParserFactory mechanismParserFactory;

    @Override
    public boolean accept(String dataType) {
        return MechanismEnum.FieldValueType.BOOLEAN.getCode().equalsIgnoreCase(dataType);
    }

    @Override
    public JSONArray parse(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        // 处理全局类型的判断值，例如自定义值，机制函数，枚举值
        CustomFuncResult rightCondResult = mechanismParserFactory.getRightCondParserByType(conditionConfig.getRightCondType())
                .preParse(conditionConfig, metaDataList);
        if (!rightCondResult.getIsNeedNextJudge()) {
            return rightCondResult.getMetaDataList();
        }
        if (MechanismEnum.ConditionType.HIGH_GRADE.getCode().equalsIgnoreCase(conditionConfig.getType())) {
            return parseHighGrade(conditionConfig, metaDataList);
        }
        return parsePrimary(conditionConfig, metaDataList);
    }

    /**
     * 基础条件比较
     * @param conditionConfig
     * @param metaDataList
     * @return 返回结果
     */
    private JSONArray parsePrimary(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        JSONArray result = new JSONArray();
        Boolean rightToValue;
        try {
            MechanismVariable rightTo = conditionConfig.getRight().getTo();
            rightToValue = Boolean.valueOf(rightTo.getValue().toString());
        } catch (Exception e) {
            log.error("BooleanParser.parse 右侧条件判断值转换失败，入参：{}", conditionConfig);
            return result;
        }
        String dataName = conditionConfig.getLeft().getData_name();
        List<Object> resultList = metaDataList.stream().filter(v -> {
            JSONObject metaDataObj = JSON.parseObject(JSON.toJSONString(v));
            Boolean sourceValue = null;
            try {
                sourceValue = metaDataObj.getBooleanValue(dataName);
            } catch (Exception e) {
                log.warn("BooleanParser.parse 表格数据转换失败，数据为：{}", v);
            }
            // boolean类型只有一种判断方式 相等 equal
            return rightToValue.equals(sourceValue);
        }).collect(Collectors.toList());
        return JSONArray.parseArray(JSON.toJSONString(resultList));
    }

    /**
     * 高级条件 自定义函数比较
     *
     * @param conditionConfig
     * @param metaDataList
     * @return 返回结果
     */
    private JSONArray parseHighGrade(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        JSONArray resultArray = new JSONArray();
        String functionValueType = conditionConfig.getLeft().getFunctionValueType();
        Object functionValue = conditionConfig.getLeft().getFunctionValue();
        Object value = conditionConfig.getRight().getTo().getFunctionValue();
        // 目前自定义函数输出类型只有数值和boolean类型
        if (MechanismEnum.FieldValueType.NUMERIC.getCode().equalsIgnoreCase(functionValueType)) {
            try {
                BigDecimal functionValueTemp = new BigDecimal(functionValue.toString());
                BigDecimal valueTemp = new BigDecimal(value.toString());
                if (functionValueTemp.compareTo(valueTemp) == 0) {
                    resultArray.addAll(metaDataList);
                    return resultArray;
                }
            } catch (Exception e) {
                log.error("BooleanParser.compareCondition 失败，e={}", e);
                return null;
            }
            return null;
        }
        if (ObjectUtils.nullSafeEquals(functionValue.toString(), value.toString())) {
            resultArray.addAll(metaDataList);
            return resultArray;
        }
        return null;
    }

    @Override
    public Boolean isBothEqual(Object source, Object targetLeft, Object targetRight, String opType) {
        try {
            Boolean sourceBoolean = Boolean.valueOf(source.toString());
            Boolean targetRightBoolean = Boolean.valueOf(targetRight.toString());
            return sourceBoolean.equals(targetRightBoolean);
        } catch (Exception e) {
            log.error("BooleanParser.isBothEqual 失败，e={}", e);
            return Boolean.FALSE;
        }
    }
}
