package com.digiwin.athena.datamap.mechanism.convert.parsers.fieldValueType;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.util.Optional;

/**
 * @Description 日期类型解析器
 * @Author liugl
 * @Date 2024/7/22 10:29
 **/
@Service
@Slf4j
public class DateTimeParser implements FieldValueTypeParser {

    @Override
    public boolean accept(String dataType) {
        return MechanismEnum.FieldValueType.DATETIME.getCode().equalsIgnoreCase(dataType);
    }

    @Override
    public JSONArray parse(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        // TODO 支持机制参数类型
        Object rightToValueObj = conditionConfig.getRight().getTo().getValue();
        String currentDateType = conditionConfig.getRight().getType();
        Integer rightToOffsetValue = Integer.valueOf(rightToValueObj.toString().split("[/.]")[0]);
        LocalDateTime rightToDate = getOffsetDate(currentDateType, rightToOffsetValue);
        LocalDateTime rightFromDate = null;
        if (conditionConfig.getOpType().equals(MechanismEnum.OperateType.BETWEEN.getCode())) {
            Object rightFromValueObj = conditionConfig.getRight().getFrom().getValue();
            Integer rightFromOffsetValue = Integer.valueOf(rightFromValueObj.toString().split("[/.]")[0]);
            rightFromDate = getOffsetDate(currentDateType, rightFromOffsetValue);
        }
        String dataName = conditionConfig.getLeft().getData_name();
        String opType = conditionConfig.getOpType();
        return doFilter(metaDataList, dataName, opType, rightToDate, rightFromDate);
    }

    private JSONArray doFilter(JSONArray metaDataList, String dataName, String opType, LocalDateTime rightToDate, LocalDateTime rightFromDate) {
        if (metaDataList == null || metaDataList.isEmpty()) {
            return new JSONArray();
        }
        JSONArray result = new JSONArray();
        for (Object each : metaDataList) {
            JSONObject metaDataObj = JSON.parseObject(JSON.toJSONString(each));
            Optional<LocalDateTime> sourceDateTime = parseToLocalDateTime(metaDataObj.getString(dataName));
            if (!sourceDateTime.isPresent()) {
                continue;
            }
            LocalDateTime source = sourceDateTime.get();
            if (isMatch(source, rightToDate, rightFromDate, opType)) {
                result.add(metaDataObj);
            }
        }
        return result;
    }

    private boolean isMatch(LocalDateTime source, LocalDateTime rightToDate, LocalDateTime rightFromDate, String opType) {
        int toResult = source.compareTo(rightToDate);
        switch (opType) {
            case "eq":
                return toResult == 0;
            case "gt":
                return toResult > 0;
            case "lt":
                return toResult < 0;
            case "gte":
                return toResult >= 0;
            case "lte":
                return toResult <= 0;
            case "between":
                int fromResult = source.compareTo(rightFromDate);
                return fromResult >= 0 && toResult <= 0;
            default:
                return Boolean.FALSE;
        }
    }

    private Optional<LocalDateTime> parseToLocalDateTime(String source) {
        if (null == source || source.isEmpty()) {
            return Optional.empty();
        }
        DateTimeFormatter formatter = new DateTimeFormatterBuilder()
                .appendOptional(DateTimeFormatter.ofPattern("yyyy-MM-dd[' 'HH:mm:ss]"))
                .appendOptional(DateTimeFormatter.ofPattern("yyyy/MM/dd[' 'HH:mm:ss]"))
                .appendOptional(DateTimeFormatter.ofPattern("dd/MM/yyyy[' 'HH:mm:ss]"))
                .parseDefaulting(java.time.temporal.ChronoField.HOUR_OF_DAY, 0)
                .parseDefaulting(java.time.temporal.ChronoField.MINUTE_OF_HOUR, 0)
                .parseDefaulting(java.time.temporal.ChronoField.SECOND_OF_MINUTE, 0)
                .toFormatter();
        return Optional.of(LocalDateTime.parse(source, formatter));
    }

    private LocalDateTime getOffsetDate(String currentDateType, Integer rightToOffsetValue) {
        if (currentDateType.equals("currentDay")) {
            return LocalDateTime.now().plusDays(rightToOffsetValue);
        }
        return LocalDateTime.now();
    }

    @Override
    public Boolean isBothEqual(Object source, Object targetLeft, Object targetRight, String opType) {
        return Boolean.FALSE;
    }
}
