package com.digiwin.athena.datamap.mechanism.convert.parsers.fieldValueType;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.datamap.mechanism.convert.CustomFuncResult;
import com.digiwin.athena.datamap.mechanism.convert.MechanismParserFactory;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.common.MechanismVariable;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Description 数值类型解析
 * @Author liugl
 * @Date 2024/7/4 17:34
 **/
@Service
@Slf4j
public class NumericParser implements FieldValueTypeParser {

    @Autowired
    MechanismParserFactory mechanismParserFactory;

    @Override
    public boolean accept(String dataType) {
        return MechanismEnum.FieldValueType.NUMERIC.getCode().equalsIgnoreCase(dataType);
    }

    @Override
    public JSONArray parse(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        // 处理全局类型的判断值，例如自定义值，机制函数，枚举值
        CustomFuncResult rightCondResult = mechanismParserFactory.getRightCondParserByType(conditionConfig.getRightCondType())
                .preParse(conditionConfig, metaDataList);
        if (!rightCondResult.getIsNeedNextJudge()) {
            return rightCondResult.getMetaDataList();
        }
        String opType = conditionConfig.getOpType();
        if (MechanismEnum.ConditionType.HIGH_GRADE.getCode().equalsIgnoreCase(conditionConfig.getType())) {
            return parseHighGrade(opType, conditionConfig, metaDataList);
        }
        JSONArray result = new JSONArray();
        BigDecimal rightToValue;
        BigDecimal rightFromValue = null;
        try {
            MechanismVariable rightTo = conditionConfig.getRight().getTo();
            Object rightToValueObj = rightTo.getValue();
            rightToValue = new BigDecimal(rightToValueObj.toString());
            if (opType.equals(MechanismEnum.OperateType.BETWEEN.getCode())) {
                MechanismVariable rightFrom = conditionConfig.getRight().getFrom();
                Object rightFromValueObj = rightFrom.getValue();
                rightFromValue = new BigDecimal(rightFromValueObj.toString());
            }
        } catch (Exception e) {
            log.error("NumericParser.parse 右侧条件判断值转换失败，入参：{}", conditionConfig);
            // 如果条件值不是数值类型或者是空值则不比较，直接返回空数据
            return result;
        }

        String dataName = conditionConfig.getLeft().getData_name();
        result.addAll(parsePrimary(opType, dataName, rightFromValue, rightToValue, metaDataList));
        return result;
    }

    /**
     * 高级条件 自定义函数比较
     *
     * @param opType
     * @param metaDataList
     * @return 返回结果
     */
    private JSONArray parseHighGrade(String opType, MechanismConditionConfig config, JSONArray metaDataList) {
        JSONArray resultArray = new JSONArray();
        Object functionValue = config.getLeft().getFunctionValue();
        try {
            Object fromValue = config.getRight().getFrom().getFunctionValue();
            Object toValue = config.getRight().getTo().getFunctionValue();
            BigDecimal sourceValue = new BigDecimal(functionValue.toString());
            Boolean satisfyCond = isSatisfyCond(opType, sourceValue, fromValue == null ? null : new BigDecimal(fromValue.toString()), new BigDecimal(toValue.toString()));
            if (satisfyCond) {
                resultArray.addAll(metaDataList);
                return resultArray;
            }
        } catch (Exception e) {
            log.error("NumericParser.parse 右侧条件判断值转换失败，入参config：{}", config);
            return null;
        }
        return null;
    }

    /**
     * 执行条件
     *
     * @param opType         操作符
     * @param dataName       字段名称
     * @param rightFromValue 右侧开始值
     * @param rightToValue   右侧截止值
     * @param metaDataList   任务卡表格数据
     * @return 返回满足条件的表格数据
     */
    private JSONArray parsePrimary(String opType, String dataName, BigDecimal rightFromValue, BigDecimal rightToValue, JSONArray metaDataList) {
        if (CollectionUtils.isEmpty(metaDataList)) {
            return new JSONArray();
        }
        List<Object> resultList = metaDataList.stream().filter(v -> {
            JSONObject metaDataObj = JSON.parseObject(JSON.toJSONString(v));
            BigDecimal sourceValue = null;
            try {
                sourceValue = metaDataObj.getBigDecimal(dataName);
            } catch (Exception e) {
                log.warn("NumericParser.parse 表格数据转换失败，数据为：{}", v);
            }
            return isSatisfyCond(opType, sourceValue, rightFromValue, rightToValue);
        }).collect(Collectors.toList());
        return JSONArray.parseArray(JSON.toJSONString(resultList));
    }

    /**
     * 执行操作符逻辑
     *
     * @param opType         操作符
     * @param sourceValue    任务卡表格字段数值
     * @param rightFromValue 右侧比较 开始值
     * @param rightToValue   右侧比较 截至值
     * @return
     */
    public Boolean isSatisfyCond(String opType, BigDecimal sourceValue, BigDecimal rightFromValue, BigDecimal rightToValue) {
        log.info("NumericParser.isSatisfyCond beginning...入参：{}，{}，{}，{}", opType, sourceValue, rightFromValue, rightToValue);
        // 1：大于 0：等于 -1：小于
        if (sourceValue == null) {
            return Boolean.FALSE;
        }
        int toResult = sourceValue.compareTo(rightToValue);
        switch (opType) {
            case "eq":
                return toResult == 0;
            case "gt":
                return toResult == 1;
            case "lt":
                return toResult == -1;
            case "gte":
                return toResult == 1 || toResult == 0;
            case "lte":
                return toResult == -1 || toResult == 0;
            case "between":
                int fromResult = sourceValue.compareTo(rightFromValue);
                // 大于等于开始值 && 小于等于截至值
                return (fromResult == 1 || fromResult == 0) && (toResult == -1 || toResult == 0);
            default:
                return Boolean.FALSE;
        }
    }

    @Override
    public Boolean isBothEqual(Object source, Object targetLeft, Object targetRight, String opType) {
        try {
            if (ObjectUtils.isEmpty(targetRight) || ObjectUtils.isEmpty(source) || ObjectUtils.isEmpty(opType)) {
                return Boolean.FALSE;
            }
            BigDecimal sourceNum = new BigDecimal(source.toString());
            BigDecimal targetLeftNum = null;
            BigDecimal targetRightNum = new BigDecimal(targetRight.toString());
            if (!ObjectUtils.isEmpty(targetLeft)) {
                targetLeftNum = new BigDecimal(targetLeft.toString());
            }
            return isSatisfyCond(opType, sourceNum, targetLeftNum, targetRightNum);
        } catch (Exception e) {
            log.error("NumericParser.isBothEqual 失败，e={}", e);
            return Boolean.FALSE;
        }
    }
}
