package com.digiwin.athena.datamap.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.datamap.dao.DataMapActivityDAO;
import com.digiwin.athena.kg.activity.Activity;
import com.digiwin.athena.datamap.kg.DataMapMongoEntityBaseDAO;
import com.digiwin.athena.datamap.service.IReportConditionService;
import com.digiwin.athena.datamap.service.inner.DataMapKgDataPickService;
import com.digiwin.athena.datamap.service.inner.KmService;
import com.digiwin.athena.datamap.utils.AthenaUtils;
import com.digiwin.athena.kg.dto.queryCondition.QueryConditionSettingResponse;
import com.digiwin.athena.kg.dto.queryCondition.ReportConditionSetting;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import com.digiwin.athena.preset.PresetDomainEnum;
import com.digiwin.athena.preset.Rule;
import com.digiwin.athena.repository.neo4j.ActivityRepository;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lzw
 * @date 2024/7/29
 * @description:
 **/
@Lang
@Service
@Slf4j
@MyExceptionHandler
public class DataMapReportConditionService implements IReportConditionService {

    @Autowired
    KmService kmService;

    @Autowired
    ActivityRepository activityRepository;

    @Autowired
    RuleService ruleService;

    @Autowired
    DataMapTaskService dataMapTaskService;

    @Autowired
    private DatamapAppService applicationService;

    @Autowired
    @Qualifier("kgTenant")
    MongoTemplate mongoTemplateUser;

    @Autowired
    private DataMapKgDataPickService dataPickService;

    @Autowired
    DataMapMongoEntityBaseDAO dataMapMongoEntityBaseDAO;

    @Autowired
    RecordService recordService;

    @Autowired
    private DataMapActivityDAO dataMapActivityDAO;


    @Override
    public Object getQueryConditionSetting(String code) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = kmService.tenantVersion();

        QueryConditionSettingResponse settingResponse = new QueryConditionSettingResponse();
        settingResponse.setTenantId(tenantId);
        settingResponse.setCode(code);

        //查询activity
        Activity activity = dataPickService.chooseOneWithSameCode(dataMapActivityDAO.getActivityByCodeAndVersion(code, tenantVersion, tenantId));
        //校验关联的应用是否过期
        if (null == activity || !activity.getNameSpace().equalsIgnoreCase("common")) {
            this.applicationService.checkRelatedAppHasExpired("activity", code);
        }
        if (null == activity){
            //activity为null，没查到报表数据
            return settingResponse;
        }
        Activity activityPageConfig = recordService.findBetterActivity(tenantId, code, tenantVersion);

        if (null != activityPageConfig) {
            if (null != activityPageConfig.getPages()) {
                activityPageConfig.setPages((Map<String, Object>) activityPageConfig.getPages().get("condition-detail"));
            }
            //处理API信息
//            taskService.setMapProductInfo((Map) activityPageConfig.getDataSources(), tenantId);
//            taskService.setMapProductInfo((Map) activityPageConfig.getPages(), tenantId);
            //基础资料维护没有审核相关

            AthenaUtils.mergeObject(activityPageConfig, activity);
        }
        //解析activity中的FormList,得到所有的formList集合，包含formList里面嵌套的formList
        Map<String, Object> pages = activity.getPages();
        if (null != pages){
            //设置oprations
            if (CollectionUtils.isNotEmpty((List) pages.get("operations"))){
                List<Map<String, Object>> operations = (List<Map<String, Object>>) pages.get("operations");
                settingResponse.setOperations(operations);
            }
            //没有配置过租户级的查询条件配置，组装
            if (!pages.containsKey("layout")){
                settingResponse.setIsDesigner(false);
                return settingResponse;
            }
            List<Map<String, Object>> buildResult = new ArrayList<>();//最终的组装结果
            if (CollectionUtils.isNotEmpty((List) pages.get("layout"))){
                List<Map<String, Object>> layout = (List<Map<String, Object>>) pages.get("layout");
                for (Map<String, Object> map : layout){
                    buildFormList(buildResult,map);
                }
            }
            if (CollectionUtils.isEmpty(buildResult)){
                log.info("build formList is Empty");
                return settingResponse;
            }
            //查询rules，设置rules
            int count = -1;
            Object rulesObj = ruleService.postQuery(code, count);
            if (rulesObj!=null){
                List<Map<String, Object>> rules = new ArrayList<>();
                buildRules(rules,rulesObj);//组装rules
                settingResponse.setRules(rules);
            }
            //获取查询条件设置
            ReportConditionSetting setting = findQueryConditionSetting(tenantId,code,tenantVersion);
            if (null == setting){
                settingResponse.setLayout(buildResult);
                //首次进行查询条件设置，库里没数据
                return settingResponse;
            }
            //设置显隐
            this.setHidden(buildResult,setting.getHiddenData());
            //设置排序，这里的排序指的是每个formList内部的排序
            this.setSort(buildResult,setting.getSortData());
            settingResponse.setLayout(buildResult);

        }

        return settingResponse;
    }

    /**
     * 处理rules，结构和uibot的show接口保持一致
     * @param rules
     * @param rulesObj
     */
    private void buildRules( List<Map<String, Object>> rules,Object rulesObj) {
        Map<String,Object> rulesMap = (Map<String, Object>) rulesObj;
        if (rulesMap != null && rulesMap.get("rules")!=null){
            List<Map<String, Object>> rulesList = (List<Map<String, Object>>) rulesMap.get("rules");
            if (CollectionUtils.isNotEmpty(rulesList)){
                for (Map<String, Object> rule : rulesList) {
                    rule.forEach((k, v) -> {
                        if ("content".equals(k)) {
                            rules.add((Map<String, Object>) v);
                        }
                    });
                }
            }
        }
    }


    /**
     * 保存查询条件设置
     * @param setting
     * @throws Exception
     */
    @Override
    public void postSaveConditionSetting(ReportConditionSetting setting) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = kmService.tenantVersion();
        setting.setVersion(tenantVersion);
        setting.setTenantId(tenantId);
        String code = setting.getCode();
        //生成租户规则对象
        List<Rule> ruleList = buildRuleList(setting);
        //删除knowledgegraph原有的查询条件设置
        Query query = Query.query(Criteria.where("tenantId").is(tenantId).and("code").is(code).and("version").is(tenantVersion));
        mongoTemplateUser.remove(query,"reportConditionSetting");
        //往knowledgegraph写入查询条件设置
        mongoTemplateUser.insert(setting,"reportConditionSetting");
        //删除租户规则库里已存在的自定义规则
        String pluginId = code+"_"+tenantId+"_"+"hiddenCondition";
        ruleService.postRomeveTanentRule(pluginId);
        //往租户规则库里写入自定义规则
        ruleService.postBatchSaveTanentRule(ruleList);
//        return null;
    }

    private List<Rule> buildRuleList(ReportConditionSetting setting) {
        List<Map<String, Object>> hiddenData = setting.getHiddenData();
        if (CollectionUtils.isEmpty(hiddenData)){
            return new ArrayList<>();
        }
        List<Rule> resultList = new ArrayList<>();
        for (Map<String, Object> map : hiddenData){
            if (map.containsKey("hiddenField") && map.get("hiddenField")!=null){
                List<String> hiddenFields = (List<String>) map.get("hiddenField");
                if (CollectionUtils.isNotEmpty(hiddenFields)){
                    for (String str : hiddenFields){
                        String[] strArr = str.split(";");
                        Rule rule = buildRule(setting.getTenantId(),setting.getVersion(),setting.getCode(),strArr);
                        resultList.add(rule);
                    }
                }
            }
        }
        return resultList;
    }

    private Rule buildRule(String tenantId, String version, String code, String[] strArr) {
        Rule rule = new Rule();
        rule.setVersion(version);
        rule.setTenantId(tenantId);
        rule.setDomainId(code);
        rule.setDomain(PresetDomainEnum.ACTIVITY);
        rule.setPluginId(code+"_"+tenantId+"_"+"hiddenCondition");
        JSONObject content = new JSONObject();
        content.put("schema",strArr[1]);
        content.put("path",strArr[0]);
        content.put("linkageSchemas",new JSONArray());
        content.put("key","hidden");
        JSONObject trigger = new JSONObject();
        trigger.put("condition","true");
        trigger.put("point","default");
        content.put("trigger",trigger);
        rule.setContent(content);
        return rule;
    }
    /**
     * 清除查询条件设置
     * @param code
     * @throws Exception
     */
    @Override
    public void postResetConditionSetting(String code) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = kmService.tenantVersion();
        mongoTemplateUser.remove(Query.query(Criteria.where("tenantId").is(tenantId).and("version").is(tenantVersion).and("code").is(code)),"reportConditionSetting");
        //删除租户规则库里已存在的自定义规则
        String pluginId = code+"_"+tenantId+"_"+"hiddenCondition";
        ruleService.postRomeveTanentRule(pluginId);
    }

    private void setSort(List<Map<String, Object>> buildResult, List<Map<String, Object>> sortData) {
        if (CollectionUtils.isEmpty(buildResult) || CollectionUtils.isEmpty(sortData)){
            return;
        }
        for (Map<String, Object> map : buildResult){
            String id = (String) map.getOrDefault("id","");
            Optional<Map<String, Object>> sortItemOpt = sortData.stream()
                    .filter(item -> id.equals(item.get("id")))
                    .findFirst();
            if (sortItemOpt.isPresent() && map.get("group")!=null) {
                List<Map<String, Object>> groupResult = (List<Map<String, Object>>) map.get("group");
                // 如果匹配到sortData对象，根据sortField排序group字段
                Map<String, Integer> sortIndexMap =createSortIndexMap(sortItemOpt.get());
                groupResult.sort(Comparator.comparingInt(item -> {
                    String pathSchema = (String) item.get("path") + ";" + item.get("schema");
                    return sortIndexMap.getOrDefault(pathSchema, 99999); // 不在sortField中的放后面,给与足够大的数值99999，排序时放后面
                }));
            }
        }
    }


    public static Map<String, Integer> createSortIndexMap(Map<String, Object> sortItem) {
        List<String> sortFields = (List<String>) sortItem.get("sortField");
        if (sortFields == null){
            return new HashMap<>();
        }
        //使用 toMap 的重载版本,如果出现键值冲突，合并函数将选择保留现有的值(即)
        return sortFields.stream()
                .collect(Collectors.toMap(
                        field -> field,
                        field -> sortFields.indexOf(field),
                        (existingValue, newValue) -> existingValue // 合并函数，这里简单地选择保留现有值
                ));
    }

    private void setHidden(List<Map<String, Object>> buildResult, List<Map<String, Object>> hiddenData) {
        if (CollectionUtils.isEmpty(buildResult) || CollectionUtils.isEmpty(hiddenData)){
            return;
        }
        Map<String, Object> hiddenDataMap = new HashMap<>();
        // 遍历hiddenData列表并转换
        if (CollectionUtils.isNotEmpty(hiddenData)){
            for (Map<String, Object> entry : hiddenData) {
                String id = (String) entry.get("id");
                List<String> hiddens = (List<String>) entry.get("hiddenField");
                // 将id和hiddenField添加到结果Map中
                hiddenDataMap.put(id, hiddens);
            }
        }
        //找到buildResult中每个formList在hiddenData中有没有设置隐藏的字段，如果有，则设置字段的hidden属性为true
        for (Map<String, Object> map : buildResult){
            String id = (String) map.getOrDefault("id","");
            if (hiddenDataMap.containsKey(id) && CollectionUtils.isNotEmpty((List<String>) hiddenDataMap.get(id)) && map.get("group")!=null){
                List<String> hiddenFieldList = (List<String>) hiddenDataMap.get(id);
                List<Map<String, Object>> groupResult = (List<Map<String, Object>>) map.get("group");
                for (Map<String, Object> groupMap : groupResult){
                    String grouPath = (String) groupMap.getOrDefault("path","");
                    String grouSchema = (String) groupMap.getOrDefault("schema","");
                    if (hiddenFieldList.contains(grouPath+";"+grouSchema)){
                        groupMap.put("hidden",true);
                    }
                }
            }
        }
    }

    private void buildFormList(List<Map<String, Object>> buildResult, Map<String, Object> map) {

        String type = (String) map.getOrDefault("type","");
        String schema = (String) map.getOrDefault("schema","");
        String id = (String) map.getOrDefault("id","");
        if ("FORM_LIST".equals(type)){
            Map<String, Object> result = new HashMap<>();
            List<Map<String, Object>> groupResult = new ArrayList<>();
            if (null != map.get("group") && CollectionUtils.isNotEmpty((List) map.get("group"))) {
                List<Map<String, Object>> group = (List<Map<String, Object>>) map.get("group");
                for (Map<String, Object> groupMap : group){
                    String groupType = (String) groupMap.getOrDefault("type","");
                    if (!"FORM_LIST".equals(groupType)){

                        Map<String, Object> resultMap = buildMap(groupMap);
                        groupResult.add(resultMap);
                    }else {

                        buildFormList(buildResult,groupMap);

                    }
                }
            }
            if (CollectionUtils.isNotEmpty(groupResult)){
                result.put("schema",schema);
                result.put("id",id);
                result.put("group",groupResult);
                buildResult.add(result);
            }

        }

    }

    private Map<String, Object> buildMap(Map<String, Object> groupMap) {
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("schema", groupMap.getOrDefault("schema",""));
        resultMap.put("path", groupMap.getOrDefault("path",""));
        resultMap.put("headerName", groupMap.getOrDefault("headerName",""));
        resultMap.put("placeholder", groupMap.getOrDefault("placeholder",""));
        resultMap.put("type", groupMap.getOrDefault("type",""));
        resultMap.put("hidden",false);
        return resultMap;
    }

    /**
     * 查询租户级的查询条件设置
     * @param tenantId
     * @param activityCode
     * @param activityVersion
     * @return
     * @throws DWBusinessException
     */
    public ReportConditionSetting findQueryConditionSetting(String tenantId, String activityCode, String activityVersion) throws DWBusinessException {
        Query query = Query.query(Criteria.where("code").is(activityCode).and("version").is(activityVersion).and("tenantId").is(tenantId));
        ReportConditionSetting setting = mongoTemplateUser.findOne(query
                , ReportConditionSetting.class,"reportConditionSetting");
        return setting;
    }

}
