package com.digiwin.athena.datamap.service.impl;

import cn.hutool.core.collection.ListUtil;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.datamap.povo.*;
import com.digiwin.athena.datamap.service.IBatchService;
import com.digiwin.athena.datamap.supports.EocInfoLocal;
import com.digiwin.athena.domain.core.Activity;
import com.digiwin.athena.domain.core.Project;
import com.digiwin.athena.domain.core.Task;
import com.digiwin.athena.domain.core.flow.FlowGraph;
import com.digiwin.athena.domain.core.flow.FlowNode;
import com.digiwin.athena.domain.core.view.PageView;
import com.digiwin.athena.domain.definition.times.ExpectedDuration;
import com.digiwin.athena.dto.BasicQuery;
import com.digiwin.athena.dto.SortField;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.service.DataPickService;
import com.digiwin.athena.kmservice.support.DapContext;
import com.digiwin.athena.kmservice.utils.MergeUtil;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import io.seata.common.util.CollectionUtils;
import org.bson.types.ObjectId;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import java.util.*;
import java.util.concurrent.*;

/**
 * 批量 Service
 **/
@Lang
@MyExceptionHandler
@Service
public class DatamapBatchService implements IBatchService {

    @Autowired
    private DataPickService dataPickService;

    @Autowired
    @Qualifier("dataMapSystem")
    private MongoTemplate mongoTemplate;

    private final ThreadPoolExecutor batchPool =
            new ThreadPoolExecutor(Runtime.getRuntime().availableProcessors() * 2, 200, 60L, TimeUnit.SECONDS, new ArrayBlockingQueue<>(10), new ThreadPoolExecutor.CallerRunsPolicy());

    @Override
    public List<Project> postProjectBasic(BatchRequest q) throws DWBusinessException {
        if (null != q.getCodes() && !q.getCodes().isEmpty()) {
            BasicQuery basicQuery = new BasicQuery();
            Map<String, Object> map = new HashMap<>();
            map.put("code", q.getCodes());
            basicQuery.setQuery(map);
            basicQuery.setEocInfo(q.getEocInfo());
            List<Project> projects = dataPickService.find(basicQuery, Project.class);

            projects.forEach(p -> p.setTaskId(p.getCode()));
            return projects;
        }
        return null;
    }

    @Override
    public List<Task> postTaskBasic(BatchRequest q) throws DWBusinessException {
        if (null != q.getCodes() && !q.getCodes().isEmpty()) {
            BasicQuery basicQuery = new BasicQuery();
            Map<String, Object> map = new HashMap<>();
            map.put("code", q.getCodes());
            basicQuery.setQuery(map);
            basicQuery.setEocInfo(q.getEocInfo());
            List<Task> tasks = dataPickService.find(basicQuery, Task.class);
            for (Task task : tasks) {
                task.setActivityId(task.getCode());
            }
            return tasks;
        }
        return null;
    }

    @Override
    public CommonResp postProjects(CommonReq q) throws DWBusinessException {
        List<Project> projects = dataPickService.find(convertQuery(q), Project.class);
        CommonResp resp = new CommonResp();
        resp.setDatas(projects);
        return resp;
    }

    private BasicQuery convertQuery(CommonReq q) {
        BasicQuery basicQuery = new BasicQuery();
        basicQuery.setPage(q.getPageIndex());
        basicQuery.setPageSize(q.getPageSize());
        basicQuery.setQuery(q.getQuery());
        if (null != q.getSortField()) {
            String order = q.getSortOrder();
            if (null == order) {
                order = "asc";
            }
            basicQuery.setSortFields(new ArrayList<>());
            basicQuery.getSortFields().add(SortField.of(q.getSortField(), order));
        }
        return basicQuery;
    }

    @Override
    public CommonResp postTasks(CommonReq q) throws DWBusinessException {
        List<Task> projects = dataPickService.find(convertQuery(q), Task.class);
        CommonResp resp = new CommonResp();
        resp.setDatas(projects);
        return resp;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public CommonResp postQueryCodeAndName(List<String> codes, String type, String version) {
        CommonResp resp = new CommonResp();

        Query query = new Query();
        query.fields().include("name").include("code").include("lang.name");
        query.addCriteria(Criteria.where("code").in(codes).and("version").is(version));

        List<Map> maps = this.mongoTemplate.find(query, Map.class, type);
        resp.setDatas(maps);

        return resp;
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    @Override
    public CommonResp postMongo(MongoCrudReq q) {

        CommonResp obj = new CommonResp();

        Query query1 = new Query();
        q.getQuery().forEach((k, v) -> {
            if (v instanceof Collection) {
                Collection collection = (Collection) v;
                List list = new ArrayList(collection);
                query1.addCriteria(Criteria.where(k).in(list));
            } else {
                query1.addCriteria(Criteria.where(k).is(v));
            }
        });

        switch (q.getOp()) {
            case "select":
                List<Map> list = mongoTemplate.find(query1, Map.class, q.getCol());
                for (Map doc : list) {
                    try {
                        ObjectId objectId = (ObjectId) doc.get("_id");
                        doc.put("_id", objectId.toHexString());
                    } catch (Exception e) {
                        // ignore
                    }
                }
                obj.setData(list);
                break;
            case "insert":
                mongoTemplate.insert(q.getDocs(), q.getCol());
                break;
            case "save":
                Map doc = (Map) q.getDoc();
                String _id = (String) doc.get("_id");
                if (null != _id) {
                    ObjectId objectId = new ObjectId(_id);
                    doc.put("_id", objectId);

                }
                mongoTemplate.save(q.getDoc(), q.getCol());
                break;
            case "delete":
                // mongoTemplate.remove(query1,q.getCol());
                break;
            default:
                break;
        }

        return obj;
    }

    @Override
    public Map<String, Map<String, Object>> postTaskDetails(BatchRequest q) throws DWBusinessException {
        Map<String, Map<String, Object>> result = new HashMap<>();
        if (CollectionUtils.isEmpty(q.getCodes())) {
            return result;
        }
        EocInfoLocal.setEocInfo(q.getEocInfo());
        int batchSize = 10;
        if (q.getCodes().size() < batchSize * 1.5) {
            return doPostTaskDetails(q);
        }
        List<List<String>> codesArr = ListUtil.split(q.getCodes(), batchSize);
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        DWServiceContext context = DWServiceContext.getContext();
        DapContext dapContext = ServiceUtils.getContext();
        CompletableFuture
                .allOf(
                        codesArr.stream().map(codes -> CompletableFuture.runAsync(() -> {
                            DWServiceContext.setContext(context);
                            ServiceUtils.setContext(dapContext);
                            try {
                                EocInfoLocal.setEocInfo(q.getEocInfo());
                                RequestContextHolder.setRequestAttributes(requestAttributes);
                                BatchRequest q2 = new BatchRequest();
                                q2.setCodes(codes);
                                q2.setEocInfo(q.getEocInfo());
                                Map<String, Map<String, Object>> data = doPostTaskDetails(q2);
                                result.putAll(data);
                            } catch (DWBusinessException e) {
                                throw new RuntimeException(e);
                            } finally {
                                RequestContextHolder.resetRequestAttributes();
                            }
                        }, batchPool))
                                .toArray(CompletableFuture[]::new))
                .join();
        return result;
    }

    public Map<String, Map<String, Object>> doPostTaskDetails(BatchRequest q) throws DWBusinessException {
        Map<String, Map<String, Object>> taskMap = new HashMap<>();
        if (null != q.getCodes() && !q.getCodes().isEmpty()) {

            Map<String, Object> map = new HashMap<>();
            map.put("code", q.getCodes());
            BasicQuery basicQuery1 = BasicQuery.of(map, q.getEocInfo());
            List<Task> tasks = dataPickService.find(basicQuery1, Task.class);
            Map<String, String> pageCodes = new HashMap<>();
            Map<String, String> flowCodes = new HashMap<>();
            for (Task task : tasks) {
                task.setActivityId(task.getCode());
                taskMap.put(task.getCode(), DataMapViewService.taskToMap(task));
                if (null != task.getPageCode()) {
                    pageCodes.put(task.getPageCode(), task.getCode());
                }
                if (null != task.getFlowCode()) {
                    flowCodes.put(task.getFlowCode(), task.getCode());
                }
            }

            // flow
            Map<String, Activity> taskActivity = new HashMap<>();
            if (!flowCodes.isEmpty()) {
                Map<String, List<String>> taskActivityCodes = new HashMap<>();
                Map<String, Object> map2 = new HashMap<>();
                map2.put("code", flowCodes.keySet());
                BasicQuery basicQuery2 = BasicQuery.of(map2, q.getEocInfo());
                List<FlowGraph> flows = dataPickService.find(basicQuery2, FlowGraph.class);
                List<String> allCodes = new ArrayList<>();
                for (FlowGraph flow : flows) {
                    String taskCode = flowCodes.get(flow.getCode());
                    List<String> codes = new ArrayList<>();
                    for (FlowNode node : flow.getNodes()) {
                        if ("activity".equalsIgnoreCase(node.getType())) {
                            codes.add(node.getActivityCode());
                            allCodes.add(node.getActivityCode());
                        }
                    }
                    taskActivityCodes.put(taskCode, codes);
                }
                if (!allCodes.isEmpty()) {
                    Map<String, Object> q2 = new HashMap<>();
                    q2.put("code", allCodes);
                    q2.put("milestone", true);
                    List<Activity> activities = dataPickService.find(BasicQuery.of(q2, q.getEocInfo()), Activity.class);
                    for (Activity activity : activities) {
                        for (Map.Entry<String, List<String>> en : taskActivityCodes.entrySet()) {
                            if (en.getValue().contains(activity.getCode())) {
                                taskActivity.put(en.getKey(), activity);
                                break;
                            }
                        }
                    }
                }
            }
            // pageview
            Map<String, PageView> taskPageView = new HashMap<>();
            if (!pageCodes.isEmpty()) {
                Map<String, Object> map2 = new HashMap<>();
                map2.put("code", pageCodes.keySet());
                BasicQuery basicQuery2 = BasicQuery.of(map2, q.getEocInfo());
                List<PageView> pageViews = dataPickService.find(basicQuery2, PageView.class);
                for (PageView pageView : pageViews) {
                    String taskCode = pageCodes.get(pageView.getCode());
                    taskPageView.put(taskCode, pageView);
                }
            }
            for (Task task : tasks) {
                Map<String, Object> viewMap = taskMap.get(task.getCode());
                PageView page = taskPageView.get(task.getCode());
                if (null != page) {
                    if (null != page.getLang() && null != task.getLang()) {
                        page.getLang().putAll(task.getLang());
                    }
                    MergeUtil.mergeObjectToMap(viewMap, page, ListUtil.of("id", "name", "code"));
                    if (null != q.getPageCode() && null != page.getPages()) {
                        viewMap.put("pages", page.getPages().get(q.getPageCode()));
                    }
                }
                Activity activity = taskActivity.get(task.getCode());
                if (null != activity) {
                    if (null != activity.getConfig()) {
                        viewMap.put("approve", activity.getConfig().get("approve"));
                        viewMap.put("dynamicApproves", activity.getConfig().get("dynamicApproves"));
                        viewMap.put("approveLevel", activity.getConfig().get("approveLevel"));

                    }
                    if (null != activity.getExecutor()) {
                        viewMap.put("personOnDuty", activity.getExecutor());
                    }
                    if (null != activity.getPresetVariables()) {
                        viewMap.put("presetVariables", activity.getPresetVariables());
                    }
                    viewMap.put("executorMeta", activity.getExecutorMeta());
                }
                // 为了兼容两种引擎的逾期时间配置
                if (null != task.getDueDate() && null == task.getExpectedDuration()) {
                    if (task.getDueDate().getValue() instanceof Number) {
                        Number seconds = (Number) task.getDueDate().getValue();
                        ExpectedDuration expectedDuration = new ExpectedDuration();
                        int value = seconds.intValue() / 86400;
                        int mod = seconds.intValue() % 86400;
                        if (value > 0 && mod == 0) {
                            expectedDuration.setType("DAY");
                        } else {
                            value = seconds.intValue() / 3600;
                            mod = seconds.intValue() % 3600;
                            if (value > 0 && mod == 0) {
                                expectedDuration.setType("HOUR");
                            } else {
                                value = seconds.intValue() / 60;
                                expectedDuration.setType("MINUTE");
                            }
                        }
                        expectedDuration.setValue(value);
                        task.setExpectedDuration(expectedDuration);
                        viewMap.put("expectedDuration", expectedDuration);
                    }
                }
            }

        }
        return taskMap;
    }

    public static void main(String[] args) {
        List<String> list = Arrays.asList("task_ba44810f4a4cb5d968e72d4386a9dab8", "task_bcce6bc17969c8e99bc1bda92fcff9c2", "task_d0f5fb44a5fd632418a5f673a14a8c39");
        System.out.println(ListUtil.split(list, 10));

    }
}
