package com.digiwin.athena.datamap.service.impl;

import com.digiwin.app.container.exceptions.DWArgumentException;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.datamap.service.IPresetService;
import com.digiwin.athena.datamap.spi.DbPresetService;
import com.digiwin.athena.kg.ComponentConstants;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.utils.I18nUtils;
import com.digiwin.athena.preset.dto.PresetDTO;
import com.mongodb.BasicDBObject;
import com.mongodb.client.MongoCollection;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import static com.mongodb.client.model.Filters.and;
import static com.mongodb.client.model.Filters.eq;
import static com.mongodb.client.model.Projections.excludeId;

@Lang
@Service
@MyExceptionHandler
public class PresetService implements IPresetService {

    @Autowired
    GenericService genericService;

    @Autowired
    private DbPresetService dbPresetService;

    @Override
    public void put(String colName, PresetDTO content) throws Exception {
        if (StringUtils.isEmpty(colName)) {
            throw new DWArgumentException("colName", I18nUtils.getValue("preset.colNameEmpty"));
        }

        Map<String, Object> preset = new HashMap<>();
        String key = content.getKey();
        if (StringUtils.isEmpty(key)) {
            throw new DWArgumentException("content", I18nUtils.getValue("preset.contentKeyEmpty"));
        }
        preset.put("key", key);
        preset.put("tenantId", content.getTenantId());
        preset.put("content", content.getContent());
        preset.put("name", content.getName());
        preset.put("remark", content.getRemark());
        preset.put("createBy", content.getCreateBy());
        preset.put("createDate", content.getCreateDate());
        preset.put("modifyBy", content.getModifyBy());
        preset.put("modifyDate", content.getModifyDate());
        if (content.getDomain() != null) {
            preset.put("domain", content.getDomain().toString());
        }
        preset.put("domainId", content.getDomainId());
        preset.put("version", content.getVersion());
        try {
            MongoCollection<Document> col = dbPresetService.getCollection(colName, "key", "version");
            col.insertOne(new Document(preset));
            genericService.delMatch(ComponentConstants.REDIS_KNOWLEDGE_GRAPH_PRESET  + "*");
        } catch (Exception e) {
            throw new DWBusinessException("P.PL.500.0004", e.getLocalizedMessage());
        }
    }

    @Override
    public void delete(String colName, String key, String tenantVersion) throws Exception {
        MongoCollection<Document> col = dbPresetService.getCollection(colName, "key", "version");
        col.deleteOne(and(eq("key", key), eq("version", tenantVersion)));
        genericService.delMatch(ComponentConstants.REDIS_KNOWLEDGE_GRAPH_PRESET  + "*");
    }

    @Override
    public Object get(String colName, String key) throws Exception {
        Map<String, Object> re = new HashMap<>();
        MongoCollection<Document> col = dbPresetService.getCollection(colName, "key", "version");
        Document doc = col.find(eq("key", key)).projection(excludeId()).first();
        if (doc != null) {
            re = doc;
        }
        return re;
    }

    @Override
    public Object getQuery(String colName, Map<String, Object> conditions) throws Exception {
        Map<String, Object> re = new HashMap<>();
        BasicDBObject sortObj = new BasicDBObject();
        sortObj.put("createDate", -1);//按创建时间倒序排序 返回最新的一笔
        Document doc;

        MongoCollection<Document> col = dbPresetService.getCollection(colName, "key", "version");
        if (CollectionUtils.isEmpty(conditions)) {
            doc = col.find().projection(excludeId()).sort(sortObj).first();
        } else {
            //目前仅实现 and 连接条件
            ArrayList<Bson> con = new ArrayList<>();
            for (Map.Entry<String, Object> entry : conditions.entrySet()) {
                con.add(eq(entry.getKey(), entry.getValue()));
            }
            doc = col.find(and(con.toArray(new Bson[0]))).projection(excludeId()).sort(sortObj).first();
        }

        if (doc != null) {
            re = doc;
        }
        return re;
    }

    @Override
    public void deleteTanentRule(String colName, String pluginId, String tenantVersion) throws Exception {
        MongoCollection<Document> col = dbPresetService.getCollection(colName, "pluginId", "version");
        col.deleteMany(and(eq("pluginId", pluginId), eq("version", tenantVersion)));
        genericService.delMatch(ComponentConstants.REDIS_KNOWLEDGE_GRAPH_PRESET  + "*");
    }
}
