package com.digiwin.athena.datamap.utils;

import lombok.extern.slf4j.Slf4j;

import java.text.ParseException;

/**
 * 
 * @author Administrator
 * 
 * 
 * Twitter的分布式自增ID算法snowflake
 * 
 * 冷知识： 一个时间戳大概占41位,所以时间戳向左移动22位共占41+22=63，留1最高位保持为0.
 * 0 时间戳(41) 数据表示id(5) 机器标识id(5) senqunceid(12)  1+41+5+5+12=64   2的8次方 Long
 * 支持32*32个节点，每个节点每毫秒支持4096个数据（溢出会阻塞）
 * 
 * 
 *
 */

@Slf4j
public class IdWorker {

    // ==============================Fields===========================================
    /** 开始时间截 (2018-01-01) */
    private final long twepoch = 1514736000000L;

    /** 机器id所占的位数 */
    private final long workerIdBits = 5L;

    /** 数据标识id所占的位数 */
    private final long datacenterIdBits = 5L;

    /** 支持的最大机器id，结果是31 (这个移位算法可以很快的计算出几位二进制数所能表示的最大十进制数) */
    private final long maxWorkerId = -1L ^ (-1L << workerIdBits);

    /** 支持的最大数据标识id，结果是31 */
    private final long maxDatacenterId = -1L ^ (-1L << datacenterIdBits);

    /** 序列在id中占的位数 */
    private final long sequenceBits = 12L;

    private final long sequenceBits2 = 8L;

    private final long sequenceBits3 = 4L;

    /** 机器ID向左移12位 */
    private final long workerIdShift = sequenceBits;

    /** 机器ID向左移8位 */
    private final long workerIdShift2 = sequenceBits2;

    /** 数据标识id向左移17位(12+5) */
    private final long datacenterIdShift = sequenceBits + workerIdBits;

    /** 时间截向左移22位(5+5+12) */
    private final long timestampLeftShift = sequenceBits + workerIdBits + datacenterIdBits;

    /** 时间截向左移13位(5+8) */
    private final long timestampLeftShift2 = sequenceBits2 + workerIdBits;

    private final long timestampLeftShift3 = sequenceBits3;

    /** 生成序列的掩码，这里为4095 (0b111111111111=0xfff=4095) */
    private final long sequenceMask = -1L ^ (-1L << sequenceBits);

    /** 生成序列的掩码，这里为255 (0b1111111111=0xff=255) */
    private final long sequenceMask2 = -1L ^ (-1L << sequenceBits2);

    /** 生成序列的掩码，这里为255 (0b1111111111=0xf=15) */
    private final long sequenceMask3 = -1L ^ (-1L << sequenceBits3);

    /** 工作机器ID(0~31) */
    private long workerId;

    /** 数据中心ID(0~31) */
    private long datacenterId;

    /** 毫秒内序列(0~4095) */
    private long sequence = 0L;

    /** 毫秒内序列(0~255) */
    private long sequence2 = 0L;

    private long sequence3 = 0L;

    /** 上次生成ID的时间截 */
    private long lastTimestamp = -1L;

    private long lastTimestamp2 = -1L;

    private long lastTimestamp3 = -1L;

    // ==============================Constructors=====================================

    public IdWorker() {
        this.workerId = 0;
        this.datacenterId = 0;
    }
    /**
     * 构造函数
     * @param workerId 工作ID (0~31)
     * @param datacenterId 数据中心ID (0~31)
     */
    public IdWorker(long workerId, long datacenterId) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
        }
        if (datacenterId > maxDatacenterId || datacenterId < 0) {
            throw new IllegalArgumentException(String.format("datacenter Id can't be greater than %d or less than 0", maxDatacenterId));
        }
        this.workerId = workerId;
        this.datacenterId = datacenterId;
    }

    // ==============================Methods==========================================
    /**
     * 获得下一个ID (该方法是线程安全的)
     * @return SnowflakeId
     */
    public synchronized long nextId() {
        long timestamp = timeGen();

        // 如果当前时间小于上一次ID生成的时间戳，说明系统时钟回退过这个时候应当抛出异常
        if (timestamp < lastTimestamp) {
            throw new RuntimeException(
                    String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp - timestamp));
        }

        // 如果是同一时间生成的，则进行毫秒内序列
        if (lastTimestamp == timestamp) {
            sequence = (sequence + 1) & sequenceMask;
            // 毫秒内序列溢出
            if (sequence == 0) {
                // 阻塞到下一个毫秒,获得新的时间戳
                timestamp = tilNextMillis(lastTimestamp);
            }
            // 时间戳改变，毫秒内序列重置
        } else {
            sequence = 0L;
        }

        // 上次生成ID的时间截
        lastTimestamp = timestamp;

        // 移位并通过或运算拼到一起组成64位的ID
        return ((timestamp - twepoch) << timestampLeftShift) | (datacenterId << datacenterIdShift) | (workerId << workerIdShift) | sequence;
    }

    /**
     * 仅支持32节点，每毫秒支持255个id生成
     *
     * @return
     */
    public synchronized long nextMiniId() {
        long timestamp = timeGen();

        // 如果当前时间小于上一次ID生成的时间戳，说明系统时钟回退过这个时候应当抛出异常
        if (timestamp < lastTimestamp2) {
            throw new RuntimeException(
                    String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp2 - timestamp));
        }

        // 如果是同一时间生成的，则进行毫秒内序列
        if (lastTimestamp2 == timestamp) {
            sequence2 = (sequence2 + 1) & sequenceMask2;
            // 毫秒内序列溢出
            if (sequence2 == 0) {
                // 阻塞到下一个毫秒,获得新的时间戳
                timestamp = tilNextMillis(lastTimestamp2);
            }
            // 时间戳改变，毫秒内序列重置
        } else {
            sequence2 = 0L;
        }

        // 上次生成ID的时间截
        lastTimestamp2 = timestamp;

        // 移位并通过或运算拼到一起组成64位的ID
        return ((timestamp - twepoch) << timestampLeftShift2) | (workerId << workerIdShift2) | sequence2;
    }

    public String nextMiniHex() {
        return Long.toHexString(nextMiniId()).toUpperCase();
    }

    /*
     * 使用(System.currentTimeMillis()-twepoch)/1000 表示从twepoch开始到现在的秒数，仅够支持68年,每秒支持最多一条数据
     *
     */
    public synchronized int nextIntId() {
        long timestamp = timeGen();

        if (timestamp < lastTimestamp3) {
            throw new RuntimeException(
                    String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp3 - timestamp));
        }

        int result = (int) ((timestamp - twepoch) / 1000);

        while (result == lastTimestamp3) {

            try {
                // 节省cpu性能
                Thread.sleep(501);
            } catch (InterruptedException e) {
                // TODO Auto-generated catch block
                log.error(e.toString());
            }

            timestamp = timeGen();
            result = (int) ((timestamp - twepoch) / 1000);
        }
        lastTimestamp3 = result;

        return result;
    }

    /**
     * 阻塞到下一个毫秒，直到获得新的时间戳
     * @param lastTimestamp 上次生成ID的时间截
     * @return 当前时间戳
     */
    protected long tilNextMillis(long lastTimestamp) {
        long timestamp = timeGen();
        while (timestamp <= lastTimestamp) {
            timestamp = timeGen();
        }
        return timestamp;
    }

    /**
     * 返回以毫秒为单位的当前时间
     * @return 当前时间(毫秒)
     */
    protected long timeGen() {
        return System.currentTimeMillis();
    }

    public static void main(String[] args) throws InterruptedException, ParseException {
        // TODO Auto-generated method stub
        // SimpleDateFormat sdf = new SimpleDateFormat("yyyy");
        //
        // long t2=sdf.parse("2018").getTime();
        // System.out.println(t2);

        IdWorker idw = new IdWorker(0, 0);

        System.out.println(idw.nextMiniHex());

    }

}
