package com.digiwin.athena.datamap.mechanism.convert;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.datamap.utils.DataMapUtils;
import com.digiwin.athena.mechanism.bo.HookBO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @Description
 * @Author liugl
 * @Date 2024/6/28 9:54
 **/
@Service
@Slf4j
public class HooksConverter {

    @Autowired
    JSFunctions jsFunctions;

    /**
     * 动态函数  需要替换其中的值
     *
     * @param function
     * @return
     */
    public HookBO hookDynamicFunctionExpression(HookDynamicFunction function) {
        StringBuilder sb = new StringBuilder();
        String name = function.getFunctionName();
        String fun = jsFunctions.function(name);
        if (null != fun) {
            Map<String, Object> parameter = function.getParameter();
            if (!ObjectUtils.isEmpty(parameter)) {
                for (Map.Entry<String, Object> entry : parameter.entrySet()) {
                    String key = entry.getKey();
                    if (entry.getValue() instanceof String) {
                        fun = fun.replaceAll(key, entry.getValue().toString());
                    } else {
                        fun = fun.replaceAll(key, JSON.toJSONString(entry.getValue()));
                    }
                }
            }
            sb.append(fun).append(";");
        }
        HookBO hookBO = new HookBO().setEventSource(function.getSourceId());
        String hookName = function.getHookName();
        try {
            Field field = HookBO.class.getDeclaredField(hookName);
            field.setAccessible(true);
            field.set(hookBO, fun);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return hookBO;
    }

    /**
     * 构建hook
     *
     * @param hooksInfo
     * @return
     */
    public List<HookBO> buildHooks(HooksInfo hooksInfo) {
        List<HookBO> hookBOs = new ArrayList<>();
        // step1:处理外层共有的hook
        if (!ObjectUtils.isEmpty(hooksInfo.getHook())) {
            hookBOs.add(hooksInfo.getHook());
        }
        // step2:处理不同类型独有的动态hook
        if (!ObjectUtils.isEmpty(hooksInfo.getDynamicFunctions())) {
            hooksInfo.getDynamicFunctions().stream().forEach(v -> {
                hookBOs.add(hookDynamicFunctionExpression(v));
            });
        }
        List<HookBO> mergeHooks = new ArrayList<>();
        hookBOs.stream().forEach(v -> {
            mergeHook(mergeHooks, v);
        });
        return mergeHooks;
    }

    /**
     * 将新旧hooks对象数据合并
     *
     * @param mergeHooks 数据对象
     * @param hookBO     新hooks数据对象
     */
    public List<HookBO> mergeHook(List<HookBO> mergeHooks, HookBO hookBO) {
        // Step1：旧数据没有hooks，直接添加hooks
        if (ObjectUtils.isEmpty(mergeHooks)) {
            mergeHooks.add(hookBO);
            return mergeHooks;
        }
        // Step2：旧数据有hooks，查询是否有相同eventSource的数据
        Optional<HookBO> oldEventHookOptional = mergeHooks.stream().filter(v -> {
            if (hookBO.getEventSource().equals(v.getEventSource())) {
                return true;
            }
            return false;
        }).findFirst();
        // step2.1: 没有相同eventSource的数据，直接将hooks对象存入
        if (!oldEventHookOptional.isPresent()) {
            mergeHooks.add(hookBO);
            return mergeHooks;
        }
        // step2.2：存在相同eventSource数据时合并数据， 只合并有Hook的字段
        DataMapUtils.mergeObject(oldEventHookOptional.get(), hookBO, HookBO.class, "^.*Hook", Boolean.TRUE);
        return mergeHooks;
    }
}
