package com.digiwin.athena.datamap.mechanism.convert.parsers.judgeBasis;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.datamap.mechanism.convert.HookDynamicFunction;
import com.digiwin.athena.datamap.mechanism.convert.HooksConverter;
import com.digiwin.athena.datamap.mechanism.convert.HooksInfo;
import com.digiwin.athena.datamap.mechanism.convert.MechanismParserFactory;
import com.digiwin.athena.datamap.mechanism.convert.parsers.hooks.HooksParser;
import com.digiwin.athena.datamap.mechanism.support.BindingContext;
import com.digiwin.athena.datamap.utils.DataMapUtils;
import com.digiwin.athena.domain.core.view.PageView;
import com.digiwin.athena.mechanism.bo.DecisionAbilityBO;
import com.digiwin.athena.mechanism.common.MechanismAbility;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.widgets.config.DecisionScene;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import com.google.common.collect.Lists;
import com.jayway.jsonpath.JsonPath;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
@Slf4j
public class ApiParser implements JudgeBasisParser {

    @Autowired
    HooksConverter hooksConverter;

    @Autowired
    MechanismParserFactory mechanismParserFactory;

    @Override
    public boolean accept(String type) {
        return MechanismEnum.ConditionType.API.getCode().equalsIgnoreCase(type);
    }

    @Override
    public void parse(MechanismAbility ability, PageView pageView, BindingContext context, HooksInfo hooksInfoAll) {
        DecisionAbilityBO bo = (DecisionAbilityBO) ability;
        context.getMechanismVariables().put("hookSelectorInitDataFieldParam", bo.getVariable().getData_name());
        // API模式：一个场景只有一个条件组，每组里面只有一个API条件
        bo.getDecisionOption().getScenes().stream().forEach(v -> {
            try {
                // step1：解析场景数据
                parseScene(v, context, pageView, hooksInfoAll);
            } catch (DWException e) {
                // 只记录异常信息，不外抛
                log.error("ApiParser.parse 解析场景判断条件失败，请求参数ability：{}，pageView：{}，context：{}，异常：{}", v, pageView, context, e);
            }
        });
        // step2：封装hookInfo数据
        packageHookInfo(pageView, bo, hooksInfoAll);
    }

    /**
     * 封装hookInfo数据
     * @param pageView
     * @param bo
     * @param hooksInfoAll
     */
    private void packageHookInfo(PageView pageView, DecisionAbilityBO bo, HooksInfo hooksInfoAll) {
        String layoutId = DataMapUtils.getTableLayoutId(pageView);
        // 设定调用api的hooks
        HashMap<String, Object> callApiParams = new HashMap<>();
        callApiParams.put("hookCallApiAppType", MechanismEnum.ConditionType.HALF_CUSTOM.getCode());
        callApiParams.put("hookCallApiFieldParam", bo.getVariable().getData_name());
        callApiParams.put("hookCallApiFieldValueParam", hooksInfoAll.getTargetMap());
        callApiParams.put("hookCallApiApiListParam", hooksInfoAll.getApiList());

        hooksInfoAll.getDynamicFunctions().add(new HookDynamicFunction()
                .setFunctionName("hookCallApi")
                .setHookName("gridReadyHook")
                .setSourceId(layoutId)
                .setParameter(callApiParams));
        hooksInfoAll.getDynamicFunctions().add(new HookDynamicFunction()
                .setFunctionName("hookCallParamPrepare")
                .setHookName("afterViewInitHook")
                .setSourceId(layoutId)
                .setParameter(callApiParams));
    }

    /**
     * 解析机制条件
     *
     * @param context
     * @param scene
     * @param pageView
     * @param hooksInfoAll
     * @return
     * @throws DWException
     */
    private void parseScene(DecisionScene scene, BindingContext context, PageView pageView, HooksInfo hooksInfoAll) throws DWException {
        if (!ObjectUtils.isEmpty(scene)) {
            Map map = JSON.parseObject(JSON.toJSONString(scene));
            List<Object> targetList = new ArrayList<>();

            // api条件限定只有一个
            MechanismConditionConfig mechanismCondition;
            try {
                mechanismCondition = scene.getCheck().getGroups().get(0).getConditions().get(0);
            } catch (Exception e) {
                log.error("解析pageView获取决策条件失败，参数scenes：{}", scene);
                throw new DWException("解析pageView获取决策条件失败");
            }
            // step1：处理每个类型特有的hooks
            HooksParser hooksParser = mechanismParserFactory.getHooksParserByConfig(mechanismCondition);
            if (null == hooksParser) {
                log.error("ApiParser.parseScene 获取HooksParser失败，入参：{}", scene);
            }
            HooksInfo hooksInfo = hooksParser.parse(mechanismCondition, context, pageView, null);
            hooksInfoAll.getDynamicFunctions().addAll(hooksInfo.getDynamicFunctions());
            try {
                // step2：处理api hook
                JSONArray mechanismApiInfoArray = JSON.parseArray(JSON.toJSONString(JsonPath.read(map, "$.check.groups[*].conditions[*].mechanismApiInfo")));
                JSONObject mechanismApiInfo = (JSONObject) mechanismApiInfoArray.get(0);

                String apiUrl = mechanismApiInfo.getString("serviceId");
                String hasSchema = mechanismApiInfo.getString("hasScheme");
                JSONObject requestJson = mechanismApiInfo.getJSONObject("requestJson");

                Map<String, Object> apiInfo = new HashMap();
                Map<String, Object> paramsInfo = new HashMap();
                Map<String, Object> req = new HashMap();
                paramsInfo.put("api", apiUrl);
                paramsInfo.put("requestJson", new Object());
                // 外部api需要Esp，因此设置为true
                paramsInfo.put("isNeedEsp", Boolean.TRUE);
                // 半定制默认唯一主键为bk_info
                paramsInfo.put("keyFieldNameList", Lists.newArrayList("bk_info"));
                req.put("req", paramsInfo);
                apiInfo.put("params", req);

                if (null != requestJson) {
                    log.info("HooksConverter.convert requestJson:{}", requestJson);
                    paramsInfo.put("requestJson", requestJson);
                }
                // 判断是否有方案，有方案走api内部方案，没有方案走页面填写的target
                if ("true".equalsIgnoreCase(StrUtil.toString(hasSchema))) {
                    apiInfo.put("type", "api");
                } else {
                    apiInfo.put("type", "target");
                }
                Object targets = JsonPath.read(map, "$.targets");
                targetList.addAll(Lists.newArrayList(JSON.parseArray(JSON.toJSONString(targets))));
                hooksInfoAll.getTargetMap().put(apiUrl, targetList);
                hooksInfoAll.getApiList().add(apiInfo);
            } catch (Exception e) {
                log.error("解析pageView获取API服务失败，scenes：{}", scene);
                throw new DWException("解析pageView获取API服务失败");
            }
        }
    }
}
